import time
from functools import wraps
import threading, requests, json


def retry(exception_to_check, tries=3, delay=5, backoff=1):
    """
    Decorator used to automatically retry a failed function. Parameters:

    exception_to_check: The type of exception to catch.
    tries: Maximum number of retry attempts.
    delay: Waiting time between each retry.
    backoff: Multiplicative factor to increase the waiting time after each retry.
    """

    def deco_retry(f):
        @wraps(f)
        def f_retry(*args, **kwargs):
            mtries, mdelay = tries, delay
            while mtries > 1:
                try:
                    return f(*args, **kwargs)
                except exception_to_check as e:
                    print(f"{str(e)}, Retrying in {mdelay} seconds...")
                    time.sleep(mdelay)
                    mtries -= 1
                    mdelay *= backoff
            return f(*args, **kwargs)

        return f_retry  # true decorator

    return deco_retry


def timeout_decorator(timeout):
    class TimeoutException(Exception):
        pass

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            result = [TimeoutException("Function call timed out")]  # Nonlocal mutable variable

            def target():
                try:
                    result[0] = func(*args, **kwargs)
                except Exception as e:
                    result[0] = e  # type: ignore

            thread = threading.Thread(target=target)
            thread.start()
            thread.join(timeout)
            if thread.is_alive():
                print(f"Function {func.__name__} timed out, retrying...")
                return wrapper(*args, **kwargs)
            if isinstance(result[0], Exception):
                raise result[0]
            return result[0]

        return wrapper

    return decorator


class CommonAgents(object):

    def __init__(
        self,
        engine="qvq-72b-preview",
        temp=0.1,  # 0.2,
        logit_bias: dict = {},
        max_new_token=4096,
        sample_n=1,
        if_reasoner=False,
        stream=False,
        stop=None,
        api_key="",
        base_url="",
        if_azure=False,
        if_anthropic=False,
        if_doubao=False,
    ):
        self.engine = engine
        self.temp = temp
        self.logit_bias = logit_bias
        self.max_new_token = max_new_token
        self.sample_n = sample_n
        self.if_reasoner = if_reasoner
        self.stream = stream
        self.stop = stop
        self.if_anthropic = if_anthropic
        if if_azure:
            from openai import AzureOpenAI

            self.client = AzureOpenAI(
                api_key=api_key,
                api_version="2024-07-01-preview",
                azure_endpoint=base_url,
            )
        elif if_anthropic:
            import anthropic

            self.client = anthropic.Anthropic(api_key=api_key, base_url=base_url)
        elif if_doubao:
            from volcenginesdkarkruntime import Ark

            self.client = Ark(
                api_key=api_key,
            )
        else:
            from openai import OpenAI

            self.client = OpenAI(
                api_key=api_key,
                base_url=base_url,
            )
        """
        deepseek_models = ["deepseek-chat", "deepseek-reasoner"]
        qwen_models = [
            "qvq-72b-preview",
            "qvq-max",
            "qwq-32b",
            "qwq-plus",
            "qwen2.5-vl-32b-instruct",
            "qwen-vl-max",
            "qwen-vl-plus",
            "qwen3_32b",
            "qwen-vl-plus-latest",
            "qwen-vl-plus-2025-01-25",
            "qwen-vl-plus-0102",
            "qwen-vl-plus-2025-05-07",
        ]
        gpt_models = [
            "gpt-4o-2024-08-06",
            "gpt-4o-2024-11-20",
            "gpt-4o-mini-2024-07-18",
            "gpt-4o-search-preview-2025-03-11",
            "gpt-4o-mini-search-preview-2025-03-11",
        ]

        google_models = ["gemini-2.5-flash", "gemini-2.5-pro"]

        reasoner_models = ["deepseek-reasoner", "qvq-max", "qwq-32b", "qwq-plus"]
        stream_models = ["qvq-max", "qwq-32b", "qwq-plus"]

        vision_models = [
            "qvq-72b-preview",
            "qvq-max",
            "qwen2.5-vl-32b-instruct",
            "qwen-vl-max",
            "qwen-vl-plus",
            "gpt-4o-2024-11-20",
            "gpt-4o-mini-2024-07-18",
            "gpt-4o-search-preview-2025-03-11",
            "gpt-4o-mini-search-preview-2025-03-11",
            "qwen-vl-plus-latest",
            "qwen-vl-plus-2025-01-25",
            "qwen-vl-plus-0102",
            "qwen-vl-plus-2025-05-07",
        ]

        if self.engine in deepseek_models:
            self.client = OpenAI(api_key="", base_url="https://api.deepseek.com")
            self.temp = 0.0
        elif self.engine in qwen_models:
            self.client = OpenAI(
                api_key="",
                base_url="https://dashscope.aliyuncs.com/compatible-mode/v1",
            )
            self.temp = 0.2
        elif self.engine in gpt_models:
            self.client = OpenAI(
                api_key="",
            )
        elif self.engine in google_models:
            self.client = OpenAI(
                base_url="https://generativelanguage.googleapis.com/v1beta/openai/",
                api_key="",
            )

        if self.engine in reasoner_models:
            self.if_reasoner = True
        if self.engine in stream_models:
            self.stream = True
        """

    @timeout_decorator(timeout=18000)
    @retry(exception_to_check=Exception, tries=3, delay=5, backoff=1)
    def send_chat_request(self, message_text):
        data_res_list = []
        reasoning_res_list = []

        if self.if_anthropic:
            response = self.client.messages.create(
                messages=message_text,  # [{"role": "user", "content": ""},{"role": "assistant", "content": ""}]
                model=self.engine,
                stream=self.stream,
                temperature=self.temp,
                max_tokens=self.max_new_token,
                # top_p=0.95,
                # frequency_penalty=0,
                # presence_penalty=0,
                # stop=self.stop,
                # n=self.sample_n,
            )
        else:
            response = self.client.chat.completions.create(
                messages=message_text,  # [{"role": "user", "content": ""},{"role": "assistant", "content": ""}]
                model=self.engine,
                stream=self.stream,
                temperature=self.temp,
                # max_tokens=self.max_new_token,
                # top_p=0.95,
                # frequency_penalty=0,
                # presence_penalty=0,
                # stop=self.stop,
                # n=self.sample_n,
            )

        data_res = ""
        reasoning_content = ""
        if self.stream == True:
            for index in range(self.sample_n):
                for chunk in response:
                    if not chunk.choices:  # type: ignore
                        print("No response[chunks] chunk.choices \nUsage:")
                        print(chunk.usage)  # type: ignore
                    else:
                        delta = chunk.choices[index].delta  # type: ignore
                        if hasattr(delta, "reasoning_content") and delta.reasoning_content != None:  # type: ignore
                            reasoning_content += delta.reasoning_content  # type: ignore
                        else:
                            data_res += delta.content  # type: ignore
        else:
            for index in range(self.sample_n):
                message = response.choices[index].message  # type: ignore
                data_res = message.content
                if self.if_reasoner:
                    reasoning_content = message.reasoning_content  # type: ignore
                else:
                    reasoning_content = "None"
        data_res_list.append(data_res)
        reasoning_res_list.append(reasoning_content)

        return data_res_list[0], data_res_list, reasoning_res_list[0], reasoning_res_list


class HttpAgents(object):

    def __init__(
        self,
        engine="qvq-72b-preview",  # "gpt-4o-2024-08-06",  # o1-preview-2024-09-12
        temp=0.1,  # 0.2,
        logit_bias: dict = {},
        max_new_token=4096,
        sample_n=1,
        if_reasoner=False,
        stream=False,
        stop=None,
        api_key="",
        base_url="",
    ):
        self.engine = engine
        self.temp = temp
        self.logit_bias = logit_bias
        self.max_new_token = max_new_token
        self.sample_n = sample_n
        self.if_reasoner = if_reasoner
        self.stream = stream
        self.stop = stop
        self.url = base_url
        self.headers = {
            "Content-Type": "application/json",
            "Authorization": api_key,
        }

    @timeout_decorator(timeout=18000)
    @retry(exception_to_check=Exception, tries=3, delay=5, backoff=1)
    def send_chat_request(self, message_text):
        data_res_list = []
        reasoning_res_list = []

        payload = json.dumps({"model": self.engine, "messages": message_text})
        response = requests.request("POST", self.url, headers=self.headers, data=payload)

        print(response)

        data_res = ""
        reasoning_content = ""
        if self.stream == True:
            for index in range(self.sample_n):
                for chunk in response:
                    if not chunk.choices:  # type: ignore
                        print("No response[chunks] chunk.choices \nUsage:")
                        print(chunk.usage)  # type: ignore
                    else:
                        delta = chunk.choices[index].delta  # type: ignore
                        if hasattr(delta, "reasoning_content") and delta.reasoning_content != None:  # type: ignore
                            reasoning_content += delta.reasoning_content  # type: ignore
                        else:
                            data_res += delta.content  # type: ignore
        else:
            for index in range(self.sample_n):
                message = response.choices[index].message  # type: ignore
                data_res = message.content
                if self.if_reasoner:
                    reasoning_content = message.reasoning_content  # type: ignore
                else:
                    reasoning_content = "None"
        data_res_list.append(data_res)
        reasoning_res_list.append(reasoning_content)

        return data_res_list[0], data_res_list, reasoning_res_list[0], reasoning_res_list
