import requests
import json
from typing import List, Dict, Optional, Union
import time
from functools import wraps
import threading
import traceback
from loguru import logger
from tqdm import tqdm
import re, sys, os

# from openai import OpenAI


def retry(exception_to_check, tries=3, delay=5, backoff=1):
    """
    Decorator used to automatically retry a failed function. Parameters:

    exception_to_check: The type of exception to catch.
    tries: Maximum number of retry attempts.
    delay: Waiting time between each retry.
    backoff: Multiplicative factor to increase the waiting time after each retry.
    """

    def deco_retry(f):
        @wraps(f)
        def f_retry(*args, **kwargs):
            mtries, mdelay = tries, delay
            while mtries > 1:
                try:
                    return f(*args, **kwargs)
                except exception_to_check as e:
                    print(f"{str(e)}, Retrying in {mdelay} seconds...")
                    time.sleep(mdelay)
                    mtries -= 1
                    mdelay *= backoff
            return f(*args, **kwargs)

        return f_retry  # true decorator

    return deco_retry


def timeout_decorator(timeout):
    class TimeoutException(Exception):
        pass

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            result = [TimeoutException("Function call timed out")]  # Nonlocal mutable variable

            def target():
                try:
                    result[0] = func(*args, **kwargs)
                except Exception as e:
                    result[0] = e  # type: ignore

            thread = threading.Thread(target=target)
            thread.start()
            thread.join(timeout)
            if thread.is_alive():
                print(f"Function {func.__name__} timed out, retrying...")
                return wrapper(*args, **kwargs)
            if isinstance(result[0], Exception):
                raise result[0]
            return result[0]

        return wrapper

    return decorator


class GPT:
    def __init__(
        self,
        base_url: str = "",
        appkey="",
        secret="",
        model="",
        vendor="",
        chatMode="",
        stream=False,
        temperature=0.2,
        if_reasoner=False,
    ):
        self.base_url = base_url
        self.appkey = appkey
        self.secret = secret
        self.model = model
        self.vendor = vendor
        self.chatMode = chatMode
        self.stream = stream
        self.temperature = temperature
        self.if_reasoner = if_reasoner

        reasoner_models = ["deepseek-reasoner", "qvq-max", "qwq-32b", "qwq-plus"]
        stream_models = ["qvq-max", "qwq-32b", "qwq-plus"]
        if self.model in reasoner_models:
            self.if_reasoner = True
        if self.model in stream_models:
            self.stream = True

    @timeout_decorator(timeout=3600)
    @retry(exception_to_check=Exception, tries=3, delay=5, backoff=1)
    def send_chat_request(self, messages) -> Dict:
        start_time = time.time()
        body = {
            "model": self.model,
            "temperature": self.temperature,
            "messages": [{"role": "user", "content": messages}] if isinstance(messages, str) else messages,
            "max_tokens": 2048,  # 使用UniGPT时加上可能会导致输出{"errorCode":"8500502","errorMsg":"context_length_exceeded"}
        }
        headers = {
            "appkey": self.appkey,
            "chatmode": self.chatMode,
            "content-Type": "application/json",
            "stream": "false",
            "timestamp": str(int(round(time.time() * 1000))),
            "udid": "Michaelmas",
            "vendor": self.vendor,
        }
        proxies = {"http": "socks5h://127.0.0.1:61107", "https": "socks5h://127.0.0.1:61107"}
        response = None
        result = ""
        reasoning_content = ""
        request_data = {"response": None, "exception": None}

        def do_request(request_data=request_data):
            try:
                # response = requests.post(self.base_url, headers=headers, json=body, timeout=180, proxies=proxies)
                response = requests.post(self.base_url, headers=headers, json=body, timeout=180)
                request_data["response"] = response
            except Exception as e:
                request_data["exception"] = e

        # Create and start the thread
        request_thread = threading.Thread(target=do_request)
        request_thread.start()

        try:
            request_thread.join()
        except KeyboardInterrupt:
            tqdm.write(f"\nShutdown signal received. Waiting for the current request to finish...")
            request_thread.join()
            tqdm.write(f"Request finished. Exiting gracefully.")
            sys.exit(1)
        except requests.exceptions.ConnectionError:
            traceback.print_exc()
            logger.warning("接口调用失败，正在重试...")
        except requests.exceptions.ReadTimeout:
            traceback.print_exc()
            logger.warning("调用超时，正在重试...")

        if request_data["exception"]:
            logger.warning("An exception occurred during the request.")
            traceback.print_exception(
                type(request_data["exception"]), request_data["exception"], request_data["exception"].__traceback__
            )
            raise request_data["exception"]
        response = request_data["response"]

        if not response:
            # raise ConnectionError("Failed to get a response from the server.")
            print("Failed to get a response from the server.")
            return "No Response", None, "No Response", None  # type: ignore

        end_time = time.time()

        # response = requests.post(self.base_url, headers=headers, json=body, timeout=180, proxies=proxies)
        # response = requests.post(self.base_url, headers=headers, json=body, timeout=180)
        # logger.info(f"耗时：{(end_time - start_time):.2f}s")
        try:
            # 非流式返回（生成文本过长会出现504 Gateway Time-out）
            if self.stream == False:
                content = response
                content = json.loads(response.content.decode())["result"]["choices"][0]["message"]["content"]
                reasoning_content, result = self.extract_content_outside_think(content)
            else:
                content = ""
                has_content = False
                for line in response.iter_lines():  # type: ignore
                    line = line.decode()
                    if line.startswith("data"):
                        line = line[5:]
                        if not has_content:
                            has_content = True
                            continue
                        if line == "[DONE]":
                            delta = json.loads(line)["choices"][0]["delta"]
                            content += delta.get("content", "")
                reasoning_content, result = self.extract_content_outside_think(content)
        except:
            tqdm.write(f"An exception occurred while processing the response, retrying...")
            tqdm.write(f"{response}")
            tqdm.write(f"{response.content.decode()}")
            traceback.print_exc()
            raise

        return result, None, reasoning_content, None  # type: ignore

    def extract_content_outside_think(self, text):
        inside_pattern = r"<think>(.*?)</think>"
        inside_content = re.findall(inside_pattern, text, flags=re.DOTALL)
        outside_pattern = r"<think>.*?</think>"
        outside_content_raw = re.split(outside_pattern, text, flags=re.DOTALL)
        cleaned_parts = (part.strip() for part in outside_content_raw if part and part.strip())
        outside_content_string = " ".join(cleaned_parts)
        return inside_content, outside_content_string


# ssh -R 61107:127.0.0.1:61107 docker_chartqa
if __name__ == "__main__":
    appkey = ""
    secret = ""
    models = {}
    chat = GPT(appkey=appkey, model="qwen3_32b", vendor="", stream=False, temperature=0.2)
    try:
        messages = [{"role": "user", "content": "你好"}]
        response = chat.send_chat_request(messages)
        print(response)
        # print(json.dumps(response, ensure_ascii=False, indent=2))
    except Exception as e:
        print(f"错误: {str(e)}")
