import re
import base64
from PIL import Image
import io
from loguru import logger
import json
import ast


# LABEL_STUDIO_URL = "http://localhost:61131"
# LABEL_STUDIO_URL = "http://localhost:61132"
LABEL_STUDIO_URL = "http://localhost:10001"
API_KEY = ""
answer_pattern = r"<answer>(.*?)</answer>"
question_pattern = r"<questions>(.*?)</questions>"


class NoAnswerFound(Exception):
    """Exception raised when the expected answer tag is not found in the model's response."""

    pass


def encode_image(picture_path, max_dimension: int = 1920):
    try:
        with Image.open(picture_path) as img:
            original_format = img.format
            width, height = img.size
            if width <= max_dimension and height <= max_dimension:
                resized_img = img
            else:
                if width > height:
                    new_width = max_dimension
                    new_height = int(max_dimension * height / width)
                else:
                    new_height = max_dimension
                    new_width = int(max_dimension * width / height)
                resized_img = img.resize((new_width, new_height), Image.Resampling.LANCZOS)
            byte_stream = io.BytesIO()
            resized_img.save(byte_stream, format=original_format)
            image_bytes = byte_stream.getvalue()
            return base64.b64encode(image_bytes).decode("utf-8")
    except FileNotFoundError:
        logger.error(f"file not found {picture_path}")
        return None
    except Exception as e:
        logger.error(f"error: {e}")
        return None


def translate(agent, origin_text: str):
    translation_prompt = f"""As an academic expert with specialized knowledge in various fields, please provide a proficient and precise translation from English to Chinese of the academic text enclosed in <text>...</text>. It is crucial to maintaining the original phrase or sentence and ensure accuracy while utilizing the appropriate language. The text is as follows:  <text> {origin_text} </text>. Please provide the translated result without any additional explanation and remove <text>...</text>. Put all text in a json format like 'origin text': 'translate result', ..."""
    message_text = [{"role": "user", "content": translation_prompt}]
    # logger.info(f"send translate request")
    translation, _, _, _ = agent.send_chat_request(message_text)
    return translation


def ch2eng(agent, origin_text: str):
    translation_prompt = f"""As an academic expert with specialized knowledge in various fields, please provide a proficient and precise translation from Chinese to English of the academic text enclosed in <text>...</text>. It is crucial to maintaining the original phrase or sentence and ensure accuracy while utilizing the appropriate language. The text is as follows:  <text> {origin_text} </text>. Please provide the translated result without any additional explanation and remove <text>...</text>. If origin text is already in English, Then just response it."""
    message_text = [{"role": "user", "content": translation_prompt}]
    # logger.info(f"send translate request")
    translation, _, _, _ = agent.send_chat_request(message_text)
    return translation


def translate_image(agent, translate_agent, image_path: str):
    system_prompt = "REMEMBER TO PUT YOUR ANSWER IN <answer></answer> tag."
    ocr_prompt = f"""
This is an OCR task, please provide all the valid text in this chart.

You should identify all of the texts (exclude numbers) in this chart, and then put them all sequentially in only one <answer></answer> tag.
    """
    prompt = [
        {"type": "image_url", "image_url": {"url": f"data:image/jpg;base64,{encode_image(image_path)}"}},
        {"type": "text", "text": ocr_prompt},
    ]
    message_text = [{"role": "system", "content": system_prompt}, {"role": "user", "content": prompt}]
    # logger.info(f"send ocr request")
    result, _, _, _ = agent.send_chat_request(message_text)
    # logger.info(f"{result}")

    # match = re.search(answer_pattern, result, re.DOTALL)
    matches = re.findall(answer_pattern, result, re.DOTALL)

    if matches:
        # ocr_result = match.group(1).strip()
        ocr_result = " ".join(matches)
    else:
        ocr_result = result
    if len(ocr_result) != 0:
        translate_result = translate(translate_agent, ocr_result)
        new_string = translate_result.replace("\n", " ")
    else:
        new_string = ocr_result
    return new_string
    # raise NoAnswerFound("The answer in <answer></answer> tag was not found in the response from the agent.")


def generate_reference_questions(agent, image_captions: list):
    general_prompt = f"""
    Purpose and Goals:

* Generate a series of challenging questions based on provided descriptions of charts and figures, every descriptions will be inside within <caption></caption> tag.

* Ensure questions require information from multiple charts and their descriptive texts.

* Create questions of various types: multiple-choice, true/false, open-ended vocabulary, numerical calculation, and open-ended questions.

* Maintain a high level of inferential difficulty for all questions.


Behaviors and Rules:

1) Question Generation Process:

a) Analyze the provided descriptions of charts and figures thoroughly to identify key data points, relationships, and trends.

b) For each question, ensure it requires the user to synthesize information from at least two different charts or their associated descriptions.

c) Vary the question types to include: multiple-choice (with 4 distinct options, only one correct), true/false, open-ended vocabulary (requiring a specific term), numerical calculation (requiring a precise numerical answer), and open-ended questions (requiring analytical thought and a brief explanation).

d) Frame questions in a clear, concise, and unambiguous manner.

e) Do not introduce new information not present in the provided charts or descriptions.

f) Ensure that the difficulty level is challenging, requiring inferential reasoning rather than direct recall.


2) Output Format:

a) Present each question clearly, labeled with its type (e.g., 'Multiple Choice:', 'True/False:', 'Numerical Calculation:').

b) For multiple-choice questions, clearly list the options (A, B, C, D).

c) Do not provide answers to the questions unless explicitly asked by the user in a follow-up prompt.

d) Put all questions in one <questions>...</questions> tag.



Overall Tone:

* Professional, analytical, and precise.

* Confident in the complexity and relevance of the questions generated.
    """
    prompt = [
        {"type": "system", "content": general_prompt},
    ]
    caption_prompt = f"""
Captions are as follows:
"""
    for image_caption in image_captions:
        caption_prompt += f"\n<caption> {image_caption['caption']} </caption>"

    caption_prompt += "\nREMEMBER: Put all questions in one <questions>...</questions> tag."

    prompt.append({"role": "user", "content": caption_prompt})

    # print(prompt)
    # logger.info(f"send generate_reference_questions request")
    result, _, _, _ = agent.send_chat_request(prompt)
    # logger.info(f"{result}")

    match = re.search(question_pattern, result, re.DOTALL)
    if match:
        question_result = match.group(1).strip()
        # print(f"result: {result}")
        return question_result
    else:
        question_result = result
        return question_result
        # raise NoAnswerFound("The answer in <questions></questions> tag was not found in the response from the agent.")


def get_topic(agent, captions: dict) -> dict:
    topics = [
        "Governance",
        "Economy",
        "Finance",
        "Social",
        "Environment",
        "Urban Development",
        "Education",
        "Health",
        "Agriculture",
        "Energy",
        "Industry",
        "Culture",
        "Transport",
        "Sports",
        "Science and Technology, Weather",
        "Entertainment",
        "Arixv",
    ]
    system_prompt = """
As an academic expert with specialized knowledge in various fields, please provide a topic about following paragraphs related to some charts enclosed in <text>...</text>.

Please provide topic names within the following scope [Governance, Economy, Finance, Social, Environment, Urban Development, Education, Health, Agriculture, Energy, Industry, Culture, Transport, Sports, Science and Technology, Weather, Entertainment, Arixv].
"""

    topic_prompt = f"""
The paragraphs are as follows:

"""

    for caption in captions:
        topic_prompt += f"""
<text> {caption["caption"]} </text>

"""

    topic_prompt += "Please provide one or more suitable topics within the above range and put the topic in a list in <answer>...</answer> tag. For example <answer> ['xxx'] </answer>"

    message_text = [{"role": "system", "content": system_prompt}, {"role": "user", "content": topic_prompt}]
    response_topic, _, _, _ = agent.send_chat_request(message_text)

    match = re.search(answer_pattern, response_topic, re.DOTALL)
    if match:
        return_topics = ast.literal_eval(match.group(1).strip())
    else:
        try:
            return_topics = ast.literal_eval(response_topic.strip())
        except:
            logger.error(f"Error durling get_topic(), topic: {response_topic}")
            raise NoAnswerFound("The answer in <answer></answer> tag was not found in the response from the agent.")

    for return_topic in return_topics:
        if return_topic not in topics:
            retry_prompt = f"Your response is not in {topics}. Reanswer and make sure your answer in given topics. Provide one or more suitable topics and put the topic in a list in <answer>...</answer> tag. For example <answer> ['xxx'] </answer>"
            message_text = [
                {"role": "system", "content": system_prompt},
                {"role": "user", "content": topic_prompt},
                {"role": "assistant", "content": response_topic},
                {"role": "user", "content": retry_prompt},
            ]
            response_topic, _, _, _ = agent.send_chat_request(message_text)
            match = re.search(answer_pattern, response_topic, re.DOTALL)
            if match:
                return_topics = ast.literal_eval(match.group(1).strip())
            else:
                try:
                    return_topics = ast.literal_eval(response_topic.strip())
                except:
                    logger.error(f"Error durling get_topic(), topic: {response_topic}")
                    raise NoAnswerFound(
                        "The answer in <answer></answer> tag was not found in the response from the agent."
                    )
            break

    return return_topics


def qa_elements_ch2eng(taxonomy) -> list:
    return_taxonomy = []
    for elements in taxonomy:
        return_elements = []
        for element in elements:
            if element == "图表数值":
                return_elements.append("numerical difficulity")
            elif element == "多图表数值比较":
                return_elements.append("numerical comparation")
            elif element == "多图表数值计算":
                return_elements.append("numerical calculation")
            elif element == "图表数值有辨认难度":
                return_elements.append("numerical identification")
            elif element == "图表形状/颜色":
                return_elements.append("difficulity of shape/color")
            elif element == "图表颜色有辨认难度":
                return_elements.append("color mix")
            elif element == "图表形状有辨认难度":
                return_elements.append("complex shape")
            elif element == "图表重叠":
                return_elements.append("charts overlap")
            elif element == "3维图表辨认":
                return_elements.append("3D charts")
            elif element == "图表类型":
                return_elements.append("chart type")
            elif element == "图表走势":
                return_elements.append("chart trend")
            elif element == "图表走势对比":
                return_elements.append("trend comparation")
            elif element == "图表对比数量多":
                return_elements.append("a lot of comparations")
            elif element == "上下文推理":
                return_elements.append("context reasoning")
            elif element == "图表摘要上下文":
                return_elements.append("context of captions")
            elif element == "需要额外的背景知识或常识":
                return_elements.append("general knowledge")
            elif element == "图表其他元素(非数值)识别":
                return_elements.append("element identification")
            elif element == "元素颜色有辨认难度":
                return_elements.append("color identification")
            elif element == "元素形状有辨认难度":
                return_elements.append("shape identification")
        if len(return_elements) != 0:
            return_taxonomy.append(return_elements)
    return return_taxonomy
