# Import the SDK and the client module
from label_studio_sdk.client import LabelStudio
import json
import copy
import sys, os
from tqdm import tqdm
from concurrent.futures import ThreadPoolExecutor, as_completed
from threading import Lock

from labelstudio_api import update_annotation_reference_question

sys.path.append(os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))
from gpt_api.unigpt import GPT

sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
from common_prompts import translate, LABEL_STUDIO_URL, API_KEY


class PD:
    def __init__(self, MAX_WORKERS=10) -> None:
        self.failed_json_path = "chartqa/data/failed/failed_reference_question.json"
        self.failed_ids = []
        self.failed_ids_lock = Lock()
        self.MAX_WORKERS = MAX_WORKERS

        self.client = LabelStudio(base_url=LABEL_STUDIO_URL, api_key=API_KEY)
        self.text_qwen_chat = GPT(model="qwen3_32b", vendor="", stream=False, temperature=0.2)

    def process_task(self, task):
        try:
            task_id, project_id, is_success = update_annotation_reference_question(
                task=task, client=self.client, text_agent=self.text_qwen_chat
            )
            if not is_success:
                with self.failed_ids_lock:
                    self.failed_ids.append({"task_id": task_id, "project_id": project_id})
            else:
                tqdm.write(f"FINISHED task: {task.id}")
        except Exception as e:
            tqdm.write(f"ERROR processing task {task.id}: {e}")
            with self.failed_ids_lock:
                self.failed_ids.append({"task_id": task.id, "project_id": task.project, "error": str(e)})

    def process_projects(self):
        all_tasks = []
        projects = self.client.projects.list()
        for project in tqdm(list(projects), desc="Gathering Projects"):
            tasks_in_project = self.client.tasks.list(project=project.id)  # , include="id")
            all_tasks.extend(tasks_in_project)

        with ThreadPoolExecutor(max_workers=self.MAX_WORKERS) as executor:
            futures = {executor.submit(self.process_task, task) for task in all_tasks}

            for future in tqdm(as_completed(futures), total=len(all_tasks), desc="Processing Tasks"):
                try:
                    future.result()
                except Exception as e:
                    tqdm.write(f"A worker thread encountered a critical error: {e}")

        with open(self.failed_json_path, "w") as f:
            json.dump(self.failed_ids, f, ensure_ascii=False, indent=2)


if __name__ == "__main__":

    pd = PD(MAX_WORKERS=5)
    pd.process_projects()
