import warnings
import math

import torch.nn as nn
import torch

def set_initial_value(param, init_method, a=0, mode='fan_in', nonlinearity='relu'):
    if init_method == 'signed_kaiming_constant':
        fan = nn.init._calculate_correct_fan(param, mode)
        gain = nn.init.calculate_gain(nonlinearity)
        std = gain / (fan ** (1/2))
        param.data = std * param.data.sign()

    elif init_method == 'kaiming_normal':
        nn.init.kaiming_normal_(param, a=a, mode=mode, nonlinearity=nonlinearity)
    elif init_method == 'kaiming_uniform':
        nn.init.kaiming_uniform_(param, a=a, mode=mode, nonlinearity=nonlinearity)
    elif init_method == 'trunc_normal':
        trunc_normal_(param, std=.02)
    elif init_method == 'signed_trunc_constant':
        def norm_cdf(x):
            # Computes standard normal cumulative distribution function
            return (1. + math.erf(x / math.sqrt(2.))) / 2.
        def gauss_pdf(x):
            return (1 / math.sqrt(2*math.pi)) * math.exp(-(x**2)/2)
        mean =  0
        std  = .02
        a    = -2.
        b    =  2.
        alpha = (a - mean) / std
        beta  = (b - mean) / std
        Z     = norm_cdf(beta) - norm_cdf(alpha)
        modified_var = (std**2) * (1 - (beta * gauss_pdf(beta) - alpha * gauss_pdf(alpha)) / Z - ((gauss_pdf(beta) - gauss_pdf(alpha)) / Z)**2)
        modified_std = modified_var**(1/2)
        param.data = modified_std * param.data.sign()
    else:
        raise NotImplementedError

def _trunc_normal_(tensor, mean, std, a, b):
    # Cut & paste from PyTorch official master until it's in a few official releases - RW
    # Method based on https://people.sc.fsu.edu/~jburkardt/presentations/truncated_normal.pdf
    def norm_cdf(x):
        # Computes standard normal cumulative distribution function
        return (1. + math.erf(x / math.sqrt(2.))) / 2.

    if (mean < a - 2 * std) or (mean > b + 2 * std):
        warnings.warn("mean is more than 2 std from [a, b] in nn.init.trunc_normal_. "
                      "The distribution of values may be incorrect.",
                      stacklevel=2)

    # Values are generated by using a truncated uniform distribution and
    # then using the inverse CDF for the normal distribution.
    # Get upper and lower cdf values
    l = norm_cdf((a - mean) / std)
    u = norm_cdf((b - mean) / std)

    # Uniformly fill tensor with values from [l, u], then translate to
    # [2l-1, 2u-1].
    tensor.uniform_(2 * l - 1, 2 * u - 1)

    # Use inverse cdf transform for normal distribution to get truncated
    # standard normal
    tensor.erfinv_()

    # Transform to proper mean, std
    tensor.mul_(std * math.sqrt(2.))
    tensor.add_(mean)

    # Clamp to ensure it's in the proper range
    tensor.clamp_(min=a, max=b)
    return tensor


def trunc_normal_(tensor, mean=0., std=1., a=-2., b=2.):
    # type: (Tensor, float, float, float, float) -> Tensor
    r"""Fills the input Tensor with values drawn from a truncated
    normal distribution. The values are effectively drawn from the
    normal distribution :math:`\mathcal{N}(\text{mean}, \text{std}^2)`
    with values outside :math:`[a, b]` redrawn until they are within
    the bounds. The method used for generating the random values works
    best when :math:`a \leq \text{mean} \leq b`.

    NOTE: this impl is similar to the PyTorch trunc_normal_, the bounds [a, b] are
    applied while sampling the normal with mean/std applied, therefore a, b args
    should be adjusted to match the range of mean, std args.

    Args:
        tensor: an n-dimensional `torch.Tensor`
        mean: the mean of the normal distribution
        std: the standard deviation of the normal distribution
        a: the minimum cutoff value
        b: the maximum cutoff value
    Examples:
        >>> w = torch.empty(3, 5)
        >>> nn.init.trunc_normal_(w)
    """
    with torch.no_grad():
        return _trunc_normal_(tensor, mean, std, a, b)