"""
软约束评估器的LLM提示词模板
"""

import json
from typing import Dict, Any, Tuple


def get_classic_attractions_prompt(destination:str, daily_activities: dict, locale: str = "en-US") -> str:
    """
    Get prompt for classic attractions coverage evaluation
    
    Args:
        daily_activities: Daily activity arrangement dictionary
        locale: Language environment
        
    Returns:
        str: Formatted prompt
    """
    itinerary_str = json.dumps(daily_activities, ensure_ascii=False, indent=2)
    
    if locale == "zh-CN":
        return f"""请评估以下行程中的景点是否覆盖了目的地({destination})的经典必玩景点。

行程如下：
{itinerary_str}

请根据以下标准进行评估：
1. 是否包含了该目的地最著名的地标性景点
2. 是否涵盖了不同类型的经典景点（历史文化、自然风光、现代建筑等）
3. 景点的知名度和推荐度
4. 请结合行程的天数来评判，如果因为天数不够导致部分次优景点无法覆盖，可放松评测标准

请以JSON格式返回评估结果：
{{
    "explanation": "详细解释",
    "score": 评分(1-5的整数等级，1=完全没有任何经典景点，2=只有很少的经典景点，3=有部分经典景点，4=有大部分经典景点，5=有全部经典景点),
    "missing_attractions": ["缺失的重要经典景点列表"]
}}"""
    else:
        return f"""Please evaluate whether the attractions in the following itinerary cover the classic must-visit attractions of the destination ({destination}).

Itinerary:
{itinerary_str}

Please evaluate based on the following criteria:
1. Does it include the most famous landmark attractions of the destination
2. Does it cover different types of classic attractions (historical culture, natural scenery, modern architecture, etc.)
3. The popularity and recommendation level of the attractions
4. Please consider the number of days in the itinerary. If some secondary attractions cannot be covered due to insufficient days, you can relax the evaluation criteria

Please return the evaluation result in JSON format:
{{
    "explanation": "detailed explanation",
    "score": score (integer rating 1-5, where 1=no classic attractions, 2=only a few classic attractions, 3=some classic attractions, 4=most classic attractions, 5=all classic attractions),
    "missing_attractions": ["list of missing important classic attractions"]
}}"""


def get_itinerary_diversity_prompt(daily_activities: dict, locale: str = "en-US") -> str:
    """
    Get prompt for itinerary diversity evaluation
    
    Args:
        daily_activities: Daily activity arrangement dictionary
        locale: Language environment
        
    Returns:
        str: Formatted prompt
    """
    itinerary_str = json.dumps(daily_activities, ensure_ascii=False, indent=2)
    
    if locale == "zh-CN":
        return f"""请评估以下行程的丰富性和多样化程度，判断是否存在同质化问题。

行程安排：
{itinerary_str}

请根据以下标准进行评估：
1. 景点类型的多样性（历史文化、自然风光、娱乐休闲、购物美食等）
2. 活动体验的丰富性（观光、体验、互动、休闲等）
3. 节奏安排的合理性（动静结合、室内外结合）
4. 是否避免了重复性或同质化的活动
5. 请同时参考目的地的主要景点类型，如果目的地的主要景点为单一类型，则可以适当放宽同质化问题的评判标准
6. 请结合一天最多参观5个景点的标准，如果时间有限，可适当放宽同质化问题的评判标准

请以JSON格式返回评估结果：
{{
    "explanation": "详细解释",
    "score": 评分(1-5的整数等级，1=同质化问题占整个线路80%以上，2=同质化问题占整个线路60%左右，3=同质化问题占整个线路40%左右，4=同质化问题占整个线路20%左右，5=同质化问题较小或不存在),
    "diversity_issues": ["发现的同质化或单调问题列表"]
}}"""
    else:
        return f"""Please evaluate the richness and diversity of the following itinerary to determine if there are homogenization issues.

Itinerary:
{itinerary_str}

Please evaluate based on the following criteria:
1. Diversity of attraction types (historical culture, natural scenery, entertainment, shopping & dining, etc.)
2. Richness of activity experiences (sightseeing, hands-on experiences, interactive, leisure, etc.)
3. Reasonable pace arrangement (balance of active/quiet, indoor/outdoor)
4. Avoiding repetitive or homogeneous activities
5. Please consider the main attraction types of the destination. If the main attractions of the destination are of a single type, you can relax the evaluation criteria for homogeneity issues
6. Please adhere to the standard of visiting a maximum of 5 attractions per day. If time is limited, you can appropriately relax the evaluation criteria for homogeneity issues.


Please return the evaluation result in JSON format:
{{
    "explanation": "detailed explanation",
    "score": score (integer rating 1-5, where 1=homogenization problem accounts for more than 80% of the itinerary, 2=homogenization problem accounts for about 60% of the itinerary, 3=homogenization problem accounts for about 40% of the itinerary, 4=homogenization problem accounts for about 20% of the itinerary, 5=homogenization problem is small or nonexistent),
    "diversity_issues": ["list of identified homogenization or monotony issues"]
}}"""


def parse_evaluation_diverse_response(response: str) -> Dict[str, Any]:
    """
    Parse the evaluation response of the LLM

    Args:
        response: LLM response string

    Returns:
        Dict: Parsed evaluation result
    """
    try:
        # Try to parse JSON directly
        if isinstance(response, str):
            if "```json" in response:
                response = response.replace("```json", "").replace("```", "").strip()
            result = json.loads(response)
        else:
            print("cant load response", response)
            result = response

        # Validate necessary fields

        if "diversity_issues" not in result:
            result["diversity_issues"] = "Evaluation completed but missing detailed diversity_issues"

        if "score" not in result:
            result["score"] = 3

        if "explanation" not in result:
            result["explanation"] = "Evaluation completed but missing detailed explanation"

        return result

    except json.JSONDecodeError as e:
        # JSON parsing failed, return default result
        return {
            "score": 3,
            "diversity_issues": f"Failed to parse LLM response: {str(e)}, original response: {response[:200]}...",
            "parse_error": True
        }
    except Exception as e:
        return {
            "final_score": 3,
            "diversity_issues": f"Failed to process evaluation result: {str(e)}",
            "error": True
        }

def parse_evaluation_classic_response(response: str) -> Dict[str, Any]:
    """
    Parse the evaluation response of the LLM

    Args:
        response: LLM response string

    Returns:
        Dict: Parsed evaluation result
    """
    try:
        # Try to parse JSON directly
        if isinstance(response, str):
            if "```json" in response:
                response = response.replace("```json", "").replace("```", "").strip()
            result = json.loads(response)
        else:
            print("cant load response", response)
            result = response

        # Validate necessary fields

        if "missing_attractions" not in result:
            result["missing_attractions"] = "Evaluation completed but missing detailed missing_attractions"

        if "score" not in result:
            result["score"] = 3

        if "explanation" not in result:
            result["explanation"] = "Evaluation completed but missing detailed explanation"

        return result

    except json.JSONDecodeError as e:
        # JSON parsing failed, return default result
        return {
            "score": 3,
            "missing_attractions": f"Failed to parse LLM response: {str(e)}, original response: {response[:200]}...",
            "parse_error": True
        }
    except Exception as e:
        return {
            "final_score": 3,
            "missing_attractions": f"Failed to process evaluation result: {str(e)}",
            "error": True
        }

