# ===========================================================================
# Project:      Sparse Model Soups
# File:         models/cifar100.py
# Description:  CIFAR-100 Models
# ===========================================================================

import torch.nn.functional as F
from torch import nn
from utilities.utilities import Utilities as Utils

class WideResNet20(nn.Module):
    # WideResNet implementation but with widen_factor=2 and depth=22 instead of 10 and 28 respectively.
    # In Repo of Git-Rebasin, this is referred to as ResNet20
    def __init__(self, depth=22, widen_factor=10, dropout_rate=0.3, num_classes=100):
        super(WideResNet20, self).__init__()
        self.in_planes = 16

        assert ((depth - 4) % 6 == 0), 'Wide-resnet depth should be 6n+4'
        n = (depth - 4) / 6
        k = widen_factor

        nStages = [16, 16 * k, 32 * k, 64 * k]

        class wide_basic(nn.Module):
            def __init__(self, in_planes, planes, dropout_rate, stride=1):
                super(wide_basic, self).__init__()
                self.bn1 = nn.BatchNorm2d(in_planes)
                self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, padding=1, bias=True)

                self.dropout = nn.Dropout(p=dropout_rate)
                self.bn2 = nn.BatchNorm2d(planes)
                self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=True)

                self.shortcut = nn.Sequential()
                if stride != 1 or in_planes != planes:
                    self.shortcut = nn.Sequential(
                        nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=True),
                    )

            def forward(self, x):
                out = self.dropout(self.conv1(F.relu(self.bn1(x))))
                out = self.conv2(F.relu(self.bn2(out)))
                out += self.shortcut(x)

                return out

        self.conv1 = self.conv3x3(3, nStages[0])
        self.layer1 = self._wide_layer(wide_basic, nStages[1], n, dropout_rate, stride=1)
        self.layer2 = self._wide_layer(wide_basic, nStages[2], n, dropout_rate, stride=2)
        self.layer3 = self._wide_layer(wide_basic, nStages[3], n, dropout_rate, stride=2)
        self.bn1 = nn.BatchNorm2d(nStages[3], momentum=0.9)
        self.linear = nn.Linear(nStages[3], num_classes)

    def conv3x3(self, in_planes, out_planes, stride=1):
        return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=True)

    def _wide_layer(self, block, planes, num_blocks, dropout_rate, stride):
        strides = [stride] + [1] * (int(num_blocks) - 1)
        layers = []

        for stride in strides:
            layers.append(block(self.in_planes, planes, dropout_rate, stride))
            self.in_planes = planes

        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = F.relu(self.bn1(out))
        out = F.avg_pool2d(out, 8)
        out = out.view(out.size(0), -1)
        out = self.linear(out)

        return out

    @staticmethod
    def get_permutation_spec():
        conv = lambda name, p_in, p_out, bias=True: {f"{name}.weight": (p_out, p_in, None, None,),
                                                     f"{name}.bias": (p_out,)} if bias else {
            f"{name}.weight": (p_out, p_in, None, None,)}
        norm = lambda name, p: {f"{name}.weight": (p,), f"{name}.bias": (p,)}


        dense = lambda name, p_in, p_out, bias=True: {f"{name}.weight": (p_out, p_in),
                                                      f"{name}.bias": (p_out,)} if bias else {
            f"{name}.weight": (p_out, p_in)}

        # This is for easy blocks that use a residual connection, without any change in the number of channels.
        easyblock = lambda name, p: {
            **norm(f"{name}.bn1", p),
            **conv(f"{name}.conv1", p, f"P_{name}_inner"),
            **norm(f"{name}.bn2", f"P_{name}_inner"),
            **conv(f"{name}.conv2", f"P_{name}_inner", p),
        }

        # This is for blocks that use a residual connection, but change the number of channels via a Conv.
        shortcutblock = lambda name, p_in, p_out: {
            **norm(f"{name}.bn1", p_in),
            **conv(f"{name}.conv1", p_in, f"P_{name}_inner"),
            **norm(f"{name}.bn2", f"P_{name}_inner"),
            **conv(f"{name}.conv2", f"P_{name}_inner", p_out),
            **conv(f"{name}.shortcut.0", p_in, p_out),
            #**norm(f"{name}.shortcut.1", p_out),   # Removed this since not occuring in state dict
        }

        return Utils.permutation_spec_from_axes_to_perm({
            **conv("conv1", None, "P_bg0"),
            #
            **shortcutblock("layer1.0", "P_bg0", "P_bg1"),
            **easyblock("layer1.1", "P_bg1", ),
            **easyblock("layer1.2", "P_bg1"),
            # **easyblock("layer1.3", "P_bg1"),

            **shortcutblock("layer2.0", "P_bg1", "P_bg2"),
            **easyblock("layer2.1", "P_bg2", ),
            **easyblock("layer2.2", "P_bg2"),
            # **easyblock("layer2.3", "P_bg2"),

            **shortcutblock("layer3.0", "P_bg2", "P_bg3"),
            **easyblock("layer3.1", "P_bg3", ),
            **easyblock("layer3.2", "P_bg3"),
            # **easyblock("layer3.3", "P_bg3"),

            **norm("bn1", "P_bg3"),

            **dense("linear", "P_bg3", None),

        })

class WideResNet(nn.Module):
    # Based on https://github.com/meliketoy/wide-resnet.pytorch
    def __init__(self, depth=28, widen_factor=10, dropout_rate=0.3, num_classes=100):
        super(WideResNet, self).__init__()
        self.in_planes = 16

        assert ((depth - 4) % 6 == 0), 'Wide-resnet depth should be 6n+4'
        n = (depth - 4) / 6
        k = widen_factor

        nStages = [16, 16 * k, 32 * k, 64 * k]

        class wide_basic(nn.Module):
            def __init__(self, in_planes, planes, dropout_rate, stride=1):
                super(wide_basic, self).__init__()
                self.bn1 = nn.BatchNorm2d(in_planes)
                self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, padding=1, bias=True)

                self.dropout = nn.Dropout(p=dropout_rate)
                self.bn2 = nn.BatchNorm2d(planes)
                self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=True)

                self.shortcut = nn.Sequential()
                if stride != 1 or in_planes != planes:
                    self.shortcut = nn.Sequential(
                        nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=True),
                    )

            def forward(self, x):
                out = self.dropout(self.conv1(F.relu(self.bn1(x))))
                out = self.conv2(F.relu(self.bn2(out)))
                out += self.shortcut(x)

                return out

        self.conv1 = self.conv3x3(3, nStages[0])
        self.layer1 = self._wide_layer(wide_basic, nStages[1], n, dropout_rate, stride=1)
        self.layer2 = self._wide_layer(wide_basic, nStages[2], n, dropout_rate, stride=2)
        self.layer3 = self._wide_layer(wide_basic, nStages[3], n, dropout_rate, stride=2)
        self.bn1 = nn.BatchNorm2d(nStages[3], momentum=0.9)
        self.linear = nn.Linear(nStages[3], num_classes)

    def conv3x3(self, in_planes, out_planes, stride=1):
        return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=True)

    def _wide_layer(self, block, planes, num_blocks, dropout_rate, stride):
        strides = [stride] + [1] * (int(num_blocks) - 1)
        layers = []

        for stride in strides:
            layers.append(block(self.in_planes, planes, dropout_rate, stride))
            self.in_planes = planes

        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = F.relu(self.bn1(out))
        out = F.avg_pool2d(out, 8)
        out = out.view(out.size(0), -1)
        out = self.linear(out)

        return out
