#!/bin/bash

# Model names
MODEL_1="Qwen/Qwen2.5-7B-Instruct"
MODEL_2="Qwen/Qwen2.5-3B-Instruct"
MODEL_3="meta-llama/Llama-3.2-3B-Instruct"
MODEL_4="meta-llama/Llama-3.1-8B-Instruct"

# Default values
NUM_QUERY=50
ROUND=1
TEMP=0.7
# Option values
SELF_CONSISTENCY=false
NUM_SAMPLES=1
SELF_REFINEMENT=true
NUM_ROUND=3

# Dataset Specific
DATASET="advbench"
DATA_TOPIC="algebra"  # Only used for CollegeMath dataset
ADAPREFIX=true # Only used for safety domain
ADAPREFIX2=false # Only used for safety domain

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --model)
            MODEL_CHOICE="$2"
            shift 2
            ;;
        --dataset)
            DATASET="$2"
            shift 2
            ;;
        --num_query)
            NUM_QUERY="$2"
            shift 2
            ;;
        --round)
            ROUND="$2"
            shift 2
            ;;
        --temp)
            TEMP="$2"
            shift 2
            ;;
        --cot)
            COT=true
            shift
            ;;
        --self_consistency)
            SELF_CONSISTENCY=true
            shift
            ;;
        --num_samples)
            NUM_SAMPLES="$2"
            shift 2
            ;;
        --data_topic)
            DATA_TOPIC="$2"
            shift 2
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

# Validate required arguments
if [ -z "$MODEL_CHOICE" ] || [ -z "$DATASET" ]; then
    echo "Usage: bash run_math_single.sh --model <model_choice> --dataset <dataset> [options]"
    echo "  Required arguments:"
    echo "    --model <choice>         : Model choice (1-4)"
    echo "    --dataset <name>         : Dataset name (gsm8k, math500, collegemath)"
    echo "  Optional arguments:"
    echo "    --num_query <number>     : Number of queries (default: 500)"
    echo "    --round <number>         : Number of rounds (default: 1)"
    echo "    --temp <number>          : Temperature (default: 0.7)"
    echo "    --cot                    : Enable Chain-of-Thought"
    echo "    --self_consistency       : Enable self-consistency"
    echo "    --num_samples <number>   : Number of samples for self-consistency (default: 7)"
    echo "    --data_topic <topic>     : Topic for collegemath dataset (default: algebra)"
    echo "                               Options: algebra, calculus, precalculus, differential_equation,"
    echo "                                        linear_algebra, probability, vector_calculus"
    echo "  Model choices:"
    echo "    1: Qwen2.5-1.5B"
    echo "    2: Qwen2.5-3B"
    echo "    3: Llama-3.2-3B"
    echo "    4: Llama-3.1-8B"
    exit 1
fi

# Validate dataset name
if [[ "$DATASET" != "gsm8k" && "$DATASET" != "math500" && "$DATASET" != "collegemath" ]]; then
    echo "❌ Invalid dataset name: $DATASET"
    echo "Valid options: gsm8k, math500, collegemath"
    exit 1
fi

# Map model choice to actual model name
case $MODEL_CHOICE in
    1)
        MODEL_ARG="$MODEL_1"
        ;;
    2)
        MODEL_ARG="$MODEL_2"
        ;;
    3)
        MODEL_ARG="$MODEL_3"
        ;;
    4)
        MODEL_ARG="$MODEL_4"
        ;;
    *)
        echo "❌ Invalid model choice: $MODEL_CHOICE"
        exit 1
        ;;
esac

# Construct command
CMD="python src/inference_self.py --model_1 $MODEL_ARG"
CMD+=" --num_query $NUM_QUERY"
CMD+=" --round $ROUND"
CMD+=" --temp $TEMP"
CMD+=" --dataset $DATASET"

# Add adversarial prefix if dataset is advbench
if [ "$DATASET" == "advbench" ]; then
    CMD+=" --advprefix $ADAPREFIX"
    CMD+=" --advprefix2 $ADAPREFIX2"
fi

# Select self-consistency or self-refinement
if [ "$SELF_CONSISTENCY" = true ]; then
    CMD+=" --self_consistency"
    CMD+=" --num_samples $NUM_SAMPLES"
fi

if [ "$SELF_REFINEMENT" = true ]; then
    CMD+=" --self_REFINEMENT"
    CMD+=" --round $NUM_ROUND"
fi


# Run the script
echo "🚀 Running: $CMD"
eval $CMD

echo "✅ All processes completed!" 
