#!/bin/bash
export VLLM_WORKER_MULTIPROC_METHOD=spawn

# Model names
MODEL_1="Qwen/Qwen2.5-7B-Instruct"
MODEL_2="Qwen/Qwen2.5-3B-Instruct"
MODEL_3="Qwen/Qwen2.5-32B-Instruct"
MODEL_4="Qwen/Qwen2.5-14B-Instruct"
MODEL_5="Qwen/Qwen2.5-1.5B-Instruct"
MODEL_6="Qwen/Qwen2.5-72B-Instruct"

# Default values
NUM_QUERY=500
ROUND=1
TEMP=0.7
COT=false
SELF_CONSISTENCY=true
SELF_REFINEMENT=false
NUM_SAMPLES=7
REFINEMENT_ROUNDS=16
DATASET=""
DATA_TOPIC="algebra"  # Only used for CollegeMath dataset
USE_SERVER=false
GPU_INDEX=0
CLUSTER2=false
CLUSTER1=false  # Add cluster1 computing option

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --model)
            MODEL_CHOICE="$2"
            shift 2
            ;;
        --dataset)
            DATASET="$2"
            shift 2
            ;;
        --num_query)
            NUM_QUERY="$2"
            shift 2
            ;;
        --round)
            ROUND="$2"
            shift 2
            ;;
        --temp)
            TEMP="$2"
            shift 2
            ;;
        --cot)
            COT=true
            shift
            ;;
        --self_consistency)
            SELF_CONSISTENCY=true
            shift
            ;;
        --self_refinement)
            SELF_REFINEMENT=true
            SELF_CONSISTENCY=false
            shift
            ;;
        --num_samples)
            NUM_SAMPLES="$2"
            shift 2
            ;;
        --refinement_rounds)
            REFINEMENT_ROUNDS="$2"
            shift 2
            ;;
        --data_topic)
            DATA_TOPIC="$2"
            shift 2
            ;;
        --use_server)
            USE_SERVER=true
            shift
            ;;
        --cluster2)
            CLUSTER2=true
            shift
            ;;
        --gpu)
            GPU_INDEX="$2"
            shift 2
            ;;
        --cluster1)
            CLUSTER1=true
            shift
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

# Validate required arguments
if [ -z "$MODEL_CHOICE" ] || [ -z "$DATASET" ]; then
    echo "Usage: bash run_math_single.sh --model <model_choice> --dataset <dataset> [options]"
    echo "  Required arguments:"
    echo "    --model <choice>         : Model choice (1-4)"
    echo "    --dataset <name>         : Dataset name (gsm8k, math500, collegemath, aime2024, aime2025)"
    echo "  Optional arguments:"
    echo "    --num_query <number>     : Number of queries (default: 500)"
    echo "    --round <number>         : Number of rounds (default: 1)"
    echo "    --temp <number>          : Temperature (default: 0.7)"
    echo "    --cot                    : Enable Chain-of-Thought"
    echo "    --self_consistency       : Enable self-consistency"
    echo "    --self_refinement        : Enable self-refinement"
    echo "    --num_samples <number>   : Number of samples for self-consistency (default: 7)"
    echo "    --refinement_rounds <number> : Number of refinement rounds (default: 16)"
    echo "    --data_topic <topic>     : Topic for collegemath dataset (default: algebra)"
    echo "                               Options: algebra, calculus, precalculus, differential_equation,"
    echo "                                        linear_algebra, probability, vector_calculus"
    echo "    --use_server             : Use vLLM server APIs instead of direct integration"
    echo "    --cluster2                 : Load model from /model-weights/ directory instead of HuggingFace"
    echo "    --gpu <index>            : GPU index for model (default: 0)"
    echo "    --cluster1                  : Enable cluster1 computing configuration"
    echo "  Model choices:"
    echo "    1: Qwen2.5-1.5B"
    echo "    2: Qwen2.5-3B"
    echo "    3: Llama-3.2-3B"
    echo "    4: Llama-3.1-8B"
    exit 1
fi

# Validate dataset name
if [[ "$DATASET" != "gsm8k" && "$DATASET" != "math500" && "$DATASET" != "collegemath" && "$DATASET" != "aime2024" && "$DATASET" != "aime2025" ]]; then
    echo "❌ Invalid dataset name: $DATASET"
    echo "Valid options: gsm8k, math500, collegemath, aime2024, aime2025"
    exit 1
fi

# Map model choice to actual model name
case $MODEL_CHOICE in
    1)
        MODEL_ARG="$MODEL_1"
        ;;
    2)
        MODEL_ARG="$MODEL_2"
        ;;
    3)
        MODEL_ARG="$MODEL_3"
        ;;
    4)
        MODEL_ARG="$MODEL_4"
        ;;
    5)
        MODEL_ARG="$MODEL_5"
        ;;
    *)
        echo "❌ Invalid model choice: $MODEL_CHOICE"
        exit 1
        ;;
esac

# Construct command
CMD="python src/inference_self.py --model_1 $MODEL_ARG"
CMD+=" --gpu_1 $GPU_INDEX"
CMD+=" --num_query $NUM_QUERY"
CMD+=" --round $ROUND"
CMD+=" --temp $TEMP"
CMD+=" --dataset $DATASET"

# Add server option if enabled
if [ "$USE_SERVER" = true ]; then
    CMD+=" --use_server"
fi

# Add cluster2 option if enabled
if [ "$CLUSTER2" = true ]; then
    CMD+=" --cluster2"
fi

# Add cluster1 option if enabled
if [ "$CLUSTER1" = true ]; then
    CMD+=" --cluster1"
fi

# Add data_topic if dataset is collegemath
if [ "$DATASET" == "collegemath" ]; then
    CMD+=" --data_topic $DATA_TOPIC"
fi

if [ "$COT" = true ]; then
    CMD+=" --cot"
fi

if [ "$SELF_CONSISTENCY" = true ]; then
    CMD+=" --self_consistency"
    CMD+=" --num_samples $NUM_SAMPLES"
fi

if [ "$SELF_REFINEMENT" = true ]; then
    CMD+=" --self_refinement"
    CMD+=" --refinement_rounds $REFINEMENT_ROUNDS"
fi

# Run the script
echo "🚀 Running: $CMD"
eval $CMD

echo "✅ All processes completed!" 