# Adopted from https://github.com/lm-sys/FastChat. Below is the original copyright:
# Adopted from tatsu-lab@stanford_alpaca. Below is the original copyright:
#    Copyright 2023 Rohan Taori, Ishaan Gulrajani, Tianyi Zhang, Yann Dubois, Xuechen Li
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.

import os
import logging
import torch
import wandb
import transformers
import numpy as np

from typing import Dict, Optional
from dataclasses import dataclass, field


from llava.model import *
from llava.data.training_datasets import *
from llava import conversation as conversation_lib
from llava.train.inner_loop_optimizers import LSLRGradientDescentLearningRule
from llava.train.llava_trainer import LLaVATrainer, MAPD_LLaVA_Trainer, TasksDatasetCallback, InnerLRCallback, ICTDatasetCallback, ICTZeroShotCallback, SaveModelCallback

random.seed(0)
torch.manual_seed(222)
torch.cuda.manual_seed_all(222)
np.random.seed(222)

torch.backends.cudnn.deterministic = True
torch.backends.cudnn.benchmarks = False

local_rank = None
def rank0_print(*args):
    if local_rank == 0:
        print(*args)


@dataclass
class ModelArguments:
    model_name_or_path: Optional[str] = field(default="facebook/opt-125m")
    version: Optional[str] = field(default="v0")
    freeze_backbone: bool = field(default=False)
    tune_mm_mlp_adapter: bool = field(default=False)
    vision_tower: Optional[str] = field(default=None)
    mm_vision_select_layer: Optional[int] = field(default=-1)   # default to the last layer
    pretrain_mm_mlp_adapter: Optional[str] = field(default=None)
    mm_projector_type: Optional[str] = field(default='linear')
    mm_use_im_start_end: bool = field(default=False)
    mm_use_im_patch_token: bool = field(default=True)
    mm_patch_merge_type: Optional[str] = field(default='flat')
    mm_vision_select_feature: Optional[str] = field(default="patch")
    prefix_length: int = field(default=64)
    extrapolate_lr: bool = field(default=False)
    add_dropout: bool = field(default=False)
    task_head: bool = field(default=False)


@dataclass
class TrainingArguments(transformers.TrainingArguments):
    cache_dir: Optional[str] = field(default=None)
    optim: str = field(default="adamw_torch")
    remove_unused_columns: bool = field(default=False)
    freeze_mm_mlp_adapter: bool = field(default=False)
    mpt_attn_impl: Optional[str] = field(default="triton")
    model_max_length: int = field(
        default=512,
        metadata={
            "help":
            "Maximum sequence length. Sequences will be right padded (and possibly truncated)."
        },
    )
    double_quant: bool = field(
        default=True,
        metadata={"help": "Compress the quantization statistics through double quantization."}
    )
    quant_type: str = field(
        default="nf4",
        metadata={"help": "Quantization data type to use. Should be one of `fp4` or `nf4`."}
    )
    bits: int = field(
        default=16,
        metadata={"help": "How many bits to use."}
    )
    lora_enable: bool = False
    lora_r: int = 64
    lora_alpha: int = 16
    lora_dropout: float = 0.05
    lora_weight_path: str = ""
    lora_bias: str = "none"
    mm_projector_lr: Optional[float] = None
    group_by_modality_length: bool = field(default=False)
    meta_learning_rate: Optional[float] = field(default=1e-3)
    task_learning_rate: Optional[float] = field(default=0.1)
    min_learning_rate: Optional[float] = field(default=0.00001)
    number_of_training_steps_per_iter: Optional[int] = field(default=5)
    number_of_evaluation_steps_per_iter: Optional[int] = field(default=5)
    learnable_per_layer_per_step_inner_loop_learning_rate: Optional[bool] = field(default=False)
    enable_inner_loop_optimizable_bn_params: Optional[bool] = field(default=False)
    second_order: Optional[bool] = field(default=True)
    first_order_to_second_order_epoch: Optional[int] = field(default=-1)
    use_multi_step_loss_optimization: Optional[bool] = field(default=False)
    multi_step_loss_num_epochs: Optional[int] = field(default=1)
    run_name: Optional[str] = field(default="trial1")
    project_name: Optional[str] = field(default="MAPD_train")
    grad_acc_part: Optional[int] = field(default=2, metadata={"help": "Divides inner loop batch for inner \
                                                              loop grad accumulation. CAUTION: Can only be used \
                                                              if inner_checkpointing is set to True"})
    inner_checkpointing: Optional[bool] = field(default=True, metadata={"help": "Initiates maml inner loop checkpointing"})


def maybe_zero_3(param, ignore_status=False, name=None):
    from deepspeed import zero
    from deepspeed.runtime.zero.partition_parameters import ZeroParamStatus
    if hasattr(param, "ds_id"):
        if param.ds_status == ZeroParamStatus.NOT_AVAILABLE:
            if not ignore_status:
                logging.warning(f"{name}: param.ds_status != ZeroParamStatus.NOT_AVAILABLE: {param.ds_status}")
        with zero.GatheredParameters([param]):
            param = param.data.detach().cpu().clone()
    else:
        param = param.detach().cpu().clone()
    return param


# Borrowed from peft.utils.get_peft_model_state_dict
def get_peft_state_maybe_zero_3(named_params, bias):
    if bias == "none":
        to_return = {k: t for k, t in named_params if "lora_" in k}
    elif bias == "all":
        to_return = {k: t for k, t in named_params if "lora_" in k or "bias" in k}
    elif bias == "lora_only":
        to_return = {}
        maybe_lora_bias = {}
        lora_bias_names = set()
        for k, t in named_params:
            if "lora_" in k:
                to_return[k] = t
                bias_name = k.split("lora_")[0] + "bias"
                lora_bias_names.add(bias_name)
            elif "bias" in k:
                maybe_lora_bias[k] = t
        for k, t in maybe_lora_bias:
            if bias_name in lora_bias_names:
                to_return[bias_name] = t
    else:
        raise NotImplementedError
    to_return = {k: maybe_zero_3(v, ignore_status=True) for k, v in to_return.items()}
    return to_return


def get_peft_state_non_lora_maybe_zero_3(named_params, require_grad_only=True):
    to_return = {k: t for k, t in named_params if "lora_" not in k}
    if require_grad_only:
        to_return = {k: t for k, t in to_return.items() if t.requires_grad}
    to_return = {k: maybe_zero_3(v, ignore_status=True).cpu() for k, v in to_return.items()}
    return to_return


def get_mm_adapter_state_maybe_zero_3(named_params, keys_to_match):
    to_return = {k: t for k, t in named_params if any(key_match in k for key_match in keys_to_match)}
    to_return = {k: maybe_zero_3(v, ignore_status=True).cpu() for k, v in to_return.items()}
    return to_return


def find_all_linear_names(model):
    cls = torch.nn.Linear
    lora_module_names = set()
    multimodal_keywords = ['mm_projector', 'vision_tower', 'vision_resampler']
    for name, module in model.named_modules():
        if any(mm_keyword in name for mm_keyword in multimodal_keywords):
            continue
        if isinstance(module, cls):
            names = name.split('.')
            lora_module_names.add(names[0] if len(names) == 1 else names[-1])

    if 'lm_head' in lora_module_names: # needed for 16-bit
        lora_module_names.remove('lm_head')
    return list(lora_module_names)


def safe_save_model_for_hf_trainer(trainer: transformers.Trainer,
                                   output_dir: str, task_head: bool):
    """Collects the state dict and dump to disk."""
    
    if getattr(trainer.args, "tune_mm_mlp_adapter", False):
        # Only save Adapter
        if task_head:
            keys_to_match = ['mm_projector', 'lm_head_w']
        else:
            keys_to_match = ['mm_projector']
        
        if getattr(trainer.args, "use_im_start_end", False):
            keys_to_match.extend(['embed_tokens', 'embed_in'])

        weight_to_save = get_mm_adapter_state_maybe_zero_3(trainer.model.named_parameters(), keys_to_match)
        trainer.model.config.save_pretrained(output_dir)

        current_folder = output_dir.split('/')[-1]
        parent_folder = os.path.dirname(output_dir)
        if trainer.args.local_rank == 0 or trainer.args.local_rank == -1:
            if current_folder.startswith('checkpoint-'):
                mm_projector_folder = os.path.join(parent_folder, "mm_projector")
                os.makedirs(mm_projector_folder, exist_ok=True)
                torch.save(weight_to_save, os.path.join(mm_projector_folder, f'{current_folder}.bin'))
            else:
                torch.save(weight_to_save, os.path.join(output_dir, f'mm_projector.bin'))
        return

    if trainer.deepspeed:
        torch.cuda.synchronize()
        trainer.save_model(output_dir)
        return

    state_dict = trainer.model.state_dict()
    if trainer.args.should_save:
        cpu_state_dict = {
            key: value.cpu()
            for key, value in state_dict.items()
        }
        del state_dict
        trainer._save(output_dir, state_dict=cpu_state_dict)  # noqa


def smart_tokenizer_and_embedding_resize(
    special_tokens_dict: Dict,
    tokenizer: transformers.PreTrainedTokenizer,
    model: transformers.PreTrainedModel,
):
    """Resize tokenizer and embedding.

    Note: This is the unoptimized version that may make your embedding size not be divisible by 64.
    """
    num_new_tokens = tokenizer.add_special_tokens(special_tokens_dict)
    model.resize_token_embeddings(len(tokenizer))

    if num_new_tokens > 0:
        input_embeddings = model.get_input_embeddings().weight.data
        output_embeddings = model.get_output_embeddings().weight.data

        input_embeddings_avg = input_embeddings[:-num_new_tokens].mean(
            dim=0, keepdim=True)
        output_embeddings_avg = output_embeddings[:-num_new_tokens].mean(
            dim=0, keepdim=True)

        input_embeddings[-num_new_tokens:] = input_embeddings_avg
        output_embeddings[-num_new_tokens:] = output_embeddings_avg



def make_supervised_data_module(tokenizer: transformers.PreTrainedTokenizer,
                                data_args, local_rank) -> Dict:
    """Make dataset and collator for supervised fine-tuning."""
    
    if data_args.data_mode == "mapd":
        train_dataset = MAPDDataset(mode="train", tokenizer=tokenizer, data_args=data_args, local_rank=local_rank)
        val_dataset = None
        #val_dataset = MAPDDataset(mode="val", tokenizer=tokenizer, data_args=data_args, local_rank=local_rank)
        data_collator = DataCollatorForMAPDDataset(tokenizer=tokenizer)
    elif data_args.data_mode == "mltasks":
        train_dataset = MultiTaskDataset(mode="train", tokenizer=tokenizer, data_args=data_args, local_rank=local_rank)
        val_dataset = None
        #val_dataset = MultiTaskDataset(mode="val", tokenizer=tokenizer, data_args=data_args, local_rank=local_rank)
        data_collator = DataCollatorForMultiTaskDataset(tokenizer=tokenizer)   
    elif data_args.data_mode == "supervised":  
        train_dataset = SupervisedDataset(mode="train", tokenizer=tokenizer, data_args=data_args)
        val_dataset=None
        #val_dataset = SupervisedDataset(mode="val", tokenizer=tokenizer, data_args=data_args)
        data_collator = DataCollatorForSupervisedDataset(tokenizer=tokenizer)
    elif data_args.data_mode == "ict":  
        train_dataset = ICTDataset(mode="train", tokenizer=tokenizer, data_args=data_args, local_rank=local_rank)
        val_dataset=None
        #val_dataset = ICTDataset(mode="val", tokenizer=tokenizer, data_args=data_args, local_rank=local_rank)
        data_collator = DataCollatorForICTDataset(tokenizer=tokenizer)

    return dict(train_dataset=train_dataset,
                eval_dataset=val_dataset,
                data_collator=data_collator)


def train(attn_implementation=None):
    
    global local_rank
    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments))
    model_args, data_args, training_args = parser.parse_args_into_dataclasses()
    local_rank = training_args.local_rank
    compute_dtype = (torch.float16 if training_args.fp16 else (torch.bfloat16 if training_args.bf16 else torch.float32))
    training_args.output_dir = os.path.join(training_args.output_dir, training_args.run_name)
    
    if local_rank == 0:
        wandb.init(project=training_args.project_name, name=training_args.run_name)

    bnb_model_from_pretrained_args = {}
    if training_args.bits in [4, 8]:
        from transformers import BitsAndBytesConfig
        bnb_model_from_pretrained_args.update(dict(
            device_map={"": training_args.device},
            load_in_4bit=training_args.bits == 4,
            load_in_8bit=training_args.bits == 8,
            quantization_config=BitsAndBytesConfig(
                load_in_4bit=training_args.bits == 4,
                load_in_8bit=training_args.bits == 8,
                llm_int8_skip_modules=["mm_projector"],
                llm_int8_threshold=6.0,
                llm_int8_has_fp16_weight=False,
                bnb_4bit_compute_dtype=compute_dtype,
                bnb_4bit_use_double_quant=training_args.double_quant,
                bnb_4bit_quant_type=training_args.quant_type # {'fp4', 'nf4'}
            )
        ))
    
    if model_args.vision_tower is not None:
        if 'mpt' in model_args.model_name_or_path:
            config = transformers.AutoConfig.from_pretrained(model_args.model_name_or_path, trust_remote_code=True)
            config.attn_config['attn_impl'] = training_args.mpt_attn_impl
            model = LlavaMptForCausalLM.from_pretrained(
                model_args.model_name_or_path,
                config=config,
                cache_dir=training_args.cache_dir,
                **bnb_model_from_pretrained_args
            )
        elif 'qwen' in model_args.model_name_or_path:
            model = LlavaQwen2ForCausalLM.from_pretrained(
                model_args.model_name_or_path,
                cache_dir=training_args.cache_dir,
                attn_implementation=attn_implementation,
                torch_dtype=(torch.bfloat16 if training_args.bf16 else None),
                low_cpu_mem_usage=False,
                **bnb_model_from_pretrained_args
            )
        else:
            # only LLAVA LLAMA initialized here and NOT projector/vision_encoder due to the LLAMA config 
            model = LlavaLlamaForCausalLM.from_pretrained(
                model_args.model_name_or_path,
                cache_dir=training_args.cache_dir,
                attn_implementation=attn_implementation,
                torch_dtype=(torch.bfloat16 if training_args.bf16 else None),
                **bnb_model_from_pretrained_args
            )
    else:
        if 'qwen' in model_args.model_name_or_path:
            model = transformers.Qwen2ForCausalLM.from_pretrained(
                model_args.model_name_or_path,
                cache_dir=training_args.cache_dir,
                attn_implementation=attn_implementation,
                torch_dtype=(torch.bfloat16 if training_args.bf16 else None),
                **bnb_model_from_pretrained_args
        )
        model = transformers.LlamaForCausalLM.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            attn_implementation=attn_implementation,
            torch_dtype=(torch.bfloat16 if training_args.bf16 else None),
            **bnb_model_from_pretrained_args
        )
    model.config.use_cache = False
    model.config.prefix_length = model_args.prefix_length
    data_args.prefix_length = model_args.prefix_length
    model.config.add_dropout = model_args.add_dropout
    model.config.task_head = model_args.task_head

    if model_args.freeze_backbone:
        model.model.requires_grad_(False)

    if training_args.bits in [4, 8]:
        from peft import prepare_model_for_kbit_training
        model.config.torch_dtype=(torch.float32 if training_args.fp16 else (torch.bfloat16 if training_args.bf16 else torch.float32))
        model = prepare_model_for_kbit_training(model, use_gradient_checkpointing=training_args.gradient_checkpointing)
    
    if hasattr(model, "enable_input_require_grads"):
        model.enable_input_require_grads()
    else:
        def make_inputs_require_grad(module, input, output):
            output.requires_grad_(True)
        model.get_input_embeddings().register_forward_hook(make_inputs_require_grad)
    
    if training_args.lora_enable:
        from peft import LoraConfig, get_peft_model
        lora_config = LoraConfig(
            r=training_args.lora_r,
            lora_alpha=training_args.lora_alpha,
            target_modules=find_all_linear_names(model),
            lora_dropout=training_args.lora_dropout,
            bias=training_args.lora_bias,
            task_type="CAUSAL_LM",
        )
        if training_args.bits == 16:
            if training_args.bf16:
                model.to(torch.bfloat16)
            if training_args.fp16:
                model.to(torch.float16)
        rank0_print("Adding LoRA adapters...")
        model = get_peft_model(model, lora_config)

    if 'mpt' in model_args.model_name_or_path:
        tokenizer = transformers.AutoTokenizer.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            model_max_length=training_args.model_max_length,
            padding_side="right"
        )
    elif 'qwen' in model_args.model_name_or_path:
        tokenizer = transformers.AutoTokenizer.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            model_max_length=training_args.model_max_length,
            padding_side="right"
        )
    else:
        tokenizer = transformers.AutoTokenizer.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            model_max_length=training_args.model_max_length,
            padding_side="right",
            use_fast=False,
        )
    
    if model_args.version == "v0":
        if tokenizer.pad_token is None:
            smart_tokenizer_and_embedding_resize(
                special_tokens_dict=dict(pad_token="[PAD]"),
                tokenizer=tokenizer,
                model=model,
            )
    elif model_args.version == "v0.5":
        tokenizer.pad_token = tokenizer.unk_token
    else:
        tokenizer.pad_token = tokenizer.unk_token
        
        if model_args.version in conversation_lib.conv_templates:
            conversation_lib.default_conversation = conversation_lib.conv_templates[model_args.version]
        else:
            conversation_lib.default_conversation = conversation_lib.conv_templates["vicuna_v1"]


    if model_args.vision_tower is not None:
        model.get_model().initialize_vision_modules(
            model_args=model_args,
            fsdp=training_args.fsdp
        )
        
        vision_tower = model.get_vision_tower()
        vision_tower.to(dtype=torch.bfloat16 if training_args.bf16 else torch.float16, device=training_args.device)

        data_args.image_processor = vision_tower.image_processor
        data_args.is_multimodal = True
        
        model.config.image_aspect_ratio = data_args.image_aspect_ratio
        model.config.tokenizer_padding_side = tokenizer.padding_side
        model.config.tokenizer_model_max_length = tokenizer.model_max_length
    
        model.config.tune_mm_mlp_adapter = training_args.tune_mm_mlp_adapter = model_args.tune_mm_mlp_adapter
        if model_args.tune_mm_mlp_adapter:
            model.requires_grad_(False)
            for p in model.get_model().mm_projector.parameters():
                p.requires_grad = True

        model.config.freeze_mm_mlp_adapter = training_args.freeze_mm_mlp_adapter
        if training_args.freeze_mm_mlp_adapter:
            for p in model.get_model().mm_projector.parameters():
                p.requires_grad = False

        if training_args.bits in [4, 8]:
            model.get_model().mm_projector.to(dtype=compute_dtype, device=training_args.device)

        model.config.mm_use_im_start_end = data_args.mm_use_im_start_end = model_args.mm_use_im_start_end
        model.config.mm_projector_lr = training_args.mm_projector_lr
        training_args.use_im_start_end = model_args.mm_use_im_start_end
        model.config.mm_use_im_patch_token = model_args.mm_use_im_patch_token
        model.initialize_vision_tokenizer(model_args, tokenizer=tokenizer)
    
    model.init_task_head()

    if data_args.data_mode == "mapd":
        # Initialize inner loop optimizer class
        model.inner_loop_optimizer = LSLRGradientDescentLearningRule(device=training_args.device,
                                                                        init_learning_rate=training_args.task_learning_rate,
                                                                        total_num_inner_loop_steps=training_args.number_of_training_steps_per_iter,
                                                                        use_learnable_learning_rates=training_args.learnable_per_layer_per_step_inner_loop_learning_rate,
                                                                        extrapolate_lr=model_args.extrapolate_lr)
        
        def get_inner_loop_parameter_dict(params):
            """
            Returns a dictionary with the parameters to use for inner loop updates.
            :param params: A dictionary of the network's parameters.
            :return: A dictionary of the parameters to use for the inner loop optimization process.
            """
            return {
                name: param
                for name, param in params
                if param.requires_grad
                and (
                    not training_args.enable_inner_loop_optimizable_bn_params
                    and "norm_layer" not in name
                    or training_args.enable_inner_loop_optimizable_bn_params
                )
            }

        inner_lr_weights=None
        if model_args.pretrain_mm_mlp_adapter is not None:
            mm_weights = torch.load(model_args.pretrain_mm_mlp_adapter, map_location='cpu')
            inner_lr_weights = {k:v for k,v in mm_weights.items() if "inner_loop_optimizer" in k}
        
        model.inner_loop_optimizer.initialise(
            names_weights_dict=get_inner_loop_parameter_dict(params=model.named_parameters()), pretrained_weights_dict=inner_lr_weights)

        model.model.mm_projector.to(dtype=torch.bfloat16 if training_args.bf16 else torch.float16, device=training_args.device)   
        model.inner_loop_optimizer.to(dtype=torch.bfloat16 if training_args.bf16 else torch.float16, device=training_args.device)
        if model_args.task_head:
            model.lm_head_w.to(device=training_args.device)

    if training_args.bits in [4, 8]:
        from peft.tuners.lora import LoraLayer
        for name, module in model.named_modules():
            if isinstance(module, LoraLayer):
                if training_args.bf16:
                    module = module.to(torch.bfloat16)
            if 'norm' in name:
                module = module.to(torch.float32)
            if 'lm_head' in name or 'embed_tokens' in name:
                if hasattr(module, 'weight'):
                    if training_args.bf16 and module.weight.dtype == torch.float32:
                        module = module.to(torch.bfloat16)

    data_module = make_supervised_data_module(tokenizer=tokenizer,
                                              data_args=data_args, local_rank=local_rank)

    if data_args.data_mode == "mapd":
        trainer = MAPD_LLaVA_Trainer(model=model,
                        tokenizer=tokenizer,
                        args=training_args,
                        callbacks=[TasksDatasetCallback(data_module["train_dataset"]),
                                #TasksDatasetCallback(data_module["eval_dataset"]),
                                InnerLRCallback(),
                                SaveModelCallback()],
                        **data_module)
    elif data_args.data_mode == "mltasks":
        trainer = LLaVATrainer(model=model,
                        tokenizer=tokenizer,
                        args=training_args,
                        callbacks=[TasksDatasetCallback(data_module["train_dataset"]),
                                #TasksDatasetCallback(data_module["eval_dataset"]),
                                SaveModelCallback()],
                        **data_module)
    elif data_args.data_mode == "supervised":
        trainer = LLaVATrainer(model=model,
                        tokenizer=tokenizer,
                        args=training_args,
                        **data_module,
                        callbacks=[SaveModelCallback()]
                        )
    elif data_args.data_mode == "ict":
        trainer = LLaVATrainer(model=model,
                        tokenizer=tokenizer,
                        args=training_args,
                        callbacks=[ICTDatasetCallback(data_module["train_dataset"]),
                                   #ICTDatasetCallback(data_module["eval_dataset"]),
                                   ICTZeroShotCallback(data_module["train_dataset"]),
                                   #ICTZeroShotCallback(data_module["eval_dataset"]),
                                   SaveModelCallback()],
                        **data_module
                        )
             
    trainer.train()
    trainer.save_state()
    model.config.use_cache = True

    if training_args.lora_enable:
        state_dict = get_peft_state_maybe_zero_3(
            model.named_parameters(), training_args.lora_bias
        )
        non_lora_state_dict = get_peft_state_non_lora_maybe_zero_3(
            model.named_parameters()
        )
        if training_args.local_rank == 0 or training_args.local_rank == -1:
            model.config.save_pretrained(training_args.output_dir)
            model.save_pretrained(training_args.output_dir, state_dict=state_dict)
            torch.save(non_lora_state_dict, os.path.join(training_args.output_dir, 'non_lora_trainables.bin'))
    else:
        safe_save_model_for_hf_trainer(trainer=trainer,
                                       output_dir=training_args.output_dir, task_head=model_args.task_head)


if __name__ == "__main__":
    train()
