# ditto
from typing import Union

import torch

from torch.autograd import Variable


class DittoSSLOptimizer:
    def __init__(self, device, local_model, global_model, kwargs):
        self.device = device
        self.local_model = local_model
        self.global_model = global_model
        self.lr = kwargs['lr']
        self.g_lambda = kwargs['lambda']
        self.ssl_method = kwargs["ssl_method"]
        self.basic_opt = kwargs['client_optimizer']
        self.accumulation_steps = kwargs['accumulation_steps']
        self.momentum = kwargs['momentum']
        self.wd = kwargs['wd']

        if self.basic_opt == "sgd":
            self.g_optimizer = torch.optim.SGD(self.global_model.parameters(), lr=self.lr,
                                               weight_decay=self.wd, momentum=self.momentum)
            self.l_optimizer = torch.optim.SGD(self.local_model.parameters(), lr=self.lr,
                                               weight_decay=self.wd, momentum=self.momentum)
            # TODO(ZY): a local opt does not exist
        else:
            # TODO(ZY): Why add this filter, also why there is parameter without grad?
            self.g_optimizer = torch.optim.Adam(self.global_model.parameters(), lr=self.lr, weight_decay=0.001,
                                                amsgrad=True)
            self.l_optimizer = torch.optim.Adam(self.local_model.parameters(), lr=self.lr, weight_decay=0.001,
                                                amsgrad=True)

    def step(self, x_accumulator):

        # fedavg
        for (x1, x2) in x_accumulator:
            x1, x2 = x1.to(self.device), x2.to(self.device)
            z1, z1_, z2, z2_ = self.global_model.predict(x1, x2)
            global_loss = self.global_model.loss_fn(z1, z1_, z2, z2_)
            global_loss = global_loss / self.accumulation_steps
            global_loss.backward()
        self.g_optimizer.step()
        self.g_optimizer.zero_grad()

        # outer optimization (optimize personalized model)
        total_loss = 0.0
        for (x1, x2) in x_accumulator:
            x1, x2 = x1.to(self.device), x2.to(self.device)
            z1, z1_, z2, z2_ = self.local_model.predict(x1, x2)
            local_loss = self.local_model.loss_fn(z1, z1_, z2, z2_)
            local_loss = local_loss / self.accumulation_steps

            total_loss += local_loss

            local_loss.backward()

        reg_loss = 0.0
        for (p, g_p) in zip(self.local_model.trainable_parameters(),
                            self.global_model.trainable_parameters()):
            reg_loss += (self.g_lambda * 0.5) * torch.linalg.norm(p - g_p.data) ** 2
        reg_loss.backward()

        self.l_optimizer.step()
        self.l_optimizer.zero_grad()

        return total_loss
