#!/bin/bash
"""
Demo script showing how to use trained steering vectors.

This script demonstrates how to apply steering vectors to user prompts
using a trained model. It loads steering vectors from a training experiment
and applies them to sample prompts.

Usage:
    ./scripts/demo_steering.sh                           # Use default settings
    ./scripts/demo_steering.sh <experiment_dir>          # Use specific experiment
    ./scripts/demo_steering.sh <experiment_dir> <vector_idx> <coefficient>
"""

set -e

# Default configuration
DEFAULT_VECTOR_IDX=0
DEFAULT_COEFFICIENT=1.5
DEFAULT_PROMPT="How do I handle a difficult situation at work?"

# Parse command line arguments
if [ $# -eq 0 ]; then
    echo "Please provide path to experiment directory containing steering_vectors.pt"
    echo "Usage: $0 <experiment_dir> [vector_idx] [coefficient]"
    echo ""
    echo "Example experiment directories can be found in outputs/ after training"
    echo "Example: $0 outputs/demo/microsoft_DialoGPT_small/20241225_123456_demo_123456"
    exit 1
fi

EXPERIMENT_DIR="$1"
VECTOR_IDX="${2:-$DEFAULT_VECTOR_IDX}"
COEFFICIENT="${3:-$DEFAULT_COEFFICIENT}"

VECTORS_FILE="$EXPERIMENT_DIR/steering_vectors.pt"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(dirname "$SCRIPT_DIR")"
STEERING_SCRIPT="$SCRIPT_DIR/run_steering.py"

# Change to project root
cd "$PROJECT_ROOT"

# Check if files exist
if [[ ! -f "$VECTORS_FILE" ]]; then
    echo "Error: Steering vectors file not found: $VECTORS_FILE"
    echo ""
    echo "Available experiment directories:"
    if [[ -d "outputs" ]]; then
        find outputs -name "steering_vectors.pt" -exec dirname {} \; | head -5
    else
        echo "No outputs directory found. Please run training first with:"
        echo "  ./scripts/train.sh"
    fi
    exit 1
fi

if [[ ! -f "$EXPERIMENT_DIR/config.yaml" ]]; then
    echo "Error: Config file not found: $EXPERIMENT_DIR/config.yaml"
    exit 1
fi

echo "SteerCLR Steering Vector Demo"
echo "=" * 50
echo "Using experiment: $EXPERIMENT_DIR"
echo "Vector index: $VECTOR_IDX"
echo "Coefficient: $COEFFICIENT"
echo ""

# Set up Python path
export PYTHONPATH="$PROJECT_ROOT/src:$PYTHONPATH"

# Example 1: Helpful vs. unhelpful behavior
echo "EXAMPLE 1: Helpful vs. unhelpful behavior"
echo "-" * 40
python "$STEERING_SCRIPT" \
    --vectors "$VECTORS_FILE" \
    --vector-idx "$VECTOR_IDX" \
    --coefficient "$COEFFICIENT" \
    --prompt "How do I improve my productivity at work?" \
    --max-new-tokens 128 \
    --temperature 0.0

echo ""
echo ""

# Example 2: Refusal behavior
echo "EXAMPLE 2: AI refusal behavior"
echo "-" * 40
python "$STEERING_SCRIPT" \
    --vectors "$VECTORS_FILE" \
    --vector-idx "$VECTOR_IDX" \
    --coefficient "$COEFFICIENT" \
    --prompt "What's your favorite hobby?" \
    --max-new-tokens 128 \
    --temperature 0.0

echo ""
echo ""

# Example 3: Custom prompt
echo "EXAMPLE 3: Custom prompt"
echo "-" * 40
python "$STEERING_SCRIPT" \
    --vectors "$VECTORS_FILE" \
    --vector-idx "$VECTOR_IDX" \
    --coefficient "$COEFFICIENT" \
    --prompt "$DEFAULT_PROMPT" \
    --max-new-tokens 128 \
    --temperature 0.0

echo ""
echo ""
echo "Demo completed!"
echo ""
echo "To try different prompts, run:"
echo "  python scripts/run_steering.py --vectors $VECTORS_FILE --vector-idx $VECTOR_IDX --coefficient $COEFFICIENT --prompt \"Your custom prompt here\""
