"""
Defines the BYOL-SGHM model for CIFAR-20 classification using a ResNet-34 backbone.

The BYOL-SGHM model is a self-supervised learning approach that combines BYOL (Bootstrap Your Own Latent) and SGHM (Stochastic Gradient Hamiltonian Monte Carlo) techniques to learn effective representations from unlabeled data. This model is specifically designed for the CIFAR-20 dataset, which consists of 20 classes of natural images.

The ResNet-34 backbone is used as the feature extractor, and the BYOL and SGHM components are added on top to enable self-supervised learning.
"""
# -*- coding: UTF-8 -*-


from .byol import BYOL
from models import model_dict



@model_dict.register('byol_sghm_cifar20_r34_org')
class BYOL_CIFAR20_R34(BYOL):
    """
    Implements a cosine annealing learning rate scheduler with optional warmup.
    
    The learning rate is initialized to the provided `learning_rate` value, and then
    annealed using a cosine schedule over the specified number of epochs. An optional
    warmup period can be specified, where the learning rate linearly increases from
    a lower value to the initial `learning_rate`.
    
    Args:
        n_iter (int): The current iteration number.
    
    Returns:
        float: The learning rate for the current iteration.
    """

    def cosine_annealing_LR(self, n_iter):
        opt = self.opt

        import math
        epoch = (n_iter - 1) // self.iter_per_epoch
        eta_min = 0
        warmup_from = 0.
        # warmup
        if epoch < opt.warmup_epochs:
            warmup_to = eta_min + (opt.learning_rate - eta_min) * (
                    1 + math.cos(math.pi * opt.warmup_epochs / opt.epochs)) / 2
            p = n_iter / (opt.warmup_epochs * self.iter_per_epoch)
            lr = warmup_from + p * (warmup_to - warmup_from)
        else:
            lr = opt.learning_rate
            lr = eta_min + (lr - eta_min) * (
                    1 + math.cos(math.pi * epoch / opt.epochs)) / 2

        return lr
