# -*- coding: UTF-8 -*-


import logging
import importlib
import os
import pathlib
import re


class Register:
    """
    The `Register` class provides a way to register and manage a collection of callables (functions or classes) under a specific registry name. It allows you to associate a key (string) with a callable, and provides methods to register, retrieve, and check the existence of these callables in the registry.
    
    The `__init__` method initializes the registry with a name and an optional base class. The `__setitem__` method is used to add a new callable to the registry, ensuring that the value is a callable and that the key is unique. The `register` method is a decorator that can be used to register a callable, either by specifying a key or using the callable's name as the key.
    
    The `__getitem__` and `__contains__` methods provide access to the registered callables, and the `keys` method returns the set of registered keys.
    """

    def __init__(self, registry_name, baseclass=None):
        """
        Initializes a registry for storing and retrieving models.
        
        Args:
            registry_name (str): The name of the registry.
            baseclass (type, optional): The base class that all registered models must inherit from.
        """

        self._dict = {}
        self._name = registry_name
        self._baseclass = baseclass

    def __setitem__(self, key, value):
        """
        Sets an item in the registry. The value must be a callable.
        
        If the key is not provided, it will be set to the name of the callable.
        If the key already exists in the registry, a warning will be logged.
        
        Args:
            key (str): The key to set in the registry.
            value (callable): The callable to associate with the key.
        """

        if not callable(value):
            raise Exception(f"Value of a Registry must be a callable!\nValue: {value}")
        if key is None:
            key = value.__name__
        if key in self._dict:
            logging.warning("Key %s already in registry %s." % (key, self._name))
        self._dict[key] = value

    def register(self, target):
        """
        Decorator to register a function or class.
        
        This decorator is used to register a function or class with a dictionary. If the target is already registered, a warning is logged. If the target is a callable, it is added to the dictionary directly. If the target is a class, the decorator will add the class to the dictionary, and optionally copy any attributes from a base class specified by `_baseclass`.
        """


        def add(key, value):
            self[key] = value
            return value

        if target in self._dict:
            logging.warning(f'Cannot register duplicate ({target})')
        if callable(target):
            # @reg.register
            return add(None, target)

        # @reg.register('alias')
        def class_rebuilder(cls):
            if self._baseclass is not None:
                for p in dir(self._baseclass):
                    if p in dir(cls):
                        continue
                    setattr(cls, p, getattr(self._baseclass, p))
            return add(target, cls)

        return class_rebuilder

    def __getitem__(self, key):
        """
        Returns the value associated with the given key in the internal dictionary.
        """

        return self._dict[key]

    def __contains__(self, key):
        """
        Returns True if the given key is present in the internal dictionary, False otherwise.
        """

        return key in self._dict

    def keys(self):
        """
        Returns the keys of the internal dictionary.
        """

        return self._dict.keys()

    def __repr__(self):
        """
        Returns a string representation of the object's internal dictionary.
        """

        return str(self._dict)


def import_models(root, prefix):
    """
    Imports all Python modules from the specified root directory, using the provided prefix for the module names.
    
    Args:
        root (str): The path to the directory containing the Python modules to import.
        prefix (str): The prefix to use for the module names.
    
    Returns:
        None
    """

    root = os.path.abspath(root)
    for p in pathlib.Path(root).rglob('*.py'):
        p = str(p)
        flag = False
        for x in p.split(os.sep):
            if x.startswith('.'):
                flag = True
        if flag:
            continue
        lib = re.sub(root, prefix, p)
        lib = re.sub(os.sep, '.', lib)[:-3]
        importlib.import_module(lib)
