from __future__ import print_function

import os
import json
from pathlib import Path
import traceback
from typing import List, Optional, Union
import grpc
import hashlib
from dataclasses import dataclass, field
from loguru import logger


from copy import copy
from func_timeout import func_set_timeout
from .utils.pisa_server_control import start_server, close_server

from . import server_pb2
from . import server_pb2_grpc
# import server_pb2
# import server_pb2_grpc

MAX_MESSAGE_LENGTH = 10485760 * 10
THEOREM_SEPARATOR = "<THM_SEP>"

SPECIAL_SYMBOL = {
    "≥": "\<ge>",
    "≤": "\<le>",
    "≠": "\<noteq>",
    "∧": "\<and>",
    "∨": "\<or>",
    "¬": "\<not>",
    "⇒": "\<Rightarrow>",
    "⇔": "\<longleftrightarrow>",
    "→": "\<longrightarrow>",
    "∀": "\<forall>",
    "∃": "\<exists>",
    "∈": "\<in>",
    "⊆": "\<subseteq>",
    "∪": "\<union>",
    "∩": "\<inter>",
    "⊂": "\<subset>",
    "∅": "\<emptyset>",
    "∑": "\<sum>",
    "∫": "\<integral>",
    "≡": "\<equiv>",
    "⊢": "\<turnstile>",
    "⊨": "\<models>",
    "⊥": "\<bottom>",
    "⊤": "\<top>",
    "≜": "\<triangleq>",
    "≈": "\<approx>",
    "≅": "\<cong>",
    "⋀": "\<And>",
    "⋁": "\<Or>",
    "⋂": "\<Inter>",
    "⋃": "\<Union>",
    "⊗": "\<otimes>",
    "∗": "\<star>",
    "λ": "\<lambda>",
    "∞": "\<infinity>",
    "ℕ": "\<nat>",
    "ℤ": "\<int>",
    "ℚ": "\<rat>",
    "ℝ": "\<real>",
    "ℂ": "\<complex>",
    "↔": "\<leftrightarrow>",
    "‹": "\<open>",
    "›": "\<close>",
    "α": "\<alpha>",
    "β": "\<beta>",
    "γ": "\<gamma>",
}

def remove_newline(msg):
    return msg.replace("\n", " ")

class IsabelleFatalError(Exception):
    pass

class IsabelleRuntimeError(Exception):
    pass

def create_stub(port=9000):
    channel = grpc.insecure_channel('localhost:{}'.format(port),
                                    options=[('grpc.max_send_message_length', MAX_MESSAGE_LENGTH),
                                             ('grpc.max_receive_message_length', MAX_MESSAGE_LENGTH)])
    return server_pb2_grpc.ServerStub(channel)


def initialise_env(port, isa_path, theory_file_path, working_directory, debug=False, logger=None):
    return PisaEnv(port=port, isa_path=isa_path, starter_string=theory_file_path, working_directory=working_directory, debug=debug, logger=logger)


class PisaEnv:
    def __init__(self, 
        port=9000, 
        isa_path="/Applications/Isabelle2020.app/Isabelle",
        starter_string="theory Test imports Complex_Main begin",
        working_directory="/Users/qj213/Projects/afp-2021-02-11/thys/Functional-Automata",
        debug=False,
        logger=None,
    ):
        self.port = port
        self.isa_path = isa_path
        self.starter_string = starter_string
        self.working_directory = working_directory
        self.debug = debug
        self.logger = logger

        self.stub = None
        self.obs_string = None
        self.successful_starting = False
        self.reset()

    @func_set_timeout(1800, allowOverride=True)
    def reset(self):
        # print = self.logger.info if self.logger else print
        self.stub = create_stub(port=self.port)
        # try:
        self.stub.InitialiseIsabelle(server_pb2.IsaPath(path=self.isa_path)).message
        self.stub.IsabelleWorkingDirectory(server_pb2.IsaPath(path=self.working_directory)).message
        msg = self.stub.IsabelleContext(server_pb2.IsaContext(context=self.starter_string)).message
        # logger.info(msg)
        if "----------Path to Isabelle theory file----------" not in msg:
            self.successful_starting = False
            logger.info(f"Fail to initilize problem with error {remove_newline(msg)}")
            if "Structure Sledgehammer not declared in given context" in msg:
                raise IsabelleRuntimeError(msg)
            elif 'Theory loader: undefined entry for theory "..' in msg:
                raise IsabelleRuntimeError(msg)
            else:
                raise IsabelleFatalError(msg)
        self.successful_starting = True
        logger.info(f"Successfully initilize problem {self.starter_string}")
        # except Exception as e:
        #     print("Failure at initialising Isabelle process.\n"
        #           "Make sure the path your provide is where the Isabelle executable is.")
        #     print(e)
        #     raise IsabelleFailedError(str(e))
        return f"Starting is successful: {self.successful_starting}"

    def reset_problem(self):
        ret = self.post("<reset problem>")
        return ret == "The problem is reset."

    def exit(self):
        ret = self.post("exit")
        return ret

    # @func_set_timeout(1800, allowOverride=True)
    def step(self, old_name, step, new_name, delete_old_state=True) -> str:
        '''
        :param old_name: the name of the old state
        :param step: the step to take
        :param new_name: the name of the new state
        :param delete_old_state: if true, delete the old state
        :return: the string of the new state
        I recommend not deleting the default state "default" as it is the starting state.
        '''
        obs_string = "Step error"
        try:
            obs_string = self.stub.IsabelleCommand(
                server_pb2.IsaCommand(command=f"<apply to top level state> {old_name} <apply to top level state> {step} <apply to top level state> {new_name}")).state
            if delete_old_state:
                self.stub.IsabelleCommand(server_pb2.IsaCommand(command=f"<delete> {old_name}"))
                print(f"Deleted old state with name: {old_name}")
        except Exception as e:
            print("***Something went wrong***")
            print(e)
        return obs_string

    def is_finished(self, name_of_tls):
        returned_string = self.post(f"<is finished> {name_of_tls}").strip()
        if returned_string.startswith("t"):
            return True
        else:
            return False

    def reward(self, done):
        if done:
            return 1
        else:
            return 0
        
    def get_parsed_code(self, theory) -> List[str]:
        for symbol, value in SPECIAL_SYMBOL.items():
            if symbol in theory:
                theory = theory.replace(symbol, value)
        steps = self.post(f"<parse text> ${theory}")
        steps = steps.split('<SEP>')
        steps = [s for s in steps if s.strip() != '']
        # remove weird '$' step and whitespace steps
        steps = [s for s in steps if s != '$' and s.strip() != '']
        return steps

    def get_premises(self, name_of_tls, theorem_name, theorem_proof_string):
        message = f"<get dependent theorems>{name_of_tls}<get dependent theorems>{theorem_name}<get dependent theorems>{THEOREM_SEPARATOR}"
        # print(f"Get dependent theroem string: {message}")
        returned_string = self.post(message)
        # print(f"Returned string: {returned_string}")
        premises = returned_string.split(THEOREM_SEPARATOR)
        premises = [premise.strip() for premise in premises]
        # print(premises)
        # print("premises raw", premises)
        # print(f"Returned premises: {'||'.join(premises)}")

        # Function to break down the proof string
        def further_break(chunks, separator=None):
            new_chunks = []
            for chunk in chunks:
                new_chunks.extend(chunk.split(separator))
            return new_chunks

        # Break down the proof string into chunks which might be premises
        possible_premise_chunks = further_break([theorem_proof_string])
        # print("First filter", possible_premise_chunks)
        legit_separators = [",", "(", ")", "[", "]", "{", "}", ":", '"', "<", ">", "\\"]
        for separtor in legit_separators:
            possible_premise_chunks = further_break(possible_premise_chunks, separtor)
        # print("Second filter", possible_premise_chunks)
        possible_premise_chunks = set(chunk.strip() for chunk in possible_premise_chunks)
        # print("Third filter", possible_premise_chunks)
        
        
        # Only include theorems that are in the proof string
        explicit_premises = []
        for premise in premises:
            premise_divisions = premise.split(".")
            for i in range(len(premise_divisions)):
                possible_way_to_refer_to_premise = ".".join(premise_divisions[i:])
                # print("possible_way", possible_way_to_refer_to_premise)
                if possible_way_to_refer_to_premise in possible_premise_chunks:
                    explicit_premises.append(premise)
                    break

        explicit_premises = [premise for premise in explicit_premises if premise.strip()]
        # print(theorem_name, theorem_proof_string, explicit_premises)
        # print("*"*100)
        return explicit_premises

    def get_fact_defintion(self, name_of_tls, fact_name):
        message = f"<get fact definition>{name_of_tls}<get fact definition>{fact_name}"
        # print(f"Get fact definition: {message}")
        returned_string = self.post(message)
        # print(f"Returned definition: {returned_string}")
        return returned_string

    def get_premises_and_their_definitions(self, full_name, only_name, proof_body, debug=False):
        if debug: print("-1")
        self.initialise()
        if debug: print("0")
        # Getting unique name and clone the top level there
        tls_unique_name = str(hashlib.sha256(proof_body.encode("utf-8")).hexdigest())
        tls_unique_name = ''.join(filter(str.isalpha, tls_unique_name))
        # decorated_name = only_name.format(tls_unique_name)
        self.clone_to_new_name(tls_unique_name)
        if debug: print(0.5, "post clone")
        # Substitute proof
        # if not only_name.strip():
        #     sub_proof = f"<allow more time> theorem {proof_body}"
        # else:
        #     sub_proof = f"<allow more time> theorem {full_name}: {proof_body}"
        # if debug: print("1", sub_proof)
        # self.step_to_top_level_state(sub_proof, tls_unique_name, tls_unique_name)
        if debug: print("2, stepping")
        premises = self.get_premises(tls_unique_name, only_name, proof_body)
        if debug: print("3", premises)
        premises_and_their_definitions = [(premise, self.get_fact_defintion(tls_unique_name, premise)) for premise in premises]
        if debug: print("4", premises_and_their_definitions)
        self.post(f"<delete> {tls_unique_name}")
        return premises_and_their_definitions

    # def get_premises_and_their_definitions(self, full_theorem_def, theorem_name, theorem_proof_string):
    #     # print("Get to end: " + self.proceed_until_end_of_theorem_proof(full_theorem_def))
    #     self.initialise()
    #     premises = self.get_premises("default", theorem_name, theorem_proof_string)
    #     # print(premises)
    #     premises_and_their_definitions = [(premise, self.get_fact_defintion("default", premise)) for premise in premises]
    #     return premises_and_their_definitions

    def proceed_until_end_of_theorem_proof(self, theorem_name):
        message = f"<accumulative_step_to_theorem_end> {theorem_name}"
        return self.post(message)

    def accumulative_step_before_theorem_starts(self, theorem_name):
        message = f"<accumulative_step_before_theorem_starts> {theorem_name}"
        return self.post(message)
    
    def accumulative_step_through_a_theorem(self):
        message = f"<accumulative_step_through_a_theorem>"
        return self.post(message)

    # @func_set_timeout(1800, allowOverride=True)
    def step_to_top_level_state(self, action, tls_name, new_name, return_proof_level=False, tactic_time=10000):
        # last_obs_string = self.stub.IsabelleCommand(server_pb2.IsaCommand(command=f"<get state> {tls_name}")).state
        obs_string = "Step error"
        proof_level = -1
        try:
            obs_string = self.post(f"<apply to top level state> {tls_name} <apply to top level state> {action} <apply to top level state> {new_name} <apply to top level state> {tactic_time}")
            # print(obs_string)
        except grpc._channel._InactiveRpcError as e:
            self.logger.warning(f"Isabelle server is broke with error: {e}")
            raise IsabelleFatalError(str(e))
        except RuntimeError as e:
            if "can't start new thread" in str(e):
                self.logger.warning(f"Isabelle server is broke with error: {e}")
                raise IsabelleFatalError(str(e))
            else:
                raise e
        except Exception as e:
            logger.info("***Something went wrong***")
            logger.info(e)
            logger.info(traceback.format_exc())

        if "error" in obs_string:
            done = False
        else:
            done = self.is_finished(new_name)
        if "<prooflevel>" in obs_string:
            obs_string, proof_level = obs_string.split("<prooflevel>")
            obs_string, proof_level = obs_string.strip(), int(proof_level.strip())
        # done = True if ("subgoal" in last_obs_string and "subgoal" not in obs_string) else False
        if return_proof_level:
            return obs_string, self.reward(done), done, {}, proof_level
        return obs_string, self.reward(done), done, {}

    def proceed_after(self, line_string):
        return self.post(f"<proceed after> {line_string}", forceTimeout=10000)

    def initialise(self):
        return self.post("<initialise>", forceTimeout=10)

    def clone_to_new_name(self, new_name):
        return self.post(f"<clone> default <clone> {new_name}", forceTimeout=10)

    @func_set_timeout(1800, allowOverride=True)
    def post(self, action):
        if self.debug: print(action)
        returned = self.stub.IsabelleCommand(server_pb2.IsaCommand(command=action)).state
        if self.debug: print(returned)
        return returned

    def proceed_to_line(self, line_string, before_after, counter=1):
        assert before_after in ["before", "after"]
        try:
            command = f"<proceed to line> {line_string} <proceed to line> {before_after} <proceed to line> {counter} "
            # print(command)
            message = self.stub.IsabelleCommand(server_pb2.IsaCommand(command=command)).state
            # print(message)
            return message
        except grpc._channel._InactiveRpcError as e:
            self.logger.warning(f"Isabelle server is broke with error: {e}")
            self.logger.warning(traceback.format_exc())
            raise IsabelleFatalError(str(e))
        except Exception as e:
            logger.info("Failure to proceed before line")
            logger.info(e)
            logger.info(traceback.format_exc())

    def proceed_until_end(self):
        return self.post("<proceed until end>")

def parsed_json_to_env_and_dict(path_to_json, afp_path, port=9000, isa_path="/Applications/Isabelle2020.app/Isabelle"):
    save_dict = json.load(open(path_to_json))
    project = save_dict["project"]
    wd = os.path.join(afp_path, "thys", project)
    segments = save_dict["segments"]
    # Find starter string
    starter_string = None
    for line in segments:
        if line.strip().startswith("theory"):
            starter_string = " ".join(line.strip().split("\n"))
            break
    assert starter_string
    # print(port, isa_path, starter_string, wd, segments)
    return PisaEnv(port=port, isa_path=isa_path,
                     starter_string=starter_string,
                     working_directory=wd), save_dict


@dataclass(frozen=True)
class TacticState:
    pp: str = field(compare=False)
    id: str = field(compare=False)
    from_tactic: str = field(compare=False)
    proof_level: str = field(compare=False)
    elaborated_pp: str = field(compare=False)
    unique_identifier: str = field(default=None, compare=True)
    

    def __post_init__(self) -> None:
        # single_from_tactic = self.from_tactic.split("\n")[-1]
        # unique_identifier = f"{self.from_tactic}_{self.pp}"
        unique_identifier = f"{self.elaborated_pp}_{self.pp}_{self.from_tactic}"
        object.__setattr__(self, "unique_identifier", unique_identifier)


@dataclass(frozen=True)
class ProofFinished:
    from_tactic: str
    tactic_state_id: int
    proof_level: int
    message: Optional[str] = field(default=None, compare=False)
    


@dataclass(frozen=True)
class ProofGivenUp:
    pass


@dataclass(frozen=True)
class IsabelleError:
    error: str


@dataclass(frozen=True)
class TimeoutError:
    error: str


TacticResult = Union[
    TacticState,
    ProofFinished,
    IsabelleError,
    TimeoutError,
    ProofGivenUp,
]


State = TacticState


class DojoCrashError(Exception):
    @property
    def is_out_of_memory(self) -> bool:
        return str(self) == "OOM"


class DojoHardTimeoutError(Exception):
    pass


class DojoInitError(Exception):
    pass

@dataclass(frozen=True)
class Theorem:
    """Generic Theorem class.

    This class is intended as a parent for specific theorem classes for
    different theorem proving environments.
    """

    file_path: Path
    """Source file the theorem comes from."""

    full_name: str
    """Fully qualified name of the theorem."""

    count : int
    """When duplicated theorem exists, which theorem it's
    """

    working_directory: Path = field(default=None)
    """Isabelle file working directory.
    """

    def __post_init__(self) -> None:
        if isinstance(self.file_path, str):
            object.__setattr__(self, "file_path", Path(self.file_path))
        assert (
            self.file_path.suffix == ".thy"
        ), f"File extension must be .thy: {self.file_path}"
        if self.working_directory == None:
            object.__setattr__(self, "working_directory", self.file_path.parent)

    def __eq__(self, other: "Theorem") -> bool:
        if self.file_path == other.file_path and \
            self.full_name == other.full_name and \
                self.count == other.count:
            return True
        return False


class IsaDojo:
    def __init__(self,
                 port, 
                 jar_path, 
                 isa_path, 
                 working_directory, 
                 theory_file_path, 
                 theorem_name, 
                 server_output_path="temp/server_output.txt", 
                 debug=False, 
                 logger=None
        ):
        self.server_process_id = start_server(jar_path, port, 
                outputfile=server_output_path, errorfile=server_output_path)
        try:
            self.pisa_env = PisaEnv(port=port, isa_path=isa_path, starter_string=theory_file_path, working_directory=working_directory, debug=debug, logger=logger)
        except Exception:
            self.pisa_env = None
        self.theorem_name = theorem_name
        
    
    def __enter__(self):
        if not self.pisa_env:
            return self, None
        self.init_state = self.pisa_env.proceed_to_line(self.theorem_name, "after")
        self.pisa_env.initialise()
        self.pisa_env.clone_to_new_name("temp_0")
        if "<elaborated>" in self.init_state:
            self.init_state, self.init_elaborated_state = self.init_state.split("<elaborated>")
            self.init_state, self.init_elaborated_state = self.init_state.strip(), self.init_elaborated_state.strip()
        else:
            self.init_elaborated_state = ""
        self.init_state = TacticState(pp=self.init_state, id="temp_0", from_tactic=self.theorem_name, proof_level=0, elaborated_pp=self.init_elaborated_state)
        self.valid_states = set([self.init_state.id])
        return self, self.init_state
    
    def run_tac(self, state: TacticState, tactic: str, root_state: TacticState = None) -> TacticResult:
        if not isinstance(state, TacticState):
            raise RuntimeError(
                f"Attempting to run a tactic on an invalid state {state}."
            )
        assert isinstance(tactic, str), f"Invalid tactic {tactic}"

        if "Oops" in tactic or "oops" in tactic or "\<proof>" in tactic:
            return IsabelleError("Tactic contains `Oops` or `\<proof>`, which is not allow!")
        if len(tactic.strip()) == 0:
            return IsabelleError("Tactic is empty")

        if root_state is None:
            root_level = 1
        else:
            root_level = root_state.proof_level

        new_id = f"temp_{len(self.valid_states)}"
        state_string, reward, done, _, proof_level = \
            self.pisa_env.step_to_top_level_state(tactic, state.id, new_id, return_proof_level=True)
        if "error" in state_string:
            return IsabelleError(state_string.strip())
        elif done == 1:
            assert proof_level == 0, f"Done is 1 but proof level is {proof_level}"
            return ProofFinished(tactic, state.id, proof_level, state_string)
        elif proof_level >= 0 and proof_level < root_level:
            return ProofFinished(tactic, state.id, proof_level, state_string)
        else:
            assert "proof" in state_string, f"Invalid state string: {state_string}"
            if "<elaborated>" in state_string:
                state_string, elaborated_state = state_string.split("<elaborated>")
                state_string, elaborated_state = state_string.strip(), elaborated_state.strip()
            else:
                elaborated_state = ""
                logger.warning(f"Noraml state don't have elaborated: {state_string}")
            tactic_state = TacticState(pp=state_string, id=new_id, from_tactic=tactic, proof_level=proof_level, elaborated_pp=elaborated_state)
            self.valid_states.add(tactic_state.id)
            return tactic_state

    def run_tac_break(self, state: TacticState, tactic: str, root_state: TacticState = None) -> TacticResult:
        if not isinstance(state, TacticState):
            raise RuntimeError(
                f"Attempting to run a tactic on an invalid state {state}."
            )
        assert isinstance(tactic, str), f"Invalid tactic {tactic}"

        if "Oops" in tactic or "oops" in tactic or "\<proof>" in tactic:
            return IsabelleError("Tactic contains `Oops` or `\<proof>`, which is not allow!"), tactic
        if len(tactic.strip()) == 0:
            return IsabelleError("Tactic is empty"), tactic

        if root_state is None:
            root_level = 1
        else:
            root_level = root_state.proof_level
        
        tactics = self.pisa_env.get_parsed_code(tactic)
        results = []
        cur_id = state.id

        for idx, tac in enumerate(tactics):
            new_id = f"temp_{len(self.valid_states)}"
            state_string, reward, done, _, proof_level = \
                self.pisa_env.step_to_top_level_state(tac, cur_id, new_id, return_proof_level=True)
            actual_tac = "\n".join(tactics[:idx+1])
            if "error" in state_string:
                results.append(IsabelleError(state_string.strip()))
                break
            elif done == 1:
                assert proof_level == 0, f"Done is 1 but proof level is {proof_level}"
                results.append(ProofFinished(actual_tac, cur_id, proof_level, state_string))
                break
            elif proof_level >= 0 and proof_level < root_level:
                results.append(ProofFinished(actual_tac, cur_id, proof_level, state_string))
                break
            else:
                assert "proof" in state_string, f"Invalid state string: {state_string}"
                if "<elaborated>" in state_string:
                    state_string, elaborated_state = state_string.split("<elaborated>")
                    state_string, elaborated_state = state_string.strip(), elaborated_state.strip()
                else:
                    elaborated_state = ""
                    logger.warning(f"Noraml state don't have elaborated: {state_string}")
                tactic_state = TacticState(pp=state_string, id=new_id, from_tactic=actual_tac, proof_level=proof_level, elaborated_pp=elaborated_state)
                self.valid_states.add(tactic_state.id)
                results.append(tactic_state)
                cur_id = new_id
        
        for idx, res in enumerate(results):
            if idx == len(results)-1 or isinstance(res, ProofFinished) or isinstance(res, IsabelleError):
                actual_tac = "\n".join(tactics[:idx+1])
                return res, actual_tac
            elif isinstance(res, TacticState):
                if idx < len(results)-1 and (isinstance(results[idx+1], TacticState) or isinstance(results[idx+1], ProofFinished)):
                    continue
                if idx < len(results)-1 and isinstance(results[idx+1], IsabelleError):
                    actual_tac = "\n".join(tactics[:idx+1])
                    return res, actual_tac
            else:
                assert False, f"Shold not be here {results}"
            

    
    def check_proof(self, proof: List[str]):
        current_state = self.init_state
        for tactic in proof:
            current_state = self.run_tac(current_state, tactic)
            # if actual_tac != tactic:
            #     return False
            if not (isinstance(current_state, TacticState) or isinstance(current_state, ProofFinished)):
                return False
        if isinstance(current_state, ProofFinished):
            return True
        return False

    def __exit__(self, exc_type: None, exc_val: None, exc_tb: None) -> None:
        """Exit Dojo.

        Args:
            exc_type (None): _description_
            exc_val (None): _description_
            exc_tb (None): _description_
        """
        # Cancel the hard timeout.
        if self.pisa_env is not None:
            self.pisa_env.exit()
            del self.pisa_env
        if self.server_process_id is not None:
            close_server(self.server_process_id)


if __name__ == '__main__':
    from loguru import logger
    env = initialise_env(
        8000, 
        "/hpc2hdd/home/zyang398/Isabelle2022", 
        "/hpc2hdd/home/zyang398/miniF2F/isabelle/test/induction_nfactltnexpnm1ngt3.thy", 
        "/hpc2hdd/home/zyang398/miniF2F/isabelle/test/"
    )
    with open("miniF2F_benchmark_v0.1/test.json") as f:
        data = json.load(f)
        for line in data:
            if "induction_nfactltnexpnm1ngt3" in line["full_name"]:
                data = line
                break
    s0 = env.proceed_to_line(data["full_name"], 'after', counter=1)
    env.initialise()
    s1 = env.step_to_top_level_state('proof (cases "n = 0")', "default", "temp_0")
    s2 = env.step_to_top_level_state('case True ', "temp_0", "temp_1")
    s3 = env.step_to_top_level_state('show?thesis ', "temp_1", "temp_2")
    s4 = env.step_to_top_level_state('proof (cases "n = 0")', "temp_2", "temp_3")
    # s1 = env.step_to_top_level_state('show?thesis ', "temp_3", "temp_4")
    # s1 = env.step_to_top_level_state('show?thesis ', "temp_4", "temp_5")
    # s1 = env.step_to_top_level_state

    # print(env.step_to_top_level_state('sledgehammer', 'test', 'test1'))
    # print(env.step_to_top_level_state('delhammer primes_infinite', 'test', 'test2'))
    # print(env.step_to_top_level_state('delhammer primes_infinite,bigger_prime', 'test', 'test3'))
