

from typing import Optional, Tuple, Union


import torch
from torch import nn
from torch.nn import BCEWithLogitsLoss, CrossEntropyLoss, MSELoss



from transformers.models.gpt2.modeling_gpt2 import (
    GPT2Model,
    GPT2Block,
    GPT2Attention,
    GPT2MLP,
    GPT2LMHeadModel,
)

from transformers.modeling_outputs import (
    BaseModelOutputWithPastAndCrossAttentions, 
    CausalLMOutputWithCrossAttentions,
)

from transformers.modeling_attn_mask_utils import (
    _prepare_4d_attention_mask_for_sdpa, 
    _prepare_4d_causal_attention_mask_for_sdpa
)

from pdb import set_trace as pds
minus_constant = 100
def get_b_tilde_from_b(b):
    # flops = 0
    b_tilde = torch.ones_like(b, dtype=torch.float32)  # exp(0)
    k, n = b.shape
    sum_b_r = torch.zeros(n, dtype=torch.float32, device=b.device)
    sum_b_r_minus_1 = torch.zeros(n, dtype=torch.float32, device=b.device)
    
    for i in range(k):
        if i == 0:
            sum_b_r += b[i]
            # flops += n
            b_tilde[i, :] = torch.exp(sum_b_r - minus_constant)
            # flops += n
        else:
            sum_b_r += b[i]
            # flops += n
            sum_b_r_minus_1 += b[i - 1]
            # flops += n
            b_tilde[i, :] = torch.exp(sum_b_r - minus_constant) - torch.exp(sum_b_r_minus_1 - minus_constant)
            # flops += 3 * n
    
    # return b_tilde, flops
    return b_tilde

def recover_k_conv(Q, K, k, T, delta = 0, epsilon = 0):
    # flops = 0
    n, d = Q.shape
    v = torch.zeros(T, dtype=torch.float32, device=Q.device)  # Initial vector v
    u = torch.zeros(n, dtype=torch.float32, device=Q.device)  # Initial vector u
    s = 0  # Initial index s
    t = n - T
    
    m = torch.zeros(k, dtype=torch.long, device=Q.device)
    b = torch.zeros((k, n), dtype=torch.float32, device=Q.device)
    
    # Calculate the first b
    b[0, :] = torch.matmul(Q, K.T[:, 0])
    # flops += n * d  # Q @ K.T[:, 0]
    
    m[0] = n
    v += b[0, :T]
    # flops += T
    u += b[0, :]
    # flops += n
    
    for i in range(1, k):
        s += 1
        # s = binary_search(Q, K, k, T, delta, epsilon, v, s, t)
        m[i] = n - s
        if m[i] <= 0:
            break
        
        H_s = torch.matmul(Q, K.T[:, s])
        # flops += n * d  # Q @ (K.T)[:,s]
        
        b[i, :m[i]] = H_s[s:s + m[i]] - u[:m[i]]
        # flops += m[i]
        
        v += b[i, :T]
        # flops += T
        u += b[i, :]
        # flops += n
    
    # b_tilde, flops_2 = get_b_tilde_from_b(b)
    b_tilde = get_b_tilde_from_b(b)
    # flops += flops_2
    
    # return b_tilde, m, b, flops
    return b_tilde, m, b

def conv_with_fft(a, x, shift=0):
    n = a.shape[0]
    n = n - shift
    a_padded = torch.zeros(2 * n, dtype=torch.float32, device=a.device)
    x_padded = torch.zeros(2 * n, dtype=torch.float32, device=x.device)
    a_padded[:n] = a[:n]
    x_padded[:n] = x[-n:]
    result = torch.zeros_like(a, dtype=torch.float32)
    fft_result = torch.fft.ifft(torch.fft.fft(a_padded) * torch.fft.fft(x_padded))
    result[-n:] = fft_result[:n].real
    return result

def conv_with_fft_matrix(a, X, shift=0):
    n, d = X.shape
    result_matrix = torch.zeros_like(X, dtype=torch.float32)
    for i in range(d):
        result_matrix[:, i] = conv_with_fft(a, X[:, i], shift=shift)
    return result_matrix

def pytorch_toeplitz(V):
    '''
    It creates the Toeplitz matrix for each row in V.

    INPUT:
    V: torch tensor (batch_size x d)

    OUTPUT:
    T: (batch_size x d x d)

    EXAMPLE:
    V = torch.tensor([[1, 0.5, 0.1], [1.3, 0.9, -0.1]])
    T = pytorch_toeplitz(V)
    print(T.shape)
    print(T[0])
    print(T[1])
    '''
        
    d = V.shape[1]
    A = V.unsqueeze(1).unsqueeze(2)
    A_nofirst_flipped = torch.flip(A[:, :, :, 1:], dims=[3]) 
    A_concat = torch.concatenate([A_nofirst_flipped, A], dim=3) 
    unfold = torch.nn.Unfold(kernel_size=(1, d))
    T = unfold(A_concat)
    T = torch.flip(T, dims=[2])

    return T

class Conv_GPT2Attention(GPT2Attention):
    def __init__(self, config, is_cross_attention=False, layer_idx=None):
        super().__init__(config, is_cross_attention=is_cross_attention, layer_idx=layer_idx)
    
    def _attn(
            self, 
            query,  # [bs, num_head, seq_len, head_d]
            key,    # [bs, num_head, seq_len, head_d]
            value,  # [bs, num_head, seq_len, head_d]
            attention_mask=None,   # [bs, 1, 1, seq_len] [1, 1, 1, 10]
            head_mask=None
        ):

        attn_weights = torch.matmul(query, key.transpose(-1, -2))

        if self.scale_attn_weights:
            attn_weights = attn_weights / torch.full(
                [], value.size(-1) ** 0.5, dtype=attn_weights.dtype, device=attn_weights.device
            )

        # Layer-wise attention scaling
        if self.scale_attn_by_inverse_layer_idx:
            attn_weights = attn_weights / float(self.layer_idx + 1)

        if not self.is_cross_attention:
            # if only "normal" attention layer implements causal mask
            query_length, key_length = query.size(-2), key.size(-2)
            causal_mask = self.bias[:, :, key_length - query_length : key_length, :key_length]
            mask_value = torch.finfo(attn_weights.dtype).min
            # Need to be a tensor, otherwise we get error: `RuntimeError: expected scalar type float but found double`.
            # Need to be on the same device, otherwise `RuntimeError: ..., x and y to be on the same device`
            mask_value = torch.full([], mask_value, dtype=attn_weights.dtype, device=attn_weights.device)
            attn_weights = torch.where(causal_mask, attn_weights.to(attn_weights.dtype), mask_value)

        if attention_mask is not None:
            # Apply the attention mask
            attn_weights = attn_weights + attention_mask

        attn_weights = nn.functional.softmax(attn_weights, dim=-1)

        # Downcast (if necessary) back to V's dtype (if in mixed-precision) -- No-Op otherwise
        attn_weights = attn_weights.type(value.dtype)
        attn_weights = self.attn_dropout(attn_weights)

        # Mask heads if we want to
        if head_mask is not None:
            attn_weights = attn_weights * head_mask

        attn_output = torch.matmul(attn_weights, value)

        return attn_output, attn_weights
    
    def _naive_conv_attn(
            self, 
            query,  # [bs, num_head, seq_len, head_d]
            key,    # [bs, num_head, seq_len, head_d]
            value,  # [bs, num_head, seq_len, head_d]
            attention_mask=None,   # [bs, 1, 1, seq_len] [1, 1, 1, 10]
            head_mask=None,
            k=5,
            T=1,
            delta = 0, epsilon = 0
        ):

        attn_weights = torch.matmul(query, key.transpose(-1, -2)) # [bs, num_heads, seq_len, seq_len]

        if self.scale_attn_weights:
            attn_weights = attn_weights / torch.full(
                [], value.size(-1) ** 0.5, dtype=attn_weights.dtype, device=attn_weights.device
            )

        ###### added to implement conv basis
        seq_len = attn_weights.shape[-1]
        attn_weights_conv = pytorch_toeplitz(attn_weights[0,:,:,k-1])[:,:,:seq_len-k+1]
        attn_weights = torch.cat([attn_weights[0,:,:,:k-1], attn_weights_conv], dim = -1).unsqueeze(0)
        ######

        # Layer-wise attention scaling
        if self.scale_attn_by_inverse_layer_idx:
            attn_weights = attn_weights / float(self.layer_idx + 1)

        if not self.is_cross_attention:
            # if only "normal" attention layer implements causal mask
            query_length, key_length = query.size(-2), key.size(-2)
            causal_mask = self.bias[:, :, key_length - query_length : key_length, :key_length]
            mask_value = torch.finfo(attn_weights.dtype).min
            # Need to be a tensor, otherwise we get error: `RuntimeError: expected scalar type float but found double`.
            # Need to be on the same device, otherwise `RuntimeError: ..., x and y to be on the same device`
            mask_value = torch.full([], mask_value, dtype=attn_weights.dtype, device=attn_weights.device)
            attn_weights = torch.where(causal_mask, attn_weights.to(attn_weights.dtype), mask_value)

        
        if attention_mask is not None:
            # Apply the attention mask
            attn_weights = attn_weights + attention_mask
    

        attn_weights = nn.functional.softmax(attn_weights, dim=-1) 


        # Downcast (if necessary) back to V's dtype (if in mixed-precision) -- No-Op otherwise
        attn_weights = attn_weights.type(value.dtype)
        attn_weights = self.attn_dropout(attn_weights)

        # Mask heads if we want to
        if head_mask is not None:
            attn_weights = attn_weights * head_mask

        attn_output = torch.matmul(attn_weights, value)

        return attn_output, attn_weights

    def _conv_attn_per_head(
            self, 
            Q,  # [seq_len, head_d]
            K,    # [seq_len, head_d]
            V,  # [seq_len, head_d]
            attention_mask=None,   # [bs, 1, 1, seq_len] [1, 1, 1, 10]
            head_mask=None,
            k=5,
            T=1,
            delta = 0, epsilon = 0
        ):

        n, d = Q.shape

        if self.scale_attn_weights:
            Q = Q / torch.full(
                [], V.size(-1) ** 0.5, dtype=Q.dtype, device=Q.device
            )
        b_tilde, m, b= recover_k_conv(Q, K, k=k, T=T, delta=delta, epsilon=epsilon)

        QKV_approx = torch.zeros_like(Q, dtype=torch.float64)
        for i in range(k):
            QKV_approx += conv_with_fft_matrix(b_tilde[i, :], V, shift=n - m[i])
        
        D_approx = torch.zeros(n, dtype=torch.float64, device=Q.device)
        for i in range(k):
            D_approx += conv_with_fft(b_tilde[i, :], torch.ones(n, device=Q.device), shift=n - m[i])
        
        QKV_approx = (D_approx ** -1).unsqueeze(1) * QKV_approx

        return QKV_approx

    def _conv_attn(
            self, 
            query,  # [bs, num_head, seq_len, head_d]
            key,    # [bs, num_head, seq_len, head_d]
            value,  # [bs, num_head, seq_len, head_d]
            attention_mask=None,   # [bs, 1, 1, seq_len] [1, 1, 1, 10]
            head_mask=None,
            k=5,
            T=1,
            delta = 0, epsilon = 0
        ):
        '''
        Apply k_conv_basis_attention_score for each head.
    
        Args:
        query, key, value: torch.Tensor, shape [bs, num_head, seq_len, head_d]
        k, T, delta, epsilon: parameters for k_conv_basis_attention_score
        
        Returns:
        attn_output: torch.Tensor, shape [bs, num_head, seq_len, head_d]


        compared to _attn, omit attn_mask and attn_dropout
        '''
        bs, num_head, seq_len, head_d = query.shape
        attn_output = torch.zeros_like(query)
        for b in range(bs): # bs = 1
            for h in range(num_head): # num_head = 12
                # print(f"process head h {h}")
                Q = query[b, h]
                K = key[b, h]
                V = value[b, h]
                
                QKV_approx = self._conv_attn_per_head(Q, K, V, k=k, T=T, delta=delta, epsilon=epsilon)
                attn_output[b, h] = QKV_approx

        return attn_output, None

    def forward(
        self,
        hidden_states: Optional[Tuple[torch.FloatTensor]],
        layer_past: Optional[Tuple[torch.Tensor]] = None,
        attention_mask: Optional[torch.FloatTensor] = None,
        head_mask: Optional[torch.FloatTensor] = None,
        encoder_hidden_states: Optional[torch.Tensor] = None,
        encoder_attention_mask: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = False,
        output_attentions: Optional[bool] = False,
        k=10,
        T=1,
    ) -> Tuple[Union[torch.Tensor, Tuple[torch.Tensor]], ...]:
        '''
        copy from modeling_gpt2.py, only modify the self._conv_attn forward part
        '''
        if encoder_hidden_states is not None:
            if not hasattr(self, "q_attn"):
                raise ValueError(
                    "If class is used as cross attention, the weights `q_attn` have to be defined. "
                    "Please make sure to instantiate class with `GPT2Attention(..., is_cross_attention=True)`."
                )

            query = self.q_attn(hidden_states)
            key, value = self.c_attn(encoder_hidden_states).split(self.split_size, dim=2)
            attention_mask = encoder_attention_mask
        else:
            query, key, value = self.c_attn(hidden_states).split(self.split_size, dim=2)
        query = self._split_heads(query, self.num_heads, self.head_dim)
        key = self._split_heads(key, self.num_heads, self.head_dim)
        value = self._split_heads(value, self.num_heads, self.head_dim)

        if layer_past is not None:
            past_key, past_value = layer_past
            key = torch.cat((past_key, key), dim=-2)
            value = torch.cat((past_value, value), dim=-2)

        if use_cache is True:
            present = (key, value)
        else:
            present = None
        if self.reorder_and_upcast_attn:
            attn_output, attn_weights = self._upcast_and_reordered_attn(query, key, value, attention_mask, head_mask)
        else:
            # attn_output, attn_weights = self._attn(query, key, value, attention_mask, head_mask)
            # attn_output, attn_weights = self._conv_attn(query, key, value, attention_mask, head_mask, k = k, T = T)
            attn_output, attn_weights = self._naive_conv_attn(query, key, value, attention_mask, head_mask, k = k, T = T)
        attn_output = self._merge_heads(attn_output, self.num_heads, self.head_dim)
        attn_output = self.c_proj(attn_output)
        attn_output = self.resid_dropout(attn_output)

        outputs = (attn_output, present)
        if output_attentions:
            outputs += (attn_weights,)
        
        return outputs  # a, present, (attentions)


class Conv_GPT2Block(GPT2Block):
    def __init__(self, config, layer_idx=None):
        super().__init__(config, layer_idx=layer_idx)
        hidden_size = config.hidden_size
        inner_dim = config.n_inner if config.n_inner is not None else 4 * hidden_size
        # attention_class = GPT2_ATTENTION_CLASSES[config._attn_implementation]
        attention_class = Conv_GPT2Attention

        self.ln_1 = nn.LayerNorm(hidden_size, eps=config.layer_norm_epsilon)
        self.attn = attention_class(config=config, layer_idx=layer_idx)
        self.ln_2 = nn.LayerNorm(hidden_size, eps=config.layer_norm_epsilon)

        if config.add_cross_attention:
            self.crossattention = attention_class(config=config, is_cross_attention=True, layer_idx=layer_idx)
            self.ln_cross_attn = nn.LayerNorm(hidden_size, eps=config.layer_norm_epsilon)

        self.mlp = GPT2MLP(inner_dim, config)
    
    def forward(
        self,
        hidden_states: Optional[Tuple[torch.FloatTensor]],
        layer_past: Optional[Tuple[torch.Tensor]] = None,
        attention_mask: Optional[torch.FloatTensor] = None,
        head_mask: Optional[torch.FloatTensor] = None,
        encoder_hidden_states: Optional[torch.Tensor] = None,
        encoder_attention_mask: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = False,
        output_attentions: Optional[bool] = False,
        k=10,
        T=1,
    ) -> Union[Tuple[torch.Tensor], Optional[Tuple[torch.Tensor, Tuple[torch.FloatTensor, ...]]]]:
        '''
        copy from modeling_gpt2.py, only modify the self.attn forward part
        '''
        residual = hidden_states
        hidden_states = self.ln_1(hidden_states)
        attn_outputs = self.attn(
            hidden_states,
            layer_past=layer_past,
            attention_mask=attention_mask,
            head_mask=head_mask,
            use_cache=use_cache,
            output_attentions=output_attentions,
            k=k,
            T=T,
        )
        attn_output = attn_outputs[0]  # output_attn: a, present, (attentions)
        outputs = attn_outputs[1:]
        # residual connection
        hidden_states = attn_output + residual

        if encoder_hidden_states is not None:
            # add one self-attention block for cross-attention
            if not hasattr(self, "crossattention"):
                raise ValueError(
                    f"If `encoder_hidden_states` are passed, {self} has to be instantiated with "
                    "cross-attention layers by setting `config.add_cross_attention=True`"
                )
            residual = hidden_states
            hidden_states = self.ln_cross_attn(hidden_states)
            cross_attn_outputs = self.crossattention(
                hidden_states,
                attention_mask=attention_mask,
                head_mask=head_mask,
                encoder_hidden_states=encoder_hidden_states,
                encoder_attention_mask=encoder_attention_mask,
                output_attentions=output_attentions,
            )
            attn_output = cross_attn_outputs[0]
            # residual connection
            hidden_states = residual + attn_output
            outputs = outputs + cross_attn_outputs[2:]  # add cross attentions if we output attention weights

        residual = hidden_states
        hidden_states = self.ln_2(hidden_states)
        feed_forward_hidden_states = self.mlp(hidden_states)
        # residual connection
        hidden_states = residual + feed_forward_hidden_states

        if use_cache:
            outputs = (hidden_states,) + outputs
        else:
            outputs = (hidden_states,) + outputs[1:]

        return outputs  # hidden_states, present, (attentions, cross_attentions)


class Conv_GPT2Model(GPT2Model):
    def __init__(self, config):
        super().__init__(config)

        self.embed_dim = config.hidden_size

        self.wte = nn.Embedding(config.vocab_size, self.embed_dim)
        self.wpe = nn.Embedding(config.max_position_embeddings, self.embed_dim)

        self.drop = nn.Dropout(config.embd_pdrop)
        # self.h = nn.ModuleList([GPT2Block(config, layer_idx=i) for i in range(config.num_hidden_layers)])
        self.h = nn.ModuleList([Conv_GPT2Block(config, layer_idx=i) for i in range(config.num_hidden_layers)])
        self.ln_f = nn.LayerNorm(self.embed_dim, eps=config.layer_norm_epsilon)

        # Model parallel
        self.model_parallel = False
        self.device_map = None
        self.gradient_checkpointing = False
        self._attn_implementation = config._attn_implementation

        # Initialize weights and apply final processing
        self.post_init()
    
    def forward(
        self,
        input_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Tuple[Tuple[torch.Tensor]]] = None,
        attention_mask: Optional[torch.FloatTensor] = None,
        token_type_ids: Optional[torch.LongTensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        head_mask: Optional[torch.FloatTensor] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        encoder_hidden_states: Optional[torch.Tensor] = None,
        encoder_attention_mask: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        k=10,
        T=1,
    ) -> Union[Tuple, BaseModelOutputWithPastAndCrossAttentions]:
        '''
        copy from modeling_gpt2.py, only modify the self.h forward part
        '''
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        if input_ids is not None and inputs_embeds is not None:
            raise ValueError("You cannot specify both input_ids and inputs_embeds at the same time")
        elif input_ids is not None:
            self.warn_if_padding_and_no_attention_mask(input_ids, attention_mask)
            input_shape = input_ids.size()
            input_ids = input_ids.view(-1, input_shape[-1])
            batch_size = input_ids.shape[0]
        elif inputs_embeds is not None:
            input_shape = inputs_embeds.size()[:-1]
            batch_size = inputs_embeds.shape[0]
        else:
            raise ValueError("You have to specify either input_ids or inputs_embeds")

        device = input_ids.device if input_ids is not None else inputs_embeds.device

        if token_type_ids is not None:
            token_type_ids = token_type_ids.view(-1, input_shape[-1])

        if past_key_values is None:
            past_length = 0
            past_key_values = tuple([None] * len(self.h))
        else:
            past_length = past_key_values[0][0].size(-2)
        if position_ids is None:
            position_ids = torch.arange(past_length, input_shape[-1] + past_length, dtype=torch.long, device=device)
            position_ids = position_ids.unsqueeze(0)

        if inputs_embeds is None:
            inputs_embeds = self.wte(input_ids)
        position_embeds = self.wpe(position_ids)
        hidden_states = inputs_embeds + position_embeds

        # Attention mask.
        _use_sdpa = self._attn_implementation == "sdpa" and output_attentions is False and head_mask is None
        attention_mask = attention_mask.view(batch_size, -1) if attention_mask is not None else None
        if self._attn_implementation == "flash_attention_2":
            attention_mask = attention_mask if (attention_mask is not None and 0 in attention_mask) else None
        elif _use_sdpa:
            attention_mask = _prepare_4d_causal_attention_mask_for_sdpa(
                attention_mask=attention_mask,
                input_shape=(batch_size, input_shape[-1]),
                inputs_embeds=inputs_embeds,
                past_key_values_length=past_length,
            )
        else:
            if attention_mask is not None:
                # We create a 3D attention mask from a 2D tensor mask.
                # Sizes are [batch_size, 1, 1, to_seq_length]
                # So we can broadcast to [batch_size, num_heads, from_seq_length, to_seq_length]
                # this attention mask is more simple than the triangular masking of causal attention
                # used in OpenAI GPT, we just need to prepare the broadcast dimension here.
                attention_mask = attention_mask[:, None, None, :]

                # Since attention_mask is 1.0 for positions we want to attend and 0.0 for
                # masked positions, this operation will create a tensor which is 0.0 for
                # positions we want to attend and the dtype's smallest value for masked positions.
                # Since we are adding it to the raw scores before the softmax, this is
                # effectively the same as removing these entirely.
                attention_mask = attention_mask.to(dtype=self.dtype)  # fp16 compatibility
                attention_mask = (1.0 - attention_mask) * torch.finfo(self.dtype).min

        # If a 2D or 3D attention mask is provided for the cross-attention
        # we need to make broadcastable to [batch_size, num_heads, seq_length, seq_length]
        if self.config.add_cross_attention and encoder_hidden_states is not None:
            encoder_batch_size, encoder_sequence_length, _ = encoder_hidden_states.size()
            encoder_hidden_shape = (encoder_batch_size, encoder_sequence_length)
            if encoder_attention_mask is None:
                encoder_attention_mask = torch.ones(encoder_hidden_shape, device=device)
            if _use_sdpa:
                encoder_attention_mask = _prepare_4d_attention_mask_for_sdpa(
                    mask=encoder_attention_mask, dtype=inputs_embeds.dtype, tgt_len=input_shape[-1]
                )
            elif not self._attn_implementation == "flash_attention_2":
                encoder_attention_mask = self.invert_attention_mask(encoder_attention_mask)
        else:
            encoder_attention_mask = None

        # Prepare head mask if needed
        # 1.0 in head_mask indicate we keep the head
        # attention_probs has shape bsz x n_heads x N x N
        # head_mask has shape n_layer x batch x n_heads x N x N
        head_mask = self.get_head_mask(head_mask, self.config.n_layer)

        if token_type_ids is not None:
            token_type_embeds = self.wte(token_type_ids)
            hidden_states = hidden_states + token_type_embeds

        hidden_states = self.drop(hidden_states)

        output_shape = (-1,) + input_shape[1:] + (hidden_states.size(-1),)

        if self.gradient_checkpointing and self.training:
            if use_cache:
                logger.warning_once(
                    "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`..."
                )
                use_cache = False

        presents = () if use_cache else None
        all_self_attentions = () if output_attentions else None
        all_cross_attentions = () if output_attentions and self.config.add_cross_attention else None
        all_hidden_states = () if output_hidden_states else None
        for i, (block, layer_past) in enumerate(zip(self.h, past_key_values)):
            # Model parallel
            if self.model_parallel:
                torch.cuda.set_device(hidden_states.device)
                # Ensure layer_past is on same device as hidden_states (might not be correct)
                if layer_past is not None:
                    layer_past = tuple(past_state.to(hidden_states.device) for past_state in layer_past)
                # Ensure that attention_mask is always on the same device as hidden_states
                if attention_mask is not None:
                    attention_mask = attention_mask.to(hidden_states.device)
                if isinstance(head_mask, torch.Tensor):
                    head_mask = head_mask.to(hidden_states.device)
            if output_hidden_states:
                all_hidden_states = all_hidden_states + (hidden_states,)

            if self.gradient_checkpointing and self.training:
                outputs = self._gradient_checkpointing_func(
                    block.__call__,
                    hidden_states,
                    None,
                    attention_mask,
                    head_mask[i],
                    encoder_hidden_states,
                    encoder_attention_mask,
                    use_cache,
                    output_attentions,
                )
            else:
                outputs = block(
                    hidden_states,
                    layer_past=layer_past,
                    attention_mask=attention_mask,
                    head_mask=head_mask[i],
                    encoder_hidden_states=encoder_hidden_states,
                    encoder_attention_mask=encoder_attention_mask,
                    use_cache=use_cache,
                    output_attentions=output_attentions,
                    k=k, # added 
                    T=T, # added
                )

            hidden_states = outputs[0]
            if use_cache is True:
                presents = presents + (outputs[1],)

            if output_attentions:
                all_self_attentions = all_self_attentions + (outputs[2 if use_cache else 1],)
                if self.config.add_cross_attention:
                    all_cross_attentions = all_cross_attentions + (outputs[3 if use_cache else 2],)

            # Model Parallel: If it's the last layer for that device, put things on the next device
            if self.model_parallel:
                for k, v in self.device_map.items():
                    if i == v[-1] and "cuda:" + str(k) != self.last_device:
                        hidden_states = hidden_states.to("cuda:" + str(k + 1))

        hidden_states = self.ln_f(hidden_states)

        hidden_states = hidden_states.view(output_shape)
        # Add last hidden state
        if output_hidden_states:
            all_hidden_states = all_hidden_states + (hidden_states,)

        if not return_dict:
            return tuple(
                v
                for v in [hidden_states, presents, all_hidden_states, all_self_attentions, all_cross_attentions]
                if v is not None
            )

        return BaseModelOutputWithPastAndCrossAttentions(
            last_hidden_state=hidden_states,
            past_key_values=presents,
            hidden_states=all_hidden_states,
            attentions=all_self_attentions,
            cross_attentions=all_cross_attentions,
        )



class Conv_GPT2LMHeadModel(GPT2LMHeadModel):
    def __init__(self, config):
        super().__init__(config)
        self.transformer = Conv_GPT2Model(config)
        self.lm_head = nn.Linear(config.n_embd, config.vocab_size, bias=False)

        # Model parallel
        self.model_parallel = False
        self.device_map = None

        # Initialize weights and apply final processing
        self.post_init()
    
    def forward(
        self,
        input_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Tuple[Tuple[torch.Tensor]]] = None,
        attention_mask: Optional[torch.FloatTensor] = None,
        token_type_ids: Optional[torch.LongTensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        head_mask: Optional[torch.FloatTensor] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        encoder_hidden_states: Optional[torch.Tensor] = None,
        encoder_attention_mask: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        k=10,
        T=1,
    ) -> Union[Tuple, CausalLMOutputWithCrossAttentions]:
        '''
        copy from modeling_gpt2.py, only modify the self.transformer forward part
        '''

        r"""
        labels (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
            Labels for language modeling. Note that the labels **are shifted** inside the model, i.e. you can set
            `labels = input_ids` Indices are selected in `[-100, 0, ..., config.vocab_size]` All labels set to `-100`
            are ignored (masked), the loss is only computed for labels in `[0, ..., config.vocab_size]`
        """
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        transformer_outputs = self.transformer(
            input_ids,
            past_key_values=past_key_values,
            attention_mask=attention_mask,
            token_type_ids=token_type_ids,
            position_ids=position_ids,
            head_mask=head_mask,
            inputs_embeds=inputs_embeds,
            encoder_hidden_states=encoder_hidden_states,
            encoder_attention_mask=encoder_attention_mask,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
            k=k, # added 
            T=T, # added
        )
        hidden_states = transformer_outputs[0]

        # Set device for model parallelism
        if self.model_parallel:
            torch.cuda.set_device(self.transformer.first_device)
            hidden_states = hidden_states.to(self.lm_head.weight.device)

        lm_logits = self.lm_head(hidden_states)

        loss = None
        if labels is not None:
            # move labels to correct device to enable model parallelism
            labels = labels.to(lm_logits.device)
            # Shift so that tokens < n predict n
            shift_logits = lm_logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            loss = loss_fct(shift_logits.view(-1, shift_logits.size(-1)), shift_labels.view(-1))

        if not return_dict:
            output = (lm_logits,) + transformer_outputs[1:]
            return ((loss,) + output) if loss is not None else output

        return CausalLMOutputWithCrossAttentions(
            loss=loss,
            logits=lm_logits,
            past_key_values=transformer_outputs.past_key_values,
            hidden_states=transformer_outputs.hidden_states,
            attentions=transformer_outputs.attentions,
            cross_attentions=transformer_outputs.cross_attentions,
        )


###################### for unit test
def _conv_attn_per_head(
        Q,  # [seq_len, head_d]
        K,    # [seq_len, head_d]
        V,  # [seq_len, head_d]
        attention_mask=None,   # [bs, 1, 1, seq_len] [1, 1, 1, 10]
        head_mask=None,
        k=5,
        T=1,
        delta = 0, epsilon = 0
    ):

    n, d = Q.shape

    
    Q = Q / torch.full(
        [], V.size(-1) ** 0.5, dtype=Q.dtype, device=Q.device
    )
    b_tilde, m, b= recover_k_conv(Q, K, k=k, T=T, delta=delta, epsilon=epsilon)

    QKV_approx = torch.zeros_like(Q, dtype=torch.float64)
    for i in range(k):
        QKV_approx += conv_with_fft_matrix(b_tilde[i, :], V, shift=n - m[i])
    
    D_approx = torch.zeros(n, dtype=torch.float64, device=Q.device)
    for i in range(k):
        D_approx += conv_with_fft(b_tilde[i, :], torch.ones(n, device=Q.device), shift=n - m[i])
    
    QKV_approx = (D_approx ** -1).unsqueeze(1) * QKV_approx

    return QKV_approx