import argparse
import os
import pprint
import shutil
from datetime import datetime

import torch
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.optim
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
from tensorboardX import SummaryWriter

import tools._init_paths
from config import cfg
from config import update_config
from core.function_semi import validate
from utils.utils import create_logger
from utils.utils import save_res_csv
from train_semi import _make_model, _make_loss, _model_to_gpu

import dataset
import models


def parse_args():
    parser = argparse.ArgumentParser(description='Train keypoints network')
    # general
    parser.add_argument('--cfg',
                        help='experiment configure file name',
                        required=True,
                        type=str)

    parser.add_argument('opts',
                        help="Modify config options using the command-line",
                        default=None,
                        nargs=argparse.REMAINDER)
    # philly
    parser.add_argument('--modelDir',
                        help='model directory',
                        type=str,
                        default='')
    parser.add_argument('--logDir',
                        help='log directory',
                        type=str,
                        default='')
    parser.add_argument('--dataDir',
                        help='data directory',
                        type=str,
                        default='')
    parser.add_argument('--prevModelDir',
                        help='prev Model directory',
                        type=str,
                        default='')

    args = parser.parse_args()

    return args



def _make_test_data(cfg, logger):
    """Initialise test loader as per config parameters    """
    test_transform = transforms.Compose([
                        dataset.transformers.CenterCrop(cfg.MODEL.IMAGE_SIZE[0]),
                        dataset.transformers.ToTensor(),
                        dataset.transformers.Normalize(mean=[0.485, 0.456, 0.406],
                                                       std =[0.229, 0.224, 0.225])
                        ])

    test_dataset = eval('dataset.'+cfg.DATASET.DATASET)(
        cfg, cfg.DATASET.ROOT, 'test', False, test_transform)

    test_loader = torch.utils.data.DataLoader(test_dataset,
                                                batch_size=cfg.TEST.BS*len(cfg.GPUS),
                                                shuffle=False,
                                                num_workers=cfg.WORKERS,
                                                pin_memory=cfg.PIN_MEMORY
                                            )

    return test_loader, test_dataset



def main():
    args = parse_args()
    update_config(cfg, args)

    logger, final_output_dir = create_logger(cfg, args.cfg, 'test', False)

    logger.info(pprint.pformat(args))
    logger.info(cfg)

    # cudnn related setting
    cudnn.benchmark = cfg.CUDNN.BENCHMARK
    torch.backends.cudnn.deterministic = cfg.CUDNN.DETERMINISTIC
    torch.backends.cudnn.enabled = cfg.CUDNN.ENABLED

    # Initialise models
    models_dict = _make_model(cfg, final_output_dir, False)

    for model_name in models_dict:
        model_state_file = os.path.join(final_output_dir, '{}_best.pth'.format(model_name))
        logger.info('=> loading model from {}'.format(model_state_file))
        if cfg.USE_GPU:
            models_dict[model_name].load_state_dict(torch.load(model_state_file))
        else:
            models_dict[model_name].load_state_dict(torch.load(model_state_file, map_location=torch.device('cpu')))

    # Initialise losses
    loss_dict = _make_loss(cfg, logger)

    # Initialise data loaders
    test_loader, test_dataset = _make_test_data(cfg, logger)

    models_dict = _model_to_gpu(models_dict, cfg)

    # evaluate on validation set
    perf_indicator = validate(
        cfg, test_loader, test_dataset,
        models_dict,
        loss_dict,
        final_output_dir
    )

    #Collect data for logs
    result = dict()
    result['date_time'] = datetime.now()
    result['dataset'] = cfg.DATASET.ROOT
    result['experiment'] = os.path.basename(args.cfg).split('.')[0]
    result['supervised'] = cfg.LOSS.SUPERVISED
    result['consistency'] = cfg.LOSS.CONSISTENCY
    result['reconstruction'] = cfg.LOSS.RECONSTRUCTION
    result['run_number'] = cfg.VERSION.split('_')[1]
    result['percent_labelled'] = cfg.VERSION.split('_')[0][1:]
    result['accuracy'] = perf_indicator

    save_res_csv(result, os.path.join('experiments', 'experiments_all.csv'))



if __name__ == '__main__':
    main()
