# Adapted from https://raw.githubusercontent.com/vwxyzjn/cleanrl/refs/heads/master/cleanrl/ppo_continuous_action.py
# Original work Copyright (c) 2019 CleanRL developers
# Modifications Copyright (c) 2025 King.com Ltd

import os
import random
import time
from dataclasses import dataclass
from datetime import datetime
import shutil
import pickle

import tyro
import matplotlib.pyplot as plt
import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.optim as optim
from torch.distributions.beta import Beta
from torch.utils.tensorboard import SummaryWriter

import envs  # register custom envs
from misc.utils import get_device


@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 3
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = False
    """if toggled, cuda will be enabled by default"""
    capture_video: bool = False
    """whether to capture videos of the agent performances (check out `videos` folder)"""
    env_id: str = "CircleStopEnv-angle0.0_radius0.9-v0"
    """the id of the environment"""
    exp_str: str = "default"

    # Algorithm specific arguments
    total_timesteps: int = 10_000
    """total timesteps of the experiments"""
    learning_rate: float = 3e-4
    """the learning rate of the optimizer"""
    num_envs: int = 1
    """the number of parallel game environments"""
    num_steps: int = 2048
    """the number of steps to run in each environment per policy rollout"""
    anneal_lr: bool = True
    """Toggle learning rate annealing for policy and value networks"""
    gamma: float = 0.99
    """the discount factor gamma"""
    gae_lambda: float = 0.95
    """the lambda for the general advantage estimation"""
    num_minibatches: int = 32
    """the number of mini-batches"""
    update_epochs: int = 10
    """the K epochs to update the policy"""
    norm_adv: bool = True
    """Toggles advantages normalization"""
    clip_coef: float = 0.2
    """the surrogate clipping coefficient"""
    clip_vloss: bool = True
    """Toggles whether or not to use a clipped loss for the value function, as per the paper."""
    ent_coef: float = 0.0
    """coefficient of the entropy"""
    vf_coef: float = 0.5
    """coefficient of the value function"""
    max_grad_norm: float = 0.5
    """the maximum norm for the gradient clipping"""
    target_kl: float = None
    """the target KL divergence threshold"""

    # to be filled in runtime
    batch_size: int = 0
    """the batch size (computed in runtime)"""
    minibatch_size: int = 0
    """the mini-batch size (computed in runtime)"""
    num_iterations: int = 0
    """the number of iterations (computed in runtime)"""


def make_env(env_id, idx, capture_video, run_name, gamma, render_mode=None):
    def thunk():
        if capture_video and idx == 0:
            env = gym.make(env_id, render_mode="rgb_array")
            env = gym.wrappers.RecordVideo(env, f"videos/{run_name}")
        else:
            env = gym.make(env_id, render_mode=render_mode)
        env = gym.wrappers.FlattenObservation(env)
        env = gym.wrappers.RecordEpisodeStatistics(env)
        env = gym.wrappers.NormalizeObservation(env)
        env = gym.wrappers.NormalizeReward(env, gamma=gamma)
        env = gym.wrappers.TransformReward(env, lambda reward: np.clip(reward, -10, 10))
        return env

    return thunk


def layer_init(layer, std=np.sqrt(2), bias_const=0.0):
    torch.nn.init.orthogonal_(layer.weight, std)
    torch.nn.init.constant_(layer.bias, bias_const)
    return layer


class ActorCritic(nn.Module):
    """
    Actor and critic networks.
    We use a beta distribution for the actor, which is parameterized by alpha and beta for each action dim.
    """
    def __init__(self, envs, hidden_size=64):
        super().__init__()
        self.critic = nn.Sequential(
            layer_init(nn.Linear(np.array(envs.single_observation_space.shape).prod(), hidden_size)),
            nn.Tanh(),
            layer_init(nn.Linear(hidden_size, hidden_size)),
            nn.Tanh(),
            layer_init(nn.Linear(hidden_size, 1), std=1.0),
        )
        self.actor_alpha = nn.Sequential(
            layer_init(nn.Linear(np.array(envs.single_observation_space.shape).prod(), hidden_size)),
            nn.Tanh(),
            layer_init(nn.Linear(hidden_size, hidden_size)),
            nn.Tanh(),
            layer_init(nn.Linear(hidden_size, np.prod(envs.single_action_space.shape)), std=0.01),
        )
        self.actor_beta = nn.Sequential(
            layer_init(nn.Linear(np.array(envs.single_observation_space.shape).prod(), hidden_size)),
            nn.Tanh(),
            layer_init(nn.Linear(hidden_size, hidden_size)),
            nn.Tanh(),
            layer_init(nn.Linear(hidden_size, np.prod(envs.single_action_space.shape)), std=0.01),
        )

    def get_value(self, x):
        return self.critic(x)

    def get_action_and_value(self, x, action=None):
        alpha_param =torch.exp(self.actor_alpha(x)) + 1
        beta_param = torch.exp(self.actor_beta(x)) + 1
        probs = Beta(alpha_param, beta_param)
        if action is None:
            action = probs.sample()
        return action, probs.log_prob(action).sum(1), probs.entropy().sum(1), self.critic(x)


if __name__ == "__main__":
    args = tyro.cli(Args)
    args.batch_size = int(args.num_envs * args.num_steps)
    args.minibatch_size = int(args.batch_size // args.num_minibatches)
    args.num_iterations = args.total_timesteps // args.batch_size
    start_time = datetime.now().replace(microsecond=0)
    start_time_str = start_time.strftime("%y-%m-%d_%H-%M-%S")
    run_name = f"{args.env_id}_{args.seed}_{start_time_str}_{args.exp_str}"
    exp_dir = f"PPO_runs/{run_name}"
    os.makedirs(f"{exp_dir}/trajectories", exist_ok=True)

    writer = SummaryWriter(log_dir=f"{exp_dir}")

    with open(f"{exp_dir}/args.txt", "w") as f:
        for key, value in vars(args).items():
            f.write(f"{key}: {value}\n")

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = get_device(use_cuda=args.cuda)

    # env setup
    envs = gym.vector.SyncVectorEnv(
        [make_env(args.env_id, i, args.capture_video, run_name, args.gamma) for i in range(args.num_envs)]
    )
    assert isinstance(envs.single_action_space, gym.spaces.Box), "only continuous action space is supported"

    agent = ActorCritic(envs).to(device)
    optimizer = optim.Adam(agent.parameters(), lr=args.learning_rate, eps=1e-5)

    # ALGO Logic: Storage setup
    obs = torch.zeros((args.num_steps, args.num_envs) + envs.single_observation_space.shape).to(device)
    actions = torch.zeros((args.num_steps, args.num_envs) + envs.single_action_space.shape).to(device)
    logprobs = torch.zeros((args.num_steps, args.num_envs)).to(device)
    rewards = torch.zeros((args.num_steps, args.num_envs)).to(device)
    dones = torch.zeros((args.num_steps, args.num_envs)).to(device)
    values = torch.zeros((args.num_steps, args.num_envs)).to(device)

    # TRY NOT TO MODIFY: start the game
    global_step = 0
    start_time = time.time()
    next_obs, _ = envs.reset(seed=args.seed)
    next_obs = torch.Tensor(next_obs).to(device)
    next_done = torch.zeros(args.num_envs).to(device)

    for iteration in range(1, args.num_iterations + 1):
        # Annealing the rate if instructed to do so.
        if args.anneal_lr:
            frac = 1.0 - (iteration - 1.0) / args.num_iterations
            lrnow = frac * args.learning_rate
            optimizer.param_groups[0]["lr"] = lrnow

        for step in range(0, args.num_steps):
            global_step += args.num_envs
            obs[step] = next_obs
            dones[step] = next_done

            # ALGO LOGIC: action logic
            with torch.no_grad():
                action, logprob, _, value = agent.get_action_and_value(next_obs)
                values[step] = value.flatten()
            actions[step] = action
            logprobs[step] = logprob

            # scale action from beta range [0, 1] to action space range
            low = envs.unwrapped.single_action_space.low
            high = envs.unwrapped.single_action_space.high
            scaled_action = low + (high - low) * action.detach().cpu().numpy()

            # TRY NOT TO MODIFY: execute the game and log data.
            next_obs, reward, terminations, truncations, infos = envs.step(scaled_action)
            next_done = np.logical_or(terminations, truncations)

            for env_idx, done in enumerate(next_done):
                if done and "episode" in infos:
                    if not "final_info" in infos:
                        infos["final_info"] = [[] for _ in range(args.num_envs)]
                    infos["final_info"][env_idx] = {"episode": {"r": infos["episode"]["r"][env_idx], "l": infos["episode"]["l"][env_idx]}}

                    if "success" in infos.keys():
                        infos["final_info"][env_idx]["success"] = infos["success"][env_idx]
                        infos["final_info"][env_idx]["sparse_reward"] = infos["sparse_reward"][env_idx]

            rewards[step] = torch.tensor(reward).to(torch.float32).to(device).view(-1)
            next_obs, next_done = torch.Tensor(next_obs).to(device), torch.Tensor(next_done).to(device)

            if "final_info" in infos:
                for env_idx, info in enumerate(infos["final_info"]):
                    if info and "episode" in info:
                        print(f"global_step={global_step}, episodic_return={info['episode']['r']}, episodic_length={info['episode']['l']}")
                        writer.add_scalar("charts/episodic_return", info["episode"]["r"], global_step)
                        writer.add_scalar("charts/episodic_length", info["episode"]["l"], global_step)
                        writer.add_scalar("charts/episodic_success", info["success"], global_step)
                        writer.add_scalar("charts/last_spare_reward", info["sparse_reward"], global_step)

                        try:
                            envs.envs[env_idx].unwrapped.save_trajectory(save_path=f"{exp_dir}/trajectories/{global_step}_{env_idx}.pkl")
                            assert len(infos["final_info"]) == 1, "What happens when multiple envs finished at the same time?"
                            break
                        except AttributeError:
                            pass

        # bootstrap value if not done
        with torch.no_grad():
            next_value = agent.get_value(next_obs).reshape(1, -1)
            advantages = torch.zeros_like(rewards).to(device)
            lastgaelam = 0
            for t in reversed(range(args.num_steps)):
                if t == args.num_steps - 1:
                    nextnonterminal = 1.0 - next_done
                    nextvalues = next_value
                else:
                    nextnonterminal = 1.0 - dones[t + 1]
                    nextvalues = values[t + 1]
                delta = rewards[t] + args.gamma * nextvalues * nextnonterminal - values[t]
                advantages[t] = lastgaelam = delta + args.gamma * args.gae_lambda * nextnonterminal * lastgaelam
            returns = advantages + values

        # flatten the batch
        b_obs = obs.reshape((-1,) + envs.single_observation_space.shape)
        b_logprobs = logprobs.reshape(-1)
        b_actions = actions.reshape((-1,) + envs.single_action_space.shape)
        b_advantages = advantages.reshape(-1)
        b_returns = returns.reshape(-1)
        b_values = values.reshape(-1)

        # Optimizing the policy and value network
        b_inds = np.arange(args.batch_size)
        clipfracs = []
        for epoch in range(args.update_epochs):
            np.random.shuffle(b_inds)
            for start in range(0, args.batch_size, args.minibatch_size):
                end = start + args.minibatch_size
                mb_inds = b_inds[start:end]

                _, newlogprob, entropy, newvalue = agent.get_action_and_value(b_obs[mb_inds], b_actions[mb_inds])
                logratio = newlogprob - b_logprobs[mb_inds]
                ratio = logratio.exp()

                with torch.no_grad():
                    old_approx_kl = (-logratio).mean()
                    approx_kl = ((ratio - 1) - logratio).mean()
                    clipfracs += [((ratio - 1.0).abs() > args.clip_coef).float().mean().item()]

                mb_advantages = b_advantages[mb_inds]
                if args.norm_adv:
                    mb_advantages = (mb_advantages - mb_advantages.mean()) / (mb_advantages.std() + 1e-8)

                # Policy loss
                pg_loss1 = -mb_advantages * ratio
                pg_loss2 = -mb_advantages * torch.clamp(ratio, 1 - args.clip_coef, 1 + args.clip_coef)
                pg_loss = torch.max(pg_loss1, pg_loss2).mean()

                # Value loss
                newvalue = newvalue.view(-1)
                if args.clip_vloss:
                    v_loss_unclipped = (newvalue - b_returns[mb_inds]) ** 2
                    v_clipped = b_values[mb_inds] + torch.clamp(
                        newvalue - b_values[mb_inds],
                        -args.clip_coef,
                        args.clip_coef,
                    )
                    v_loss_clipped = (v_clipped - b_returns[mb_inds]) ** 2
                    v_loss_max = torch.max(v_loss_unclipped, v_loss_clipped)
                    v_loss = 0.5 * v_loss_max.mean()
                else:
                    v_loss = 0.5 * ((newvalue - b_returns[mb_inds]) ** 2).mean()

                entropy_loss = entropy.mean()
                loss = pg_loss - args.ent_coef * entropy_loss + v_loss * args.vf_coef

                optimizer.zero_grad()
                loss.backward()
                nn.utils.clip_grad_norm_(agent.parameters(), args.max_grad_norm)
                optimizer.step()

            if args.target_kl is not None and approx_kl > args.target_kl:
                break

        y_pred, y_true = b_values.cpu().numpy(), b_returns.cpu().numpy()
        var_y = np.var(y_true)
        explained_var = np.nan if var_y == 0 else 1 - np.var(y_true - y_pred) / var_y

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        writer.add_scalar("charts/learning_rate", optimizer.param_groups[0]["lr"], global_step)
        writer.add_scalar("losses/value_loss", v_loss.item(), global_step)
        writer.add_scalar("losses/policy_loss", pg_loss.item(), global_step)
        writer.add_scalar("losses/entropy", entropy_loss.item(), global_step)
        writer.add_scalar("losses/old_approx_kl", old_approx_kl.item(), global_step)
        writer.add_scalar("losses/approx_kl", approx_kl.item(), global_step)
        writer.add_scalar("losses/clipfrac", np.mean(clipfracs), global_step)
        writer.add_scalar("losses/explained_variance", explained_var, global_step)
        writer.add_scalar("charts/SPS", int(global_step / (time.time() - start_time)), global_step)

    traj_dir = f"{exp_dir}/trajectories"
    traj_files = [file for file in os.listdir(traj_dir) if file.endswith(".pkl")]
    trajs = []

    traj_save_dir = os.path.dirname(exp_dir)

    with open(os.path.join(exp_dir, "trajectories.pkl"), "wb") as out_file:
        for file in traj_files:
            with open(os.path.join(traj_dir, file), "rb") as in_file:
                traj = pickle.load(in_file)
                trajs.append(traj)
        pickle.dump(trajs, out_file)

    print(f"Combined {len(traj_files)} trajectories into a single file")
    print(f"Saved to {os.path.join(exp_dir, 'trajectories.pkl')}")

    shutil.rmtree(traj_dir)
    print(f"Removed {traj_dir} with individual trajectory files.")

    torch.save(agent.state_dict(), f"{exp_dir}/ppo_agent.pth")

    fig, ax = plt.subplots()
    obs, _ = envs.reset()
    episodic_returns = []
    agent = ActorCritic(envs).to(device)
    model_path = f"{exp_dir}/ppo_agent.pth"
    agent.load_state_dict(torch.load(model_path, map_location=device))
    agent.eval()
    pos_hist = []
    while len(episodic_returns) < 10:
        actions, _, _, _ = agent.get_action_and_value(torch.Tensor(obs).to(device))

        low = envs.unwrapped.single_action_space.low
        high = envs.unwrapped.single_action_space.high
        scaled_action = low + (high - low) * actions.detach().cpu().numpy()

        next_obs, reward, terminations, truncations, infos = envs.step(scaled_action)
        next_done = np.logical_or(terminations, truncations)
        pos_hist.append(infos["agent_pos"][0].copy())

        for env_idx, done in enumerate(next_done):
            if done and "episode" in infos:
                if not "final_info" in infos:
                    infos["final_info"] = [[] for _ in range(args.num_envs)]
                infos["final_info"][env_idx] = {"episode": {"r": infos["episode"]["r"][env_idx], "l": infos["episode"]["l"][env_idx]}}

                if "success" in infos.keys():
                    infos["final_info"][env_idx]["success"] = infos["success"][env_idx]
                    infos["final_info"][env_idx]["sparse_reward"] = infos["sparse_reward"][env_idx]

        if "final_info" in infos:
            for info in infos["final_info"]:
                if "episode" not in info:
                    continue
                pos_hist = np.array(pos_hist)
                ax.plot(pos_hist[0:-1, 0], pos_hist[0:-1, 1])
                ax.set_xlim(-envs.envs[0].unwrapped.grid_size, envs.envs[0].unwrapped.grid_size)
                ax.set_ylim(-envs.envs[0].unwrapped.grid_size, envs.envs[0].unwrapped.grid_size)
                pos_hist = []

                print(f"eval_episode={len(episodic_returns)}, episodic_return={info['episode']['r']}, episodic_length={info['episode']['l']}")
                episodic_returns += [info["episode"]["r"]]
        obs = next_obs

    ax.scatter(envs.envs[0].unwrapped.goal_position[0], envs.envs[0].unwrapped.goal_position[1], c='r', marker='x')

    ax.set_aspect('equal', 'box')
    plt.title(f"Rollouts after training (mean_return={np.mean(episodic_returns):.2f})")
    plt.savefig(f"{exp_dir}/rollouts.png")

    envs.close()
    writer.close()

    # EXTREMELY IMPORTANT: Last print statement must be experiment directory for bash script to capture it
    abs_dir = os.path.abspath(f"{exp_dir}")
    print(abs_dir, flush=True)
