import json
import os
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Union

import pandas as pd
import typer

app = typer.Typer()


@app.command()
def simple_gather(results_dir: str, output_path: Optional[str] = None,
                  prefix: str = "", dry_run: bool = False,
                  target_file: str = "best.json", target_dir: str = "agg"):
    """Aggregate graphgym agg results from a directory.


    If --ouput_path is not specified, then automatically construct the output
    path as {results_dir}/../aggregated_results/{prefix}{%Y-%m-%d_%H-%M-%S}.csv

    If --dry_run is specified, then do not save the output.

    """
    results: List[Dict[str, Union[str, float, int]]] = []
    for dir_, _, files in os.walk(results_dir):
        if target_file in files:
            path = os.path.join(dir_, target_file)
            print(f"Loading results from {path}")

            # Load raw results from json
            with open(path, "r") as f:
                new_result = json.load(f)

            # Add metadata inferred from path
            terms = dir_.split(os.path.sep)
            name, split = terms[-3], terms[-1]
            new_result["split"] = split
            new_result["name"] = name
            results.append(new_result)

    df = pd.DataFrame(results)
    print(df)

    if output_path is None:  # infer output paths
        base_dir = Path(results_dir).resolve().parent
        date_str = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
        output_path = base_dir / "aggregated_results" / f"{prefix}{date_str}.csv"

    if dry_run:
        print(f"Results will be saved to {output_path}")
    else:
        os.makedirs(os.path.split(output_path)[0], exist_ok=True)
        df.to_csv(output_path, index=False)
        print(f"Results saved to {output_path}")


if __name__ == "__main__":
    app()
