from functools import partial

import torch
import torch.nn as nn
import torch.nn.functional as F
from timm.models.layers import trunc_normal_

from .swin_transformer import SwinTransformer
from .vision_transformer import VisionTransformer

from timm.models.vision_transformer import PatchEmbed, Block

from .util.pos_embed import get_2d_sincos_pos_embed

# Generator
## MAE

class MaskedAutoencoderViT(nn.Module):
    """ Masked Autoencoder with VisionTransformer backbone
    """
    def __init__(self, img_size=224, patch_size=16, in_chans=3,
                 embed_dim=1024, depth=24, num_heads=16,
                 decoder_embed_dim=512, decoder_depth=8, decoder_num_heads=16,
                 mlp_ratio=4., norm_layer=nn.LayerNorm, norm_pix_loss=False):
        super().__init__()

        # --------------------------------------------------------------------------
        # MAE encoder specifics
        self.embed_dim = embed_dim
        self.patch_embed = PatchEmbed(img_size, patch_size, in_chans, embed_dim)
        num_patches = self.patch_embed.num_patches
        self.norm_layer = norm_layer
        self.cls_token = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + 1, embed_dim), requires_grad=False)  # fixed sin-cos embedding

        self.blocks = nn.ModuleList([
            Block(embed_dim, num_heads, mlp_ratio, qkv_bias=True, norm_layer=norm_layer)
            for i in range(depth)])
        self.norm = norm_layer(embed_dim)
        # --------------------------------------------------------------------------

        # --------------------------------------------------------------------------
        # MAE decoder specifics
        self.decoder_embed = nn.Linear(embed_dim, decoder_embed_dim, bias=True)

        self.mask_token = nn.Parameter(torch.zeros(1, 1, decoder_embed_dim))

        self.decoder_pos_embed = nn.Parameter(torch.zeros(1, num_patches + 1, decoder_embed_dim), requires_grad=False)  # fixed sin-cos embedding

        self.decoder_blocks = nn.ModuleList([
            Block(decoder_embed_dim, decoder_num_heads, mlp_ratio, qkv_bias=True, norm_layer=norm_layer)
            for i in range(decoder_depth)])

        self.decoder_norm = norm_layer(decoder_embed_dim)
        self.decoder_pred = nn.Linear(decoder_embed_dim, patch_size**2 * in_chans, bias=True) # decoder to patch
        # --------------------------------------------------------------------------

        self.norm_pix_loss = norm_pix_loss

        self.initialize_weights()

    def initialize_weights(self):
        # initialization
        # initialize (and freeze) pos_embed by sin-cos embedding
        pos_embed = get_2d_sincos_pos_embed(self.pos_embed.shape[-1], int(self.patch_embed.num_patches**.5), cls_token=True)
        self.pos_embed.data.copy_(torch.from_numpy(pos_embed).float().unsqueeze(0))

        decoder_pos_embed = get_2d_sincos_pos_embed(self.decoder_pos_embed.shape[-1], int(self.patch_embed.num_patches**.5), cls_token=True)
        self.decoder_pos_embed.data.copy_(torch.from_numpy(decoder_pos_embed).float().unsqueeze(0))

        # initialize patch_embed like nn.Linear (instead of nn.Conv2d)
        w = self.patch_embed.proj.weight.data
        torch.nn.init.xavier_uniform_(w.view([w.shape[0], -1]))

        # timm's trunc_normal_(std=.02) is effectively normal_(std=0.02) as cutoff is too big (2.)
        torch.nn.init.normal_(self.cls_token, std=.02)
        torch.nn.init.normal_(self.mask_token, std=.02)

        # initialize nn.Linear and nn.LayerNorm
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            # we use xavier_uniform following official JAX ViT:
            torch.nn.init.xavier_uniform_(m.weight)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def patchify(self, imgs):
        """
        imgs: (N, 3, H, W)
        x: (N, L, patch_size**2 *3)
        """
        p = self.patch_embed.patch_size[0]
        assert imgs.shape[2] == imgs.shape[3] and imgs.shape[2] % p == 0

        h = w = imgs.shape[2] // p
        x = imgs.reshape(shape=(imgs.shape[0], 3, h, p, w, p))
        x = torch.einsum('nchpwq->nhwpqc', x)
        x = x.reshape(shape=(imgs.shape[0], h * w, p**2 * 3))
        return x

    def unpatchify(self, x):
        """
        x: (N, L, patch_size**2 *3)
        imgs: (N, 3, H, W)
        """
        p = self.patch_embed.patch_size[0]
        h = w = int(x.shape[1]**.5)
        assert h * w == x.shape[1]
        
        x = x.reshape(shape=(x.shape[0], h, w, p, p, 3))
        x = torch.einsum('nhwpqc->nchpwq', x)
        imgs = x.reshape(shape=(x.shape[0], 3, h * p, h * p))
        return imgs

    def mask_unpatchify(self, mask):
        """
        mask: (N, patch_size**2)
        mask_rec: (N, H, W)
        """
        h = w = int(mask.shape[1]**.5)
        assert h * w == mask.shape[1]
        
        mask_rec = mask.reshape(shape=(mask.shape[0], h, w))
        return mask_rec

    def random_masking(self, x, mask_ratio):
        """
        Perform per-sample random masking by per-sample shuffling.
        Per-sample shuffling is done by argsort random noise.
        x: [N, L, D], sequence
        """
        N, L, D = x.shape  # batch, length, dim
        len_keep = int(L * (1 - mask_ratio))
        
        noise = torch.rand(N, L, device=x.device)  # noise in [0, 1]
        
        # sort noise for each sample
        ids_shuffle = torch.argsort(noise, dim=1)  # ascend: small is keep, large is remove
        ids_restore = torch.argsort(ids_shuffle, dim=1)

        # keep the first subset
        ids_keep = ids_shuffle[:, :len_keep]
        x_masked = torch.gather(x, dim=1, index=ids_keep.unsqueeze(-1).repeat(1, 1, D))

        # generate the binary mask: 0 is keep, 1 is remove
        mask = torch.ones([N, L], device=x.device)
        mask[:, :len_keep] = 0
        # unshuffle to get the binary mask
        mask = torch.gather(mask, dim=1, index=ids_restore)
        return x_masked, mask, ids_restore

    def forward_encoder(self, x, mask_ratio):
        # embed patches
        x = self.patch_embed(x)

        # add pos embed w/o cls token
        x = x + self.pos_embed[:, 1:, :]

        # masking: length -> length * mask_ratio
        x, mask, ids_restore = self.random_masking(x, mask_ratio)

        # append cls token
        cls_token = self.cls_token + self.pos_embed[:, :1, :]
        cls_tokens = cls_token.expand(x.shape[0], -1, -1)
        x = torch.cat((cls_tokens, x), dim=1)

        # apply Transformer blocks
        for blk in self.blocks:
            x = blk(x)
        x = self.norm(x)

        return x, mask, ids_restore

    def forward_decoder(self, x, ids_restore):
        # embed tokens
        x = self.decoder_embed(x)

        # append mask tokens to sequence
        mask_tokens = self.mask_token.repeat(x.shape[0], ids_restore.shape[1] + 1 - x.shape[1], 1)
        x_ = torch.cat([x[:, 1:, :], mask_tokens], dim=1)  # no cls token
        x_ = torch.gather(x_, dim=1, index=ids_restore.unsqueeze(-1).repeat(1, 1, x.shape[2]))  # unshuffle
        x = torch.cat([x[:, :1, :], x_], dim=1)  # append cls token

        # add pos embed
        x = x + self.decoder_pos_embed

        # apply Transformer blocks
        for blk in self.decoder_blocks:
            x = blk(x)
        x = self.decoder_norm(x)

        # predictor projection
        x = self.decoder_pred(x)

        # remove cls token
        x = x[:, 1:, :]

        return x

    def forward_loss(self, imgs, pred, mask):
        """
        imgs: [N, 3, H, W]
        pred: [N, L, p*p*3]
        mask: [N, L], 0 is keep, 1 is remove, 
        """
        target = self.patchify(imgs)
        if self.norm_pix_loss:
            mean = target.mean(dim=-1, keepdim=True)
            var = target.var(dim=-1, keepdim=True)
            target = (target - mean) / (var + 1.e-6)**.5

        loss = (pred - target) ** 2
        loss = loss.mean(dim=-1)  # [N, L], mean loss per patch

        loss = (loss * mask).sum() / mask.sum()  # mean loss on removed patches
        return loss

    def forward(self, imgs, mask_ratio=0.75):
        latent, mask, ids_restore = self.forward_encoder(imgs, mask_ratio)
        noise = torch.randn_like(latent)
        latent = latent + noise # * 1.0
        pred = self.forward_decoder(latent, ids_restore)  # [N, L, p*p*3]
        loss = self.forward_loss(imgs, pred, mask)
        img_rec = self.unpatchify(pred)
        #print('mask'+str(mask.shape))
        # visualize the mask
        img_mask = mask.detach().unsqueeze(-1).repeat(1, 1, self.patch_embed.patch_size[0]**2 *3)  # (N, H*W, p*p*3)
        img_mask = self.unpatchify(img_mask)  # 1 is removing, 0 is keeping
        mask = self.mask_unpatchify(mask)
        label = torch.zeros_like(mask, device=mask.device)
        # print('latent:'+str(latent.shape))
        # print('mask:'+str(mask.shape))
        # print('pred:'+str(pred.shape))
        # print('img_rec:'+str(img_rec.shape))
        # print('img_mask:'+str(img_mask.shape))
        # print('label:'+str(label.shape))
        return loss, img_rec, img_mask, label, mask

## Swin-Transformer

class SwinTransformerForGenerator(SwinTransformer):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        assert self.num_classes == 0

        self.mask_token = nn.Parameter(torch.zeros(1, 1, self.embed_dim))
        trunc_normal_(self.mask_token, mean=0., std=.02)

    def forward(self, x, mask):
        x = self.patch_embed(x)

        assert mask is not None
        B, L, _ = x.shape # B pH×pW embed_dim

        mask_tokens = self.mask_token.expand(B, L, -1)
        w = mask.flatten(1).unsqueeze(-1).type_as(mask_tokens) # mask:[B, pH, pW] flatten-> [B, pH×pW] unsqueeze-> [B, pH×pW, 1]
        x = x * (1. - w) + mask_tokens * w
        # mask中为1的位置表示被mask掉，因此(1. - w)*x就是将x一部分内容mask掉
        # mask_token是一个可学习的参数，分配给mask的参数

        if self.ape:
            x = x + self.absolute_pos_embed
        x = self.pos_drop(x)

        for layer in self.layers:
            x = layer(x)
        x = self.norm(x)

        x = x.transpose(1, 2)
        B, C, L = x.shape
        H = W = int(L ** 0.5)
        x = x.reshape(B, C, H, W)
        return x

    @torch.jit.ignore
    def no_weight_decay(self):
        return super().no_weight_decay() | {'mask_token'}

## ViT

class VisionTransformerForSimMIM(VisionTransformer):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        assert self.num_classes == 0

        self.mask_token = nn.Parameter(torch.zeros(1, 1, self.embed_dim))
        self._trunc_normal_(self.mask_token, std=.02)

    def _trunc_normal_(self, tensor, mean=0., std=1.):
        trunc_normal_(tensor, mean=mean, std=std, a=-std, b=std)

    def forward(self, x, mask):
        x = self.patch_embed(x)

        assert mask is not None
        B, L, _ = x.shape

        mask_token = self.mask_token.expand(B, L, -1)
        w = mask.flatten(1).unsqueeze(-1).type_as(mask_token)
        x = x * (1 - w) + mask_token * w

        cls_tokens = self.cls_token.expand(B, -1, -1)  # stole cls_tokens impl from Phil Wang, thanks
        x = torch.cat((cls_tokens, x), dim=1)

        if self.pos_embed is not None:
            x = x + self.pos_embed
        x = self.pos_drop(x)

        rel_pos_bias = self.rel_pos_bias() if self.rel_pos_bias is not None else None
        for blk in self.blocks:
            x = blk(x, rel_pos_bias=rel_pos_bias)
        x = self.norm(x)

        x = x[:, 1:]
        B, L, C = x.shape
        H = W = int(L ** 0.5)
        x = x.permute(0, 2, 1).reshape(B, C, H, W)  # 这里直接将形状调整，并没有将patch变回去
        return x

## Generator Creator

class SimMIM(nn.Module):
    def __init__(self, encoder, encoder_stride):
        super().__init__()
        self.encoder = encoder
        self.encoder_stride = encoder_stride # encoder_stride是patch_size? 将图像中多少个像素看成一个patch

        # 由于encoder划分patch之后，将输出的结果直接进行了形状调整，因此输出的结果是一个C维很大的[B, C, H/pH, W/pW]的特征图
        # 因此这里采用2D卷积将通道进行调整，C 调整为 pH×pW×3，然后再利用PixelShuffle将[B, pH×pW×3, H/pH, W/pW]变为[B, 3, H, W]
        self.decoder = nn.Sequential(
            nn.Conv2d(
                in_channels=self.encoder.num_features,
                out_channels=self.encoder_stride ** 2 * 3, kernel_size=1),
            nn.PixelShuffle(self.encoder_stride),
        )

        self.in_chans = self.encoder.in_chans
        self.patch_size = self.encoder.patch_size

    def forward(self, x, mask):
        z = self.encoder(x, mask)
        x_rec = self.decoder(z)

        mask = mask.repeat_interleave(self.patch_size, 1).repeat_interleave(self.patch_size, 2).unsqueeze(1).contiguous()
        loss_recon = F.l1_loss(x, x_rec, reduction='none')
        loss = (loss_recon * mask).sum() / (mask.sum() + 1e-5) / self.in_chans
        return x_rec, loss

    @torch.jit.ignore
    def no_weight_decay(self):
        if hasattr(self.encoder, 'no_weight_decay'):
            return {'encoder.' + i for i in self.encoder.no_weight_decay()}
        return {}

    @torch.jit.ignore
    def no_weight_decay_keywords(self):
        if hasattr(self.encoder, 'no_weight_decay_keywords'):
            return {'encoder.' + i for i in self.encoder.no_weight_decay_keywords()}
        return {}

# Discriminator
## Swin-Transformer
class SwinTransformerForDiscriminator(SwinTransformer):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        assert self.num_classes == 0

        #self.mask_token = nn.Parameter(torch.zeros(1, 1, self.embed_dim))
        #trunc_normal_(self.mask_token, mean=0., std=.02)

    def forward(self, x, x_rec, mask):
        if x_rec is not None:
            _, _, h, w = x.shape
            _, mh, mw = mask.shape
            w = mask.repeat_interleave(int(h / mh), dim=1).repeat_interleave(int(w / mw), dim=2)
            w = w.unsqueeze(1)
            x = x * (1. - w) + x_rec * w
        x = self.patch_embed(x)
        assert mask is not None
        B, L, _ = x.shape # B pH×pW embed_dim

        #w = mask.flatten(1).unsqueeze(-1).type_as(x) # mask:[B, pH, pW] flatten-> [B, pH×pW] unsqueeze-> [B, pH×pW, 1]
        #x = x * (1. - w) + x_rec * w
        # mask中为1的位置表示被mask掉，因此(1. - w)*x就是将x一部分内容mask掉
        # mask_token是一个可学习的参数，分配给mask的参数

        if self.ape:
            x = x + self.absolute_pos_embed
        x = self.pos_drop(x)

        for layer in self.layers:
            x = layer(x)
        x = self.norm(x)

        x = x.transpose(1, 2)
        B, C, L = x.shape
        H = W = int(L ** 0.5)
        x = x.reshape(B, C, H, W)
        return x

    @torch.jit.ignore
    def no_weight_decay(self):
        return super().no_weight_decay() | {'mask_token'}

## ViT

class VisionTransformerForDiscriminator(VisionTransformer):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        assert self.num_classes == 0

        # self._trunc_normal_(self.mask_token, std=.02)

    def _trunc_normal_(self, tensor, mean=0., std=1.):
        trunc_normal_(tensor, mean=mean, std=std, a=-std, b=std)

    def forward(self, x, x_rec, mask):
        if x_rec is not None:
            # _, _, h, w = x.shape
            # _, mh, mw = mask.shape
            # w = mask.repeat_interleave(int(h / mh), dim=1).repeat_interleave(int(w / mw), dim=2)
            # w = w.unsqueeze(1)
            # x = x * (1. - w) + x_rec * w
            x = x * (1. - mask) + x_rec * mask

        x = self.patch_embed(x)
        assert mask is not None
        B, L, _ = x.shape


        cls_tokens = self.cls_token.expand(B, -1, -1)  # stole cls_tokens impl from Phil Wang, thanks
        x = torch.cat((cls_tokens, x), dim=1)

        if self.pos_embed is not None:
            x = x + self.pos_embed
        x = self.pos_drop(x)

        rel_pos_bias = self.rel_pos_bias() if self.rel_pos_bias is not None else None
        for blk in self.blocks:
            x = blk(x, rel_pos_bias=rel_pos_bias)
        x = self.norm(x)

        cls_t = x[:, 0]
        
        x = x[:, 1:]
        B, L, C = x.shape
        H = W = int(L ** 0.5)
        x = x.permute(0, 2, 1).reshape(B, C, H, W)  # 这里直接将形状调整，并没有将patch变回去
        return x, cls_t

## Discriminator Creator

class Vision_Electra_Discriminator(nn.Module):
    def __init__(self, discriminator):
        super().__init__()
        # Discriminator
        self.discriminator = discriminator

        # 0/1 image classifer
        self.image_classifer = nn.Sequential(
            nn.Linear(self.discriminator.num_features, 1)
        )

        # segmentation head
        self.seg_head = nn.Sequential(
            nn.Conv2d(
                in_channels=self.discriminator.num_features,
                out_channels=1, kernel_size=1),
            # nn.Sigmoid(),
        )

        self.in_chans = self.discriminator.in_chans
        self.patch_size = self.discriminator.patch_size
        
        # image loss
        self.loss1 = torch.nn.BCEWithLogitsLoss()

        # self.loss = torch.nn.BCELoss()
        self.loss = torch.nn.BCEWithLogitsLoss()

    def forward(self, x, x_rec, mask, image_label, mask_label, label=None): # mask: [64, 14, 14] label: [64, 14, 14]
        if x_rec is None:
            z, cls_tokens = self.discriminator(x, None, mask) # [B, 768, 14, 14]
            z = self.seg_head(z).squeeze(1) # [B, 14, 14]
            cls = self.image_classifer(cls_tokens.squeeze(1))
        else:
            z, cls_tokens = self.discriminator(x, x_rec, mask) # [B, 768, 14, 14]
            z = self.seg_head(z).squeeze(1) # [B, 14, 14]
            cls = self.image_classifer(cls_tokens.squeeze(1))
        
        loss1 = self.loss1(cls, image_label)
            
        if label is None:
            loss = self.loss(z, mask_label.double())
        else:
            loss = self.loss(z, label.double())
        return z, loss + loss1

    @torch.jit.ignore
    def no_weight_decay(self):
        if hasattr(self.discriminator, 'no_weight_decay'):
            return {'discriminator.' + i for i in self.discriminator.no_weight_decay()}
        return {}

    @torch.jit.ignore
    def no_weight_decay_keywords(self):
        if hasattr(self.discriminator, 'no_weight_decay_keywords'):
            return {'discriminator.' + i for i in self.discriminator.no_weight_decay_keywords()}
        return {}


def build_vision_electra(config):
    model_type_g = config.GENERATOR.TYPE
    model_type_d = config.DISCRIMINATOR.TYPE
    if model_type_g == 'swin':
        generator = SwinTransformerForGenerator(
            img_size=config.DATA.IMG_SIZE,
            patch_size=config.GENERATOR.SWIN.PATCH_SIZE,
            in_chans=config.GENERATOR.SWIN.IN_CHANS,
            num_classes=0,
            embed_dim=config.GENERATOR.SWIN.EMBED_DIM,
            depths=config.GENERATOR.SWIN.DEPTHS,
            num_heads=config.GENERATOR.SWIN.NUM_HEADS,
            window_size=config.GENERATOR.SWIN.WINDOW_SIZE,
            mlp_ratio=config.GENERATOR.SWIN.MLP_RATIO,
            qkv_bias=config.GENERATOR.SWIN.QKV_BIAS,
            qk_scale=config.GENERATOR.SWIN.QK_SCALE,
            drop_rate=config.GENERATOR.DROP_RATE,
            drop_path_rate=config.GENERATOR.DROP_PATH_RATE,
            ape=config.GENERATOR.SWIN.APE,
            patch_norm=config.GENERATOR.SWIN.PATCH_NORM,
            use_checkpoint=config.TRAIN.USE_CHECKPOINT)
        generator_stride = 32
    elif model_type_g == 'vit':
        generator = VisionTransformerForSimMIM(
            img_size=config.DATA.IMG_SIZE,
            patch_size=config.GENERATOR.VIT.PATCH_SIZE,
            in_chans=config.GENERATOR.VIT.IN_CHANS,
            num_classes=0,
            embed_dim=config.GENERATOR.VIT.EMBED_DIM,
            depth=config.GENERATOR.VIT.DEPTH,
            num_heads=config.GENERATOR.VIT.NUM_HEADS,
            mlp_ratio=config.GENERATOR.VIT.MLP_RATIO,
            qkv_bias=config.GENERATOR.VIT.QKV_BIAS,
            drop_rate=config.GENERATOR.DROP_RATE,
            drop_path_rate=config.GENERATOR.DROP_PATH_RATE,
            norm_layer=partial(nn.LayerNorm, eps=1e-6),
            init_values=config.GENERATOR.VIT.INIT_VALUES,
            use_abs_pos_emb=config.GENERATOR.VIT.USE_APE,
            use_rel_pos_bias=config.GENERATOR.VIT.USE_RPB,
            use_shared_rel_pos_bias=config.GENERATOR.VIT.USE_SHARED_RPB,
            use_mean_pooling=config.GENERATOR.VIT.USE_MEAN_POOLING)
        generator_stride = 16
    elif model_type_g == 'mae':
        generator = MaskedAutoencoderViT(
            img_size=config.DATA.IMG_SIZE, 
            in_chans=config.GENERATOR.MAE.IN_CHANS,
            patch_size=config.GENERATOR.MAE.PATCH_SIZE, 
            embed_dim=config.GENERATOR.MAE.EMBED_DIM, 
            depth=config.GENERATOR.MAE.DEPTH, 
            num_heads=config.GENERATOR.MAE.NUM_HEADS,
            decoder_embed_dim=config.GENERATOR.MAE.DECODER_EMBED_DIM, 
            decoder_depth=config.GENERATOR.MAE.DECODER_DEPTH, 
            decoder_num_heads=config.GENERATOR.MAE.DECODER_NUM_HEADS,
            mlp_ratio=config.GENERATOR.MAE.MLP_RATIO, 
            norm_pix_loss=config.GENERATOR.MAE.NORM_PIX_LOSS,
            norm_layer=partial(nn.LayerNorm, eps=1e-6),)
    else:
        raise NotImplementedError(f"Unknown pre-train model: {model_type_g}")

    if model_type_d == 'swin':
        discriminator = SwinTransformerForDiscriminator(
            img_size=config.DATA.IMG_SIZE,
            patch_size=config.DISCRIMINATOR.SWIN.PATCH_SIZE,
            in_chans=config.DISCRIMINATOR.SWIN.IN_CHANS,
            num_classes=0,
            embed_dim=config.DISCRIMINATOR.SWIN.EMBED_DIM,
            depths=config.DISCRIMINATOR.SWIN.DEPTHS,
            num_heads=config.DISCRIMINATOR.SWIN.NUM_HEADS,
            window_size=config.DISCRIMINATOR.SWIN.WINDOW_SIZE,
            mlp_ratio=config.DISCRIMINATOR.SWIN.MLP_RATIO,
            qkv_bias=config.DISCRIMINATOR.SWIN.QKV_BIAS,
            qk_scale=config.DISCRIMINATOR.SWIN.QK_SCALE,
            drop_rate=config.DISCRIMINATOR.DROP_RATE,
            drop_path_rate=config.DISCRIMINATOR.DROP_PATH_RATE,
            ape=config.DISCRIMINATOR.SWIN.APE,
            patch_norm=config.DISCRIMINATOR.SWIN.PATCH_NORM,
            use_checkpoint=config.TRAIN.USE_CHECKPOINT)

    elif model_type_d == 'vit':
        discriminator = VisionTransformerForDiscriminator(
            img_size=config.DATA.IMG_SIZE,
            patch_size=config.DISCRIMINATOR.VIT.PATCH_SIZE,
            in_chans=config.DISCRIMINATOR.VIT.IN_CHANS,
            num_classes=0,
            embed_dim=config.DISCRIMINATOR.VIT.EMBED_DIM,
            depth=config.DISCRIMINATOR.VIT.DEPTH,
            num_heads=config.DISCRIMINATOR.VIT.NUM_HEADS,
            mlp_ratio=config.DISCRIMINATOR.VIT.MLP_RATIO,
            qkv_bias=config.DISCRIMINATOR.VIT.QKV_BIAS,
            drop_rate=config.DISCRIMINATOR.DROP_RATE,
            drop_path_rate=config.DISCRIMINATOR.DROP_PATH_RATE,
            norm_layer=partial(nn.LayerNorm, eps=1e-6),
            init_values=config.DISCRIMINATOR.VIT.INIT_VALUES,
            use_abs_pos_emb=config.DISCRIMINATOR.VIT.USE_APE,
            use_rel_pos_bias=config.DISCRIMINATOR.VIT.USE_RPB,
            use_shared_rel_pos_bias=config.DISCRIMINATOR.VIT.USE_SHARED_RPB,
            use_mean_pooling=config.DISCRIMINATOR.VIT.USE_MEAN_POOLING)
    else:
        raise NotImplementedError(f"Unknown pre-train model: {model_type_d}")

    if model_type_g == 'mae':
        generator = generator
    else:
        generator = SimMIM(generator, generator_stride)
    discriminator = Vision_Electra_Discriminator(discriminator=discriminator)

    return generator, discriminator