import random
import time
from copy import deepcopy
import numpy as np
import torch
from llama_architecture import LlamaForCausalLM, PPDLlamaModel, PPDLlamaForCausalLM
from transformers import AutoTokenizer
import os
import torch.distributed as dist
import torch.multiprocessing as mp
import argparse


#############################################

dimension_size = 5120

class SharedVariables:
    def __init__(self, model, tokenizer, input_sequence, model_path, max_new_tokens, num_processes, prompt_size):
        self.model = model
        self.tokenizer = tokenizer
        self.input_sequence = input_sequence
        self.model_path = model_path
        self.max_new_tokens = max_new_tokens
        self.num_processes = num_processes
        self.prompt_size = prompt_size


#########################################################

def autogressive_generation(rank, shared_variables, device) :
    
    max_new_tokens = shared_variables.max_new_tokens

    model = shared_variables.model
    model.to(device)

    num_of_proc = shared_variables.num_processes
    prompt_size = shared_variables.prompt_size

    #########################################################
    position = 0
    #########################################################

    if rank == 0: # Main process
    
        flag = 0
        match_tokens = 0
    
        input_sequence = shared_variables.input_sequence.to(device)

        first_hidden_state = [ torch.zeros([1, prompt_size, dimension_size], dtype=torch.float16, device=device) for idx in range(num_of_proc + 1)]
        second_hidden_state = [ torch.zeros([1, prompt_size, dimension_size], dtype=torch.float16, device=device) for idx in range(num_of_proc + 1)]

        ############################################################
        # Warm-up
        with torch.no_grad():   
            outputs = model.model(
                input_ids=input_sequence,
                output_hidden_states=True,
                use_cache=False,
                return_dict=True,
                first_half=True
            )
        ############################################################

        torch.cuda.synchronize()
        start_time = time.time()
        
        with torch.no_grad(): 
            while position < max_new_tokens :

                # 1. First forward in main
                if flag == 0:
                    outputs = model.model(
                        input_ids=input_sequence,
                        output_hidden_states=True,
                        use_cache=False,
                        return_dict=True,
                        first_half=True
                    )
                    first_hidden_state[0] = outputs.last_hidden_state
                    second_hidden_state[0] = outputs.hidden_states[-1]

                if position != (max_new_tokens - 1) :
                    logits = model.lm_head(first_hidden_state[0]).float()[:, -1, :]
                    predicted_tokens = logits.topk(num_of_proc).indices
                    predicted_tokens = predicted_tokens.transpose(0, 1).unsqueeze(-1)
                    
                    ############################################################################################
                    next_sequence = input_sequence.unsqueeze(1).repeat(num_of_proc, 1, 1)
                    next_sequence = torch.cat( (next_sequence, predicted_tokens), dim=2).to(dtype=torch.int32)
                    ############################################################################################

                    for i in range(num_of_proc):
                        dist.send(tensor=next_sequence[i], dst=i+1)

                outputs = model.model(
                        inputs_embeds=second_hidden_state[0],
                        output_hidden_states=False,
                        return_dict=True,
                        use_cache=False,
                        first_half=False
                    )

                next_token_logits = model.lm_head(outputs[0]).float()[:, -1, :]
                next_token = torch.argmax(next_token_logits, dim=-1).unsqueeze(-1)
                input_sequence = torch.cat([input_sequence, next_token], dim=-1)


                ################################################################
                if position != (max_new_tokens - 1) :
                    for src_rank in range(1, num_of_proc + 1) :
                        first_temp_tensor = torch.zeros([1, prompt_size + position + 1, dimension_size], dtype=torch.float16, device=device)
                        second_temp_tensor = torch.zeros([1, prompt_size + position + 1, dimension_size], dtype=torch.float16, device=device)

                        dist.recv(tensor=first_temp_tensor, src=src_rank)
                        dist.recv(tensor=second_temp_tensor, src=src_rank)

                        first_hidden_state[src_rank] = first_temp_tensor
                        second_hidden_state[src_rank] = second_temp_tensor

                ################################################################

                    flag = 0

                    matched_indices = (predicted_tokens == next_token).nonzero()
                    if matched_indices.numel() > 0 :
                        flag = 1
                        match_tokens = match_tokens + 1
                        target_index = matched_indices[0,0].item()

                        first_hidden_state[0] = first_hidden_state[target_index + 1]
                        second_hidden_state[0] = second_hidden_state[target_index + 1]

                position = position + 1

        torch.cuda.synchronize()
        finish_time = time.time()

        print("# Geneated Result #")
        generated_output = input_sequence.squeeze().tolist()
        response = shared_variables.tokenizer.decode(generated_output[prompt_size:], skip_special_tokens=True)
        print(response)
        print("Match Rate: ", match_tokens, "/", max_new_tokens - 1)
        print("Generated Throughput: ", max_new_tokens / (finish_time - start_time), " tokens/s" )
        print("Latency: ", (finish_time - start_time) )

    ##################################################################
    else :  # Child processes
        with torch.no_grad(): 
            while position < (max_new_tokens - 1) :

                next_sequence = torch.zeros([1, prompt_size + position + 1], dtype=torch.int32, device=device)
                dist.recv(next_sequence, src=0)

                outputs = model.model(
                    input_ids= next_sequence,
                    output_hidden_states=True,
                    return_dict=True,
                    use_cache=False,
                    first_half=True
                )

                ################################################################
                dist.send(tensor=outputs.last_hidden_state, dst=0)
                dist.send(tensor=outputs.hidden_states[-1], dst=0)
                ################################################################

                position = position + 1

        print("Rank", rank, " finished!")


################################################################################################


def init_process(rank, size, shared_variables, fn, backend='nccl'):
    os.environ['MASTER_ADDR'] = 'localhost'
    os.environ['MASTER_PORT'] = '29500'
    dist.init_process_group(backend, rank=rank, world_size=size)
    fn(rank, shared_variables, device=torch.device('cuda', rank))  # Use a separate GPU for each process


def add_fit_args(parser):    
    parser.add_argument('--num_of_process', type = int, default=3, help = 'number of process')

    args = parser.parse_args()
    return args

##########################################################################################

if __name__ == "__main__" :

    args = add_fit_args(argparse.ArgumentParser()) # args.num_of_process

    mp.set_start_method('spawn')

    model_path = "meta-llama/Llama-2-13b-chat-hf"
    model_max_length = 2048
    max_new_tokens = 128
    random_seed = 42

    torch.manual_seed(random_seed)
    torch.cuda.manual_seed(random_seed)
    torch.cuda.manual_seed_all(random_seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    np.random.seed(random_seed)
    random.seed(random_seed)

    ################################################################################################

    model = PPDLlamaForCausalLM.from_pretrained(model_path, torch_dtype=torch.float16)
    intermediate_layer = int(model.model.config.num_hidden_layers / 2)        
    model.model.set_intermediate_layer( intermediate_layer )
    model.eval()

    ################################################################################################

    tokenizer = AutoTokenizer.from_pretrained(
        model_path,
        model_max_length=model_max_length,
        padding_side="right",
        use_fast=False,
    )

    ################################################################################################

    prompt = f"""[INST] <<SYS>>
You are a helpful, respectful and honest assistant. Always answer as helpfully as possible, while being safe. Your answers should not include any harmful, unethical, racist, sexist, toxic, dangerous, or illegal content. Please ensure that your responses are socially unbiased and positive in nature.
<</SYS>>
Summarize the following article:
'''(CNN)The Palestinian Authority officially became the 123rd member of the International Criminal Court on Wednesday, a step that gives the court jurisdiction over alleged crimes in Palestinian territories. The formal accession was marked with a ceremony at The Hague, in the Netherlands, where the court is based. The Palestinians signed the ICC's founding Rome Statute in January, when they also accepted its jurisdiction over alleged crimes committed "in the occupied Palestinian territory, including East Jerusalem, since June 13, 2014." Later that month, the ICC opened a preliminary examination into the situation in Palestinian territories, paving the way for possible war crimes investigations against Israelis. As members of the court, Palestinians may be subject to counter-charges as well. Israel and the United States, neither of which is an ICC member, opposed the Palestinians' efforts to join the body. But Palestinian Foreign Minister Riad al-Malki, speaking at Wednesday's ceremony, said it was a move toward greater justice. "As Palestine formally becomes a State Party to the Rome Statute today, the world is also a step closer to ending a long era of impunity and injustice," he said, according to an ICC news release. "Indeed, today brings us closer to our shared goals of justice and peace." Judge Kuniko Ozaki, a vice president of the ICC, said acceding to the treaty was just the first step for the Palestinians. "As the Rome Statute today enters into force for the State of Palestine, Palestine acquires all the rights as well as responsibilities that come with being a State Party to the Statute. These are substantive commitments, which cannot be taken lightly," she said. Rights group Human Rights Watch welcomed the development. "Governments seeking to penalize Palestine for joining the ICC should immediately end their pressure, and countries that support universal acceptance of the court's treaty should speak out to welcome its membership," said Balkees Jarrah, international justice counsel for the group. "What's objectionable is the attempts to undermine international justice, not Palestine's decision to join a treaty to which over 100 countries around the world are members." In January, when the preliminary ICC examination was opened, Israeli Prime Minister Benjamin Netanyahu described it as an outrage, saying the court was overstepping its boundaries. The United States also said it "strongly" disagreed with the court's decision. "As we have said repeatedly, we do not believe that Palestine is a state and therefore we do not believe that it is eligible to join the ICC," the State Department said in a statement. It urged the warring sides to resolve their differences through direct negotiations. "We will continue to oppose actions against Israel at the ICC as counterproductive to the cause of peace," it said. But the ICC begs to differ with the definition of a state for its purposes and refers to the territories as "Palestine." While a preliminary examination is not a formal investigation, it allows the court to review evidence and determine whether to investigate suspects on both sides. Prosecutor Fatou Bensouda said her office would "conduct its analysis in full independence and impartiality." The war between Israel and Hamas militants in Gaza last summer left more than 2,000 people dead. The inquiry will include alleged war crimes committed since June. The International Criminal Court was set up in 2002 to prosecute genocide, crimes against humanity and war crimes. CNN's Vasco Cotovio, Kareem Khadder and Faith Karimi contributed to this report.'''[/INST]"""


    batch = tokenizer(prompt, return_tensors="pt")
    prompt_size = len(batch['input_ids'][0])

    # Class initialization
    shared_variables = SharedVariables(
        model=model,
        tokenizer=tokenizer, 
        input_sequence=batch['input_ids'], 
        model_path=model_path, 
        max_new_tokens=max_new_tokens, 
        num_processes=args.num_of_process,
        prompt_size=prompt_size
    )

    ################################################################################################

    size = args.num_of_process + 1
    processes = []

    for rank in range(size):
        p = mp.Process(target=init_process, args=(rank, size, shared_variables, autogressive_generation))
        p.start()
        processes.append(p)

    for p in processes:
        p.join()

    ################################################################################################