from llama_architecture import LlamaForCausalLM
import torch
from transformers import AutoTokenizer
import random
import numpy as np
import time


def forward(model, model_input):

        outputs = model.model(**model_input)

        hidden_states = outputs[0]
        logits = model.lm_head(hidden_states)
        logits = logits.float()

        return ( logits, outputs.past_key_values )


def greedy_decode(model, input_seq, max_length):

    with torch.no_grad():  # Disable gradient calculations for inference

        for position in range(max_length):
            model_input = model.prepare_inputs_for_generation(input_seq)
            outputs = forward(model, model_input)
            logits = outputs[0]

            # Get the logits of the last token
            next_token_logits = logits[:, -1, :]

            # Greedily select the next token
            next_token = torch.argmax(next_token_logits, dim=-1).unsqueeze(-1)

            # Append the predicted token to the input sequence
            input_seq = torch.cat([input_seq, next_token], dim=-1)

            # # Stop if the end of sequence token is generated
            # if next_token.item() == model.config.eos_token_id:
            #     break

    return input_seq.squeeze().tolist(), position


########################################################################################################################


if __name__ == "__main__" :

    random_seed = 42
    
    torch.manual_seed(random_seed)
    torch.cuda.manual_seed(random_seed)
    torch.cuda.manual_seed_all(random_seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    np.random.seed(random_seed)
    random.seed(random_seed)


    model_path = "meta-llama/Llama-2-13b-chat-hf"
    model_max_length = 2048
    max_new_tokens = 128 # 128

    device = torch.device('cuda:0')

    model = LlamaForCausalLM.from_pretrained(model_path, torch_dtype=torch.float16)
    model.to(device)
    model.eval() 

    #########################################################################################################

    tokenizer = AutoTokenizer.from_pretrained(
        model_path,
        model_max_length=model_max_length,
        padding_side="right",
        use_fast=False,
    )

    #########################################################################################################

    prompt = f"""[INST] <<SYS>>
You are a helpful, respectful and honest assistant. Always answer as helpfully as possible, while being safe. Your answers should not include any harmful, unethical, racist, sexist, toxic, dangerous, or illegal content. Please ensure that your responses are socially unbiased and positive in nature.
<</SYS>>
Summarize the following article:
'''(CNN)The Palestinian Authority officially became the 123rd member of the International Criminal Court on Wednesday, a step that gives the court jurisdiction over alleged crimes in Palestinian territories. The formal accession was marked with a ceremony at The Hague, in the Netherlands, where the court is based. The Palestinians signed the ICC's founding Rome Statute in January, when they also accepted its jurisdiction over alleged crimes committed "in the occupied Palestinian territory, including East Jerusalem, since June 13, 2014." Later that month, the ICC opened a preliminary examination into the situation in Palestinian territories, paving the way for possible war crimes investigations against Israelis. As members of the court, Palestinians may be subject to counter-charges as well. Israel and the United States, neither of which is an ICC member, opposed the Palestinians' efforts to join the body. But Palestinian Foreign Minister Riad al-Malki, speaking at Wednesday's ceremony, said it was a move toward greater justice. "As Palestine formally becomes a State Party to the Rome Statute today, the world is also a step closer to ending a long era of impunity and injustice," he said, according to an ICC news release. "Indeed, today brings us closer to our shared goals of justice and peace." Judge Kuniko Ozaki, a vice president of the ICC, said acceding to the treaty was just the first step for the Palestinians. "As the Rome Statute today enters into force for the State of Palestine, Palestine acquires all the rights as well as responsibilities that come with being a State Party to the Statute. These are substantive commitments, which cannot be taken lightly," she said. Rights group Human Rights Watch welcomed the development. "Governments seeking to penalize Palestine for joining the ICC should immediately end their pressure, and countries that support universal acceptance of the court's treaty should speak out to welcome its membership," said Balkees Jarrah, international justice counsel for the group. "What's objectionable is the attempts to undermine international justice, not Palestine's decision to join a treaty to which over 100 countries around the world are members." In January, when the preliminary ICC examination was opened, Israeli Prime Minister Benjamin Netanyahu described it as an outrage, saying the court was overstepping its boundaries. The United States also said it "strongly" disagreed with the court's decision. "As we have said repeatedly, we do not believe that Palestine is a state and therefore we do not believe that it is eligible to join the ICC," the State Department said in a statement. It urged the warring sides to resolve their differences through direct negotiations. "We will continue to oppose actions against Israel at the ICC as counterproductive to the cause of peace," it said. But the ICC begs to differ with the definition of a state for its purposes and refers to the territories as "Palestine." While a preliminary examination is not a formal investigation, it allows the court to review evidence and determine whether to investigate suspects on both sides. Prosecutor Fatou Bensouda said her office would "conduct its analysis in full independence and impartiality." The war between Israel and Hamas militants in Gaza last summer left more than 2,000 people dead. The inquiry will include alleged war crimes committed since June. The International Criminal Court was set up in 2002 to prosecute genocide, crimes against humanity and war crimes. CNN's Vasco Cotovio, Kareem Khadder and Faith Karimi contributed to this report.'''[/INST]"""



    batch = tokenizer(prompt, return_tensors="pt")
    prompt_size = len(batch['input_ids'][0])
    batch = {k: v.to(device) for k, v in batch.items()}

    
    input_sequence = batch['input_ids']

    # Warmup
    ########################################################################################
    dummy_output = model.model(input_ids=input_sequence)
    ########################################################################################

    torch.cuda.synchronize()
    start_time = time.time()
    generated_output, position = greedy_decode(model, input_sequence, max_length=max_new_tokens)
    torch.cuda.synchronize()
    finish_time = time.time()

    response = tokenizer.decode(generated_output[prompt_size:], skip_special_tokens=True)


    #########################################################################################################
    print("# Generated Output #")
    print( response )    
    print("Generated Throughput: ", max_new_tokens / (finish_time - start_time), " tokens/s" )
    print("Latency: ", (finish_time - start_time) )