import json
import torch

from base.base_dataset import BaseADDataset
from networks.main import build_network, build_autoencoder
from optim.dfdc_trainer import  DFDC_Trainer


class DFDC(object):
    """A class for the Deep Fair Discriminative Clustering method.

    Attributes:
        objective: A string specifying the objective.
        net_name: A string indicating the name of the neural network to use.
        net: The neural network.
        ae_net: The autoencoder network corresponding for network weights pretraining.
        trainer: DFDC_Trainer to train a DFDC model.
        optimizer_name: A string indicating the optimizer to use for training the DFDC network.
        ae_trainer: AETrainer to train an autoencoder in pretraining.
        ae_optimizer_name: A string indicating the optimizer to use for pretraining the autoencoder.
        results: A dictionary to save the results.
    """

    def __init__(self, dataset_name, s_weight, objective: str = 'dfdc'):
        """Inits DFDC with one of the two objectives."""

        self.objective = objective
        self.dataset_name = dataset_name
        self.net_name = None
        self.net = None  # neural network \phi

        self.trainer = None
        self.optimizer_name = None

        self.ae_net = None  # autoencoder network for pretraining
        self.ae_trainer = None
        self.ae_optimizer_name = None
        self.supervised_weight = s_weight
        self.results = {
            'train_time': None,
            'test_auc': None,
            'test_time': None,
            'test_scores': None,
        }

    def set_network(self, net_name):
        """Builds the neural network \phi."""
        self.net_name = net_name
        self.net = build_network(net_name)

    def generate_list(self, address, dataset: BaseADDataset, optimizer_name: str = 'adam', lr: float = 0.001, n_epochs: int = 50,
              lr_milestones: tuple = (), batch_size: int = 128, weight_decay: float = 1e-6, device: str = 'cuda',
              n_jobs_dataloader: int = 0):
        """Trains the model on the training data."""

        self.optimizer_name = optimizer_name
        self.trainer = DFDC_Trainer(self.dataset_name, self.objective, self.supervised_weight, optimizer_name, lr=lr,
                                       n_epochs=n_epochs, lr_milestones=lr_milestones, batch_size=batch_size,
                                       weight_decay=weight_decay, device=device, n_jobs_dataloader=n_jobs_dataloader)
        # Get the model
        self.trainer.calculate_nearest_dist(dataset, address)


    def train(self, address, dataset: BaseADDataset, optimizer_name: str = 'adam', lr: float = 0.001, n_epochs: int = 50,
              lr_milestones: tuple = (), batch_size: int = 128, weight_decay: float = 1e-6, device: str = 'cuda',
              n_jobs_dataloader: int = 0):
        """Trains the model on the training data."""

        self.optimizer_name = optimizer_name
        self.trainer = DFDC_Trainer(self.dataset_name, self.objective,self.supervised_weight, optimizer_name, lr=lr,
                                       n_epochs=n_epochs, lr_milestones=lr_milestones, batch_size=batch_size,
                                       weight_decay=weight_decay, device=device, n_jobs_dataloader=n_jobs_dataloader)
        # Get the model
        #self.trainer.calculate()
        self.net = self.trainer.train(dataset, self.net, address)
        self.results['train_time'] = self.trainer.train_time

    def test(self, dataset: BaseADDataset, device: str = 'cuda', n_jobs_dataloader: int = 0):
        """Tests the model on the test data."""

        if self.trainer is None:
            self.trainer = DFDC_Trainer(self.objective, self.supervised_weight, device=device, n_jobs_dataloader=n_jobs_dataloader)

        self.trainer.test(dataset, self.net)
        # Get results
        self.results['test_time'] = self.trainer.test_time
        self.results['test_scores'] = self.trainer.test_scores

    def save_model(self, export_model, save_ae=True):
        """Save DFDC model to export_model."""

        net_dict = self.net.state_dict()
        ae_net_dict = self.ae_net.state_dict() if save_ae else None

        torch.save({'net_dict': net_dict,
                    'ae_net_dict': ae_net_dict}, export_model)

    def load_model(self, model_path, load_ae=False):
        """Load DFDC model from model_path."""

        model_dict = torch.load(model_path)

        self.net.load_state_dict(model_dict['net_dict'])

    def save_results(self, export_json):
        """Save results dict to a JSON-file."""
        with open(export_json, 'w') as fp:
            json.dump(self.results, fp)
