import numpy as np


def line_line_intersection(
    p1: np.ndarray, direction_1: np.ndarray, p2: np.ndarray, direction_2: np.ndarray
) -> np.ndarray:
    """Compute the intersection point of two lines defined by a point and a direction."""
    d = np.array([[-direction_1[1], direction_1[0]], [-direction_2[1], direction_2[0]]])
    b = p2 - p1
    return p1 + np.linalg.inv(d) @ b


def distance_point_line_segment(p: np.ndarray, l1: np.ndarray, l2: np.ndarray) -> float:
    """Compute the distance of a point to a line segment."""
    diff = l2 - l1
    length = np.linalg.norm(diff)

    if length == 0:
        return np.linalg.norm(l1 - p)

    # Consider the line extending the segment
    t = np.clip(np.dot(p - l1, diff) / length, 0, 1)

    # Comput the projection of p onto the line
    projection = l1 + t * diff

    return np.linalg.norm(projection - p)


def rectangle_contains_point(rect: np.ndarray, p: np.ndarray) -> bool:
    """Check if a point is inside a rectangle."""
    for i in range(4):
        l1 = rect[i]
        l2 = rect[(i + 1) % 4]

        if np.cross(l2 - l1, p - l1) > 0:
            return False

    return True


def circle_contains_point(
    circle_pos: np.ndarray, circle_radius: float, p: np.ndarray
) -> bool:
    """Check if a point is inside a circle."""
    return np.linalg.norm(circle_pos - p) <= circle_radius


def order_vertices_clockwise(vertices: np.ndarray) -> np.ndarray:
    """Order the vertices of a polygon in a clockwise manner."""
    center = np.mean(vertices, axis=0)
    angles = np.arctan2(vertices[:, 1] - center[1], vertices[:, 0] - center[0])
    return vertices[np.argsort(-angles)]


def get_boundary_distance_from_line(
    p: np.ndarray, d: np.ndarray, boundary_size: float
) -> float:
    """
    For the line p + d * a, compute a for which the line intersects the
    boundary box of size boundary_size.
    """
    a_0 = np.abs((boundary_size - p[0]) / d[0])
    a_1 = np.abs((boundary_size - p[1]) / d[1])

    # if p[0] + d[0] >= 0:
    #     a_0 = (boundary_size - p[0]) / d[0]
    # else:
    #     a_0 = -(boundary_size + p[0]) / d[0]

    # if p[1] + d[1] >= 0:
    #     a_1 = (boundary_size - p[1]) / d[1]
    # else:
    #     a_1 = -(boundary_size + p[1]) / d[1]

    return min(a_0, a_1)


if __name__ == "__main__":
    # Test the function distance_point_line_segment
    l1 = np.array([0, 1])
    l2 = np.array([1, 1])

    p = np.array([0, 0])
    assert distance_point_line_segment(p, l1, l2) == 1

    p = np.array([-1, 0])
    assert distance_point_line_segment(p, l1, l2) == np.sqrt(2)

    p = np.array([2, 2])
    assert distance_point_line_segment(p, l1, l2) == np.sqrt(2)

    p = np.array([0.5, 2])
    assert distance_point_line_segment(p, l1, l2) == 1

    p = np.array([2, 3])
    assert distance_point_line_segment(p, l1, l2) == np.linalg.norm(p - l2)

    # Test the function rectangle_contains_point
    rect = np.array([[0, 1], [1, 2], [2, 1], [1, 0]])
    p = np.array([1, 1])
    assert rectangle_contains_point(rect, p)

    p = np.array([0, 0])
    assert not rectangle_contains_point(rect, p)

    p = np.array([2, 2])
    assert not rectangle_contains_point(rect, p)

    p = np.array([-0.5, 0.5])
    assert not rectangle_contains_point(rect, p)

    p = np.array([1, 0])
    assert rectangle_contains_point(rect, p)

    p = np.array([0, 1])
    assert rectangle_contains_point(rect, p)

    p = np.array([0.5, 0.5])
    assert rectangle_contains_point(rect, p)
