import argparse
import torch
import sys
import numpy as np
import os
import yaml
import torchvision

from torch.utils.data import DataLoader
import torchvision.transforms as transforms
from torchvision import datasets
from torch.utils.tensorboard import SummaryWriter

from utils import ModelEMA, ModelAverage


parser = argparse.ArgumentParser(description='PyTorch SimCLR lincls')
parser.add_argument('--optim', type=str, default='adam')
parser.add_argument('--gpu-index', default=0, type=int, help='Gpu index.')
parser.add_argument('--lr', '--learning-rate', default=0.0003, type=float,
                    metavar='LR', help='initial learning rate', dest='lr')
parser.add_argument('--wd', default=0.0, type=float, help='weight decay (default: 1e-4)')
args = parser.parse_args()


workdir = os.path.join('outputs', args.optim)
writer = SummaryWriter(workdir)

device = f'cuda:{args.gpu_index}' if torch.cuda.is_available() else 'cpu'
print("Using device:", device)

def get_stl10_data_loaders(download, shuffle=False, batch_size=256):
    train_dataset = datasets.STL10('./datasets', split='train', download=download,
                                  transform=transforms.Compose([
                                      transforms.ToTensor(),
                                  ]))

    train_loader = DataLoader(train_dataset, batch_size=batch_size,
                            num_workers=4, drop_last=False, shuffle=shuffle,
                            persistent_workers=True)

    test_dataset = datasets.STL10('./datasets', split='test', download=download,
                                  transform=transforms.Compose([
                                      transforms.ToTensor(),
                                  ]))

    test_loader = DataLoader(test_dataset, batch_size=2*batch_size,
                           num_workers=4, drop_last=False, shuffle=shuffle,
                           persistent_workers=True)
    return train_loader, test_loader

model = torchvision.models.resnet18(pretrained=False, num_classes=10).to(device)
model_ema = ModelEMA(model, 0.99)

checkpoint = torch.load(os.path.join(workdir, 'checkpoint_0200.pth.tar'), map_location=device)
state_dict = checkpoint['state_dict']

for k in list(state_dict.keys()):

    if k.startswith('backbone.'):
        if k.startswith('backbone') and not k.startswith('backbone.fc'):
            state_dict[k[len("backbone."):]] = state_dict[k]
    del state_dict[k]

log = model.load_state_dict(state_dict, strict=False)
assert log.missing_keys == ['fc.weight', 'fc.bias']

train_loader, test_loader = get_stl10_data_loaders(download=True)

# freeze all layers but the last fc
for name, param in model.named_parameters():
    if name not in ['fc.weight', 'fc.bias']:
        param.requires_grad = False

parameters = list(filter(lambda p: p.requires_grad, model.parameters()))
assert len(parameters) == 2  # fc.weight, fc.bias

optimizer = torch.optim.SGD(model.parameters(), lr=0.1, momentum=0.9)
criterion = torch.nn.CrossEntropyLoss().to(device)

def accuracy(output, target, topk=(1,)):
    """Computes the accuracy over the k top predictions for the specified values of k"""
    with torch.no_grad():
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.view(1, -1).expand_as(pred))

        res = []
        for k in topk:
            correct_k = correct[:k].reshape(-1).float().sum(0, keepdim=True)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res
    
with open(os.path.join(workdir, 'logistic_regression.log'), 'w') as f:
    epochs = 100
    for epoch in range(epochs):
        top1_train_accuracy = 0
        for counter, (x_batch, y_batch) in enumerate(train_loader):
            x_batch = x_batch.to(device)
            y_batch = y_batch.to(device)

            logits = model(x_batch)
            loss = criterion(logits, y_batch)
            top1 = accuracy(logits, y_batch, topk=(1,))
            top1_train_accuracy += top1[0]

            optimizer.zero_grad()
            loss.backward()
            optimizer.step()
            model_ema.update()

        top1_train_accuracy /= (counter + 1)
        top1_accuracy = 0
        top5_accuracy = 0
        model_ema.eval()
        for counter, (x_batch, y_batch) in enumerate(test_loader):
            x_batch = x_batch.to(device)
            y_batch = y_batch.to(device)

            # logits = model(x_batch)
            logits = model_ema(x_batch)

            top1, top5 = accuracy(logits, y_batch, topk=(1,5))
            top1_accuracy += top1[0]
            top5_accuracy += top5[0]
        model_ema.train()

        top1_accuracy /= (counter + 1)
        top5_accuracy /= (counter + 1)

        writer.add_scalar('lincls/top1', top1_accuracy, epoch)
        writer.add_scalar('lincls/top5', top5_accuracy, epoch)

        print_txt = f"Epoch {epoch}\tTop1 Train accuracy {top1_train_accuracy.item()}\tTop1 Test accuracy: {top1_accuracy.item()}\tTop5 test acc: {top5_accuracy.item()}"
        print(print_txt)
        f.write(print_txt + '\n')