import copy
import os
import torch
import torch.nn.functional as F
import optimizers
from utils import soft_update, hard_update
from model import GaussianPolicy, QNetwork, DeterministicPolicy


class EMA:
    def __init__(self, net, beta=0.9998):
        self.net = net
        self.ema = copy.deepcopy(net)
        self.beta = beta
        
    def update(self):
        net_state_dict = self.net.state_dict()
        ema_state_dict = self.ema.state_dict()
        keys = net_state_dict.keys() & ema_state_dict.keys()
        for k in keys:
            v_net = net_state_dict[k]
            v_ema = ema_state_dict[k]
            if v_ema.dtype is torch.long:
                v_ema.copy_(v_net)
            else:
                v_ema.mul_(self.beta).add_(v_net, alpha=1 - self.beta)
            
    def sample(self, state):
        return self.ema.sample(state)
    
    def train(self):
        self.ema.train()
    def eval(self):
        self.ema.eval()

class SAC(object):
    def __init__(self, num_inputs, action_space, args):
        optim_kwargs = {}
        if args.optim == 'sgd':
            optim_kwargs['lr'] = args.lr
            optim_kwargs['momentum'] = 0.9
            Opt = torch.optim.SGD
        elif args.optim == 'sps':
            Opt = optimizers.Sps
        elif args.optim == 'dog':
            Opt = optimizers.DoG
        elif args.optim == 'dasgd':
            optim_kwargs['momentum'] = 0.9
            Opt = optimizers.DAdaptSGD
        elif args.optim == 'adam':
            optim_kwargs['lr'] = args.lr
            Opt = torch.optim.Adam
        elif args.optim == 'cocob':
            Opt = optimizers.COCOB
        elif args.optim == 'ldog':
            Opt = optimizers.LDoG
        elif args.optim == 'daadam':
            Opt = optimizers.DAdaptAdam
        elif args.optim == 'prodigy':
            Opt = optimizers.Prodigy
        elif args.optim == 'pssps':
            Opt = optimizers.PSSps
        elif args.optim == 'psdasgd':
            Opt = optimizers.PSDASGD

        self.gamma = args.gamma
        self.tau = args.tau
        self.alpha = args.alpha

        self.policy_type = args.policy
        self.target_update_interval = args.target_update_interval
        self.automatic_entropy_tuning = args.automatic_entropy_tuning

        self.device = torch.device("cuda" if args.cuda else "cpu")

        to_optimize = {}
        self.critic = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(device=self.device)
        self.critic_optim = Opt(self.critic.parameters(), **optim_kwargs)
        to_optimize['critic'] = list(self.critic.parameters())

        self.critic_target = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(self.device)
        hard_update(self.critic_target, self.critic)

        if self.policy_type == "Gaussian":
            # Target Entropy = −dim(A) (e.g. , -6 for HalfCheetah-v2) as given in the paper
            if self.automatic_entropy_tuning is True:
                self.target_entropy = -torch.prod(torch.Tensor(action_space.shape).to(self.device)).item()
                self.log_alpha = torch.zeros(1, requires_grad=True, device=self.device)
                self.alpha_optim = Opt([self.log_alpha], **optim_kwargs)
                to_optimize['alpha'] = [self.log_alpha]

            self.policy = GaussianPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Opt(self.policy.parameters(), **optim_kwargs)
            to_optimize['policy'] = list(self.policy.parameters())

        else:
            self.alpha = 0
            self.automatic_entropy_tuning = False
            self.policy = DeterministicPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Opt(self.policy.parameters(), **optim_kwargs)
            to_optimize['policy'] = list(self.policy.parameters())
        self.ema_policy = EMA(self.policy)
        import copy
        to_optimize = copy.deepcopy(to_optimize)
        self.to_optimize = to_optimize
        self.optim = Opt(sum([v for v in to_optimize.values()], []), **optim_kwargs)
        self.args_optim = args.optim

    def select_action(self, state, evaluate=False, ema=False):
        state = torch.FloatTensor(state).to(self.device).unsqueeze(0)
        if evaluate is False:
            action, _, _ = self.policy.sample(state)
        elif ema is False:
            _, _, action = self.policy.sample(state)
        else:
            _, _, action = self.ema_policy.sample(state)
        return action.detach().cpu().numpy()[0]

    def update_parameters(self, memory, batch_size, updates):
        # Sample a batch from memory
        state_batch, action_batch, reward_batch, next_state_batch, mask_batch = memory.sample(batch_size=batch_size)

        state_batch = torch.FloatTensor(state_batch).to(self.device)
        next_state_batch = torch.FloatTensor(next_state_batch).to(self.device)
        action_batch = torch.FloatTensor(action_batch).to(self.device)
        reward_batch = torch.FloatTensor(reward_batch).to(self.device).unsqueeze(1)
        mask_batch = torch.FloatTensor(mask_batch).to(self.device).unsqueeze(1)

        with torch.no_grad():
            next_state_action, next_state_log_pi, _ = self.policy.sample(next_state_batch)
            qf1_next_target, qf2_next_target = self.critic_target(next_state_batch, next_state_action)
            min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - self.alpha * next_state_log_pi
            next_q_value = reward_batch + mask_batch * self.gamma * (min_qf_next_target)
        qf1, qf2 = self.critic(state_batch, action_batch)  # Two Q-functions to mitigate positive bias in the policy improvement step
        qf1_loss = F.mse_loss(qf1, next_q_value)  # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
        qf2_loss = F.mse_loss(qf2, next_q_value)  # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
        qf_loss = qf1_loss + qf2_loss

        self.optim.zero_grad()
        
        self.critic_optim.zero_grad()
        qf_loss.backward()
        # self.critic_optim.step()
        self.copy_grad(self.critic.parameters(), 'critic')

        pi, log_pi, _ = self.policy.sample(state_batch)

        qf1_pi, qf2_pi = self.critic(state_batch, pi)
        min_qf_pi = torch.min(qf1_pi, qf2_pi)

        policy_loss = ((self.alpha * log_pi) - min_qf_pi).mean() # Jπ = 𝔼st∼D,εt∼N[α * logπ(f(εt;st)|st) − Q(st,f(εt;st))]

        self.policy_optim.zero_grad()
        policy_loss.backward()
        # self.policy_optim.step()
        self.copy_grad(self.policy.parameters(), 'policy')
        self.ema_policy.update()

        if self.automatic_entropy_tuning:
            alpha_loss = -(self.log_alpha * (log_pi + self.target_entropy).detach()).mean()

            self.alpha_optim.zero_grad()
            alpha_loss.backward()
            # self.alpha_optim.step()
            self.copy_grad(self.log_alpha, 'alpha')

            self.alpha = self.log_alpha.exp()
            alpha_tlogs = self.alpha.clone() # For TensorboardX logs
        else:
            alpha_loss = torch.tensor(0.).to(self.device)
            alpha_tlogs = torch.tensor(self.alpha) # For TensorboardX logs


        if updates % self.target_update_interval == 0:
            soft_update(self.critic_target, self.critic, self.tau)

        self.optim.step()
        self.copy_data(self.critic.parameters(), 'critic')
        self.copy_data(self.policy.parameters(), 'policy')
        if self.automatic_entropy_tuning:
            self.copy_data(self.log_alpha, 'alpha')
        
        if self.args_optim in ['sps', 'pssps']:
            lr = self.optim.state['step_size']
        elif self.args_optim in ['dog', 'ldog']:
            lr = self.optim.param_groups[0]['eta'][0]
        elif self.args_optim == 'cocob':
            lr = 0.0
        elif self.args_optim in ['dasgd']:
            lr = self.optim.param_groups[0]['d'] * self.optim.param_groups[0]['lr'] / self.optim.param_groups[0]['g0_norm']
        elif self.args_optim in ['daadam', 'prodigy']:
            lr = self.optim.param_groups[0]['d'] * self.optim.param_groups[0]['lr']
        elif self.args_optim in ['psdasgd']:
            lr = self.optim.param_groups[0]['elr']
        else:
            lr = self.optim.param_groups[0]['lr']

        return qf1_loss.item(), qf2_loss.item(), policy_loss.item(), alpha_loss.item(), alpha_tlogs.item(), lr
    
    def copy_grad(self, param, tag):
        for p1, p2 in zip(self.to_optimize[tag], param):
            if p1.grad is None:
                p1.grad = p2.grad.clone()
            else:
                p1.grad.copy_(p2.grad)
            
    def copy_data(self, param, tag):
        for p1, p2 in zip(self.to_optimize[tag], param):
            p2.data.copy_(p1.data)

    # Save model parameters
    def save_checkpoint(self, env_name, suffix="", ckpt_path=None):
        if not os.path.exists('checkpoints/'):
            os.makedirs('checkpoints/')
        if ckpt_path is None:
            ckpt_path = "checkpoints/sac_checkpoint_{}_{}".format(env_name, suffix)
        print('Saving models to {}'.format(ckpt_path))
        torch.save({'policy_state_dict': self.policy.state_dict(),
                    'critic_state_dict': self.critic.state_dict(),
                    'critic_target_state_dict': self.critic_target.state_dict(),
                    'critic_optimizer_state_dict': self.critic_optim.state_dict(),
                    'policy_optimizer_state_dict': self.policy_optim.state_dict()}, ckpt_path)

    # Load model parameters
    def load_checkpoint(self, ckpt_path, evaluate=False):
        print('Loading models from {}'.format(ckpt_path))
        if ckpt_path is not None:
            checkpoint = torch.load(ckpt_path)
            self.policy.load_state_dict(checkpoint['policy_state_dict'])
            self.critic.load_state_dict(checkpoint['critic_state_dict'])
            self.critic_target.load_state_dict(checkpoint['critic_target_state_dict'])
            self.critic_optim.load_state_dict(checkpoint['critic_optimizer_state_dict'])
            self.policy_optim.load_state_dict(checkpoint['policy_optimizer_state_dict'])

            if evaluate:
                self.policy.eval()
                self.critic.eval()
                self.critic_target.eval()
            else:
                self.policy.train()
                self.critic.train()
                self.critic_target.train()