"""
Copyright (c) ANONYMOUS
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

from typing import Dict, Union

import haiku as hk
import jax
import jax.numpy as jnp
import optax
from flax import struct
from functools import partial

from ccoa.contribution.value import ValueFunction, ValueFunctionState

from .base import Contribution
from .features import RewardFeatures, RewardFeaturesState
from .dp_utils import get_idx


class CausalContributionBase(Contribution):
    def __init__(
        self,
        num_actions,
        obs_shape,
        return_type,
        hindsight_feature_type,
        reward_values,
        reward_clusters,
        g_trick,
        use_baseline,
    ):
        self.num_actions = num_actions
        self.obs_shape = obs_shape
        self.return_type = return_type
        self.hindsight_feature = hindsight_feature_type
        self.reward_values = jnp.array(reward_values)
        self.reward_clusters = reward_clusters # List of list of reward values. Each sublist will be considered a common reward feature.
        self.g_trick = g_trick
        self.use_baseline = use_baseline
        self.center = self.return_type == "advantage"
        assert not self.hindsight_feature == "state_based" or self.g_trick

    def __call__(self, state, trajectory):
        # This has shape (T, T, A), encoding the contribution of (s_t, s_t', a_t).
        # The diagonal is zero when using g_trick
        batched_coeffs = self.get_contribution_coeff(state=state, trajectory=trajectory)

        rewards_masked = trajectory.rewards * (1 - trajectory.dones[:-1])
        rewards_masked = jnp.expand_dims(rewards_masked, -1)
        advantages = jnp.sum(batched_coeffs * rewards_masked, axis=1)

        if self.g_trick:
            # When using the g_trick, the contribution s_t->s_t is not modeled, and the immediate reward not used.
            # We instead use the policy gradient contribution for the immediate reward.
            advantages = advantages + jnp.where(
                jax.nn.one_hot(trajectory.actions, self.num_actions),
                jnp.expand_dims(trajectory.rewards, -1) / jnp.exp(trajectory.logits),
                0,
            )

        if self.use_baseline:
            value = self.get_value(state, trajectory.observations)
            advantages = advantages - jnp.where(
                jax.nn.one_hot(trajectory.actions, self.num_actions),
                jnp.expand_dims(value, -1) * (1 / jnp.exp(trajectory.logits) - self.center),
                jnp.expand_dims(value, -1) * (-self.center),
            )

        return advantages

    def get_contribution_coeff(self, state, trajectory):
        """
        Computes the contribution coefficients (s_t, s_t', s_a) using the trajectory.
        The returned tensor has dimension (T,T,A) where T is the length of the trajectory.
        The diagonal (contribution s_t -> s_t) is zero when using the g_trick.
        """
        num_timesteps = trajectory.observations.shape[0]

        hindsight_objects = jax.vmap(self.get_hindsight_object, in_axes=(None, 0, 0, 0))(
            state, trajectory.observations, trajectory.actions, trajectory.rewards
        )

        batched_coeffs_norm = self.get_contribution(
            state, trajectory.observations, trajectory.logits, hindsight_objects
        )

        # If using g_trick, then do not return the contribution of s_t on (s_t, a_t).
        mask_tril = jnp.expand_dims(
            1 - jnp.tri(num_timesteps, num_timesteps, k=-1 if not self.g_trick else 0), axis=-1
        )

        if self.center:
            return (batched_coeffs_norm - 1.0) * mask_tril

        return batched_coeffs_norm * mask_tril

    def get_contribution(self, state, observations, logits, hindsight_objects):
        raise NotImplementedError("get_contribution not implemented")

    def get_groundtruth_contribution_coeff(self, state, trajectory, mdp, policy_prob):
        num_timesteps = trajectory.observations.shape[0]

        hindsight_objects = jax.vmap(self.get_hindsight_object, in_axes=(None, 0, 0, 0))(
            None, trajectory.observations, trajectory.actions, trajectory.rewards
        )
        batched_coeffs_norm = mdp.get_contribution(
            policy_prob,
            trajectory.observations,
            hindsight_objects,
            lambda o, a, r: self.get_hindsight_object(state, o, a, r),  # closed over state
            self.g_trick,
        )

        mask_tril = jnp.expand_dims(
            1 - jnp.tri(num_timesteps, num_timesteps, k=-1 if not self.g_trick else 0), -1
        )

        if self.center:
            return (batched_coeffs_norm - 1.0) * mask_tril

        return batched_coeffs_norm * mask_tril

    def expected_advantage(self, state, mdp, policy_prob):  # TODO(FIX)
        hindsight_objects = jax.vmap(
            jax.vmap(
                jax.vmap(self.get_hindsight_object, in_axes=(None, None, None, 0)),
                in_axes=(None, None, 0, None),
            ),
            in_axes=(None, 0, None, None),
        )(state, mdp.mdp_observation, jnp.arange(mdp.num_actions), mdp.mdp_reward_values)

        contribution = jax.vmap(
            jax.vmap(self.get_contribution, in_axes=(None, None, None, 0), out_axes=1),
            in_axes=(None, None, None, -2),
            out_axes=3,
        )(state, mdp.mdp_observation, jnp.log(policy_prob), hindsight_objects)

        if self.center:
            contribution = contribution - 1.0

        contribution = jax.vmap(
            jax.vmap(lambda a, b: (a * b).sum(-1), in_axes=(-1, None), out_axes=-1),
            in_axes=(0, None),
        )(contribution, mdp.mdp_reward_probs * mdp.mdp_reward_values)

        return mdp.get_advantage_from_hindsight(policy_prob, contribution, self.g_trick)

    def get_hindsight_object(self, state, observation, action, reward):
        """
        Returns the credit assignment feature associated with (s, a, r).
        NOTE: This is not batched, i.e. assumes a single observation, action and reward
        """
        if self.hindsight_feature == "reward_based":
            if self.reward_clusters is not None:
                for cluster in self.reward_clusters:
                    reward = jnp.where(jnp.isin(reward, jnp.array(cluster)),
                                       cluster[0],
                                       reward)
            index = jnp.nonzero(reward == self.reward_values, size=1)[0]
            return jax.nn.one_hot(index, len(self.reward_values)).squeeze()

        elif self.hindsight_feature == "reward_predictor":
            return self.get_feature(state, observation, action)

        elif self.hindsight_feature == "state_based":
            return observation

        else:
            raise ValueError

    def get_feature(self, state, observation, action):
        raise NotImplementedError

    def get_value(self, state, observations):
        raise NotImplementedError

@struct.dataclass
class CausalContributionState:
    features: Union[Dict, RewardFeaturesState]
    params: hk.Params
    optim: optax.OptState
    value: Union[Dict, ValueFunctionState]


class CausalContribution(CausalContributionBase):
    def __init__(
        self,
        num_actions,
        obs_shape,
        return_type,
        hindsight_feature_type,
        reward_values,
        reward_clusters,
        g_trick,
        use_baseline,
        hindsight_model,
        hindsight_optimizer,
        hindsight_steps,
        hindsight_loss_type,
        modulate_with_policy,
        contribution_clip,
        feature_module,
        value_module,
        balance_loss,
        mask_zero_reward_loss,
        clip_contrastive,
        max_grad_norm,
    ):
        super().__init__(
            num_actions,
            obs_shape,
            return_type,
            hindsight_feature_type,
            reward_values,
            reward_clusters,
            g_trick,
            use_baseline,
        )

        self.hindsight_model = hk.without_apply_rng(hk.transform(hindsight_model))
        if max_grad_norm is not None:
            assert max_grad_norm > 0
            # Optional gradient clipping
            self.hindsight_optimizer = optax.chain(
                optax.clip_by_global_norm(max_grad_norm), hindsight_optimizer
            )
        else:
            self.hindsight_optimizer = hindsight_optimizer
        self.hindsight_steps = hindsight_steps
        self.hindsight_loss_type = hindsight_loss_type
        self.modulate_with_policy = modulate_with_policy
        self.contribution_clip = contribution_clip
        self.feature_module = feature_module
        self.value_module = value_module
        self.balance_loss = balance_loss
        self.mask_zero_reward_loss = mask_zero_reward_loss
        self.clip_contrastive = clip_contrastive

        if self.modulate_with_policy:
            assert self.hindsight_loss_type == "hindsight"

        if self.hindsight_feature == "reward_predictor":
            assert isinstance(feature_module, RewardFeatures)

        if self.use_baseline:
            assert isinstance(value_module, ValueFunction)

    def reset(self, rng):
        rng_features, rng_hindsight, rng_value = jax.random.split(rng, 3)

        # Create placeholder observation
        placeholder_observation = jnp.zeros(self.obs_shape, dtype=jnp.float32)

        # Create placeholder hindsight object
        if self.hindsight_feature == "reward_based":
            features_state = dict()
            placeholder_hindsight_object = jnp.zeros(len(self.reward_values))
        elif self.hindsight_feature == "reward_predictor":
            dummy_action = 0
            dummy_state_feature_model = jnp.zeros(self.obs_shape, dtype=jnp.float32)
            features_state = self.feature_module.reset(rng_features, dummy_state_feature_model)
            placeholder_hindsight_object = self.feature_module(
                rng_features, features_state, dummy_state_feature_model, dummy_action
            )
        elif self.hindsight_feature == "state_based":
            features_state = dict()
            placeholder_hindsight_object = jnp.zeros(self.obs_shape, dtype=jnp.float32)
        else:
            raise ValueError

        # Initialise the hindsight model
        placeholder_logits = jnp.zeros((self.num_actions,))
        initial_hindsight_params = self.hindsight_model.init(
            rng_hindsight, placeholder_observation, placeholder_hindsight_object, placeholder_logits
        )

        # Initialise value model for computing the baseline
        if self.use_baseline:
            value_state = self.value_module.reset(rng_value, placeholder_observation)
        else:
            value_state = dict()

        initial_hindsight_opt_state = self.hindsight_optimizer.init(initial_hindsight_params)

        return CausalContributionState(
            features=features_state,
            params=initial_hindsight_params,
            optim=initial_hindsight_opt_state,
            value=value_state,
        )

    def get_contribution(self, state, observations, policy_logits, hindsight_objects):
        """compute the contribution coefficients"""

        logit_normed = jnp.expand_dims(policy_logits, axis=1)

        if not self.modulate_with_policy:
            logit_normed = 0 * logit_normed

        batched_coeffs = jax.vmap(
            jax.vmap(self.hindsight_model.apply, in_axes=(None, None, 0, None)), in_axes=(None, 0, None, 0)
        )(state.params, observations, hindsight_objects, policy_logits)

        if self.hindsight_loss_type == "hindsight":
            batched_coeffs = jax.nn.log_softmax(logit_normed + batched_coeffs, axis=-1)
            batched_coeffs_norm = jnp.exp(batched_coeffs - jnp.expand_dims(policy_logits, axis=1))

        if self.hindsight_loss_type == "contrastive":
            if self.clip_contrastive:
                lower = -jnp.expand_dims(policy_logits, axis=1) * jnp.ones_like(batched_coeffs)
                batched_coeffs_norm = jnp.exp(jnp.min(jnp.stack([batched_coeffs, lower]), axis=0))
            else:
                batched_coeffs_norm = jnp.exp(batched_coeffs)

        if self.contribution_clip is not None and self.contribution_clip > 0:
            batched_coeffs_norm = jnp.minimum(batched_coeffs_norm, self.contribution_clip)

        return batched_coeffs_norm

    def get_feature(self, state, observation, action):
        rng = jax.random.PRNGKey(0)
        return self.feature_module(rng, state.features, observation, action)

    def get_value(self, state, observations):
        return jax.vmap(self.value_module, in_axes=(None, 0))(state.value, observations).squeeze(-1)

    def update_feature(self, rng, state, offline_batch_sampler):
        if self.hindsight_feature == "reward_predictor": 
            state_features, metrics_features = self.feature_module.update(
                rng, state.features, offline_batch_sampler
            )

        else:
            state_features = state.features
            metrics_features=dict()
        
        return state.replace(features=state_features), metrics_features


    def update(self, rng, state, batch_sampler, offline_batch_sampler, act_fn):
        metrics = dict()

        rng_features, rng_hindsight, rng_value = jax.random.split(rng, 3)

        # Update feature network
        # Disabled as it is trained in the experiment runner directly
        state_features = state.features
        # if self.hindsight_feature == "reward_predictor": 
        #     state_features, metrics_features = self.feature_module.update(
        #         rng_features, state.features, offline_batch_sampler
        #     )
        #     metrics.update({k + "_features": metrics_features[k] for k in metrics_features})

        # else:
        #     state_features = state.features

        # Update value network
        if self.use_baseline:
            state_value, metrics_value = self.value_module.update(
                rng_value, state.value, batch_sampler
            )
            metrics.update({k + "_value": metrics_value[k] for k in metrics_value})
        else:
            state_value = state.value

        # Update hindsight network
        (params_hindsight, optim_hindsight), metric_hindsight = self._update_hindsight(
            rng=rng_hindsight, state=state, batch_sampler=batch_sampler
        )
        metrics.update({k + "_hindsight": metric_hindsight[k] for k in metric_hindsight})

        state = CausalContributionState(
            features=state_features,
            params=params_hindsight,
            optim=optim_hindsight,
            value=state_value,
        )

        return state, metrics

    def _update_hindsight(self, rng, state, batch_sampler):
        def mean_batch_loss_hindsight(
            params, observations, hindsight_objects, actions, logits, rewards
        ):
            batch_loss = jax.vmap(self._loss_hindsight, in_axes=(None, 0, 0, 0, 0, 0))(
                params, observations, hindsight_objects, actions, logits, rewards
            )
            return jnp.mean(batch_loss)

        def mean_batch_loss_contrastive(
            params, observations, hindsight_objects, actions, logits, rewards
        ):
            batch_loss = jax.vmap(self._loss_contrastive, in_axes=(None, 0, 0, 0, 0, 0))(
                params, observations, hindsight_objects, actions, logits, rewards
            )
            return jnp.mean(batch_loss)

        def update_step(carry, rng_t):
            params, optimizer_state = carry

            # Sample a batch of trajectories from the replay buffer
            batch_trajectory = batch_sampler(rng_t)

            # Obtain the hindsight objects
            observations = batch_trajectory.observations
            actions = batch_trajectory.actions
            logits = batch_trajectory.logits
            rewards = batch_trajectory.rewards

            get_hindsight_object_batched = jax.vmap(
                jax.vmap(self.get_hindsight_object, in_axes=(None, 0, 0, 0)),
                in_axes=(None, 0, 0, 0),
            )

            hindsight_objects = get_hindsight_object_batched(
                state,
                batch_trajectory.observations,
                batch_trajectory.actions,
                batch_trajectory.rewards,
            )

            # Compute loss
            if self.hindsight_loss_type == "hindsight":
                batch_loss, grads = jax.value_and_grad(mean_batch_loss_hindsight)(
                    params, observations, hindsight_objects, actions, logits, rewards
                )

            elif self.hindsight_loss_type == "contrastive":
                batch_loss, grads = jax.value_and_grad(mean_batch_loss_contrastive)(
                    params, observations, hindsight_objects, actions, logits, rewards
                )

            # Update params
            params_update, next_optimizer_state = self.hindsight_optimizer.update(
                grads, optimizer_state, params
            )
            next_params = optax.apply_updates(params, params_update)

            metrics = {"loss": batch_loss, "gradnorm": optax.global_norm(grads)}

            return [next_params, next_optimizer_state], metrics

        rng_init, rng_scan = jax.random.split(rng)

        carry, metrics = jax.lax.scan(
            f=update_step,
            init=[state.params, state.optim],
            xs=jax.random.split(rng_scan, self.hindsight_steps),
        )

        # Only select the last element from metrics.
        metrics_summary = {k + "_start": metrics[k][0] for k in metrics}
        metrics_summary.update({k + "_end": metrics[k][-1] for k in metrics})

        return carry, metrics_summary

    def _loss_hindsight(
        self, params, observations, hindsight_objects, actions, action_logits, rewards
    ):

        batched_contribution = jax.vmap(
            jax.vmap(self.hindsight_model.apply, in_axes=(None, 0, None, 0)),
            in_axes=(None, None, 0, None),
            out_axes=1,
        )
        logits = batched_contribution(params, observations, hindsight_objects, action_logits)

        if self.modulate_with_policy:
            logits = logits + jnp.expand_dims(action_logits, 1)

        loss = optax.softmax_cross_entropy_with_integer_labels(
            logits, jnp.expand_dims(actions, axis=1)
        )

        causal_mask = jnp.triu(jnp.ones_like(loss), k=0 if not self.g_trick else 1)

        reward_mask = jnp.ones((rewards.shape[0]))
        if self.balance_loss:
            # Only consider non-zero rewards
            reward_mask = jnp.where(rewards == 0.2,
                                    reward_mask,
                                    reward_mask/20)

        if self.mask_zero_reward_loss:
            # Only consider non-zero rewards
            reward_mask = jnp.where(rewards == 0,
                                    0,
                                    reward_mask)

        reward_mask = jnp.expand_dims(reward_mask, axis=0)
        normalizer = jnp.sum(causal_mask * reward_mask)
        normalizer = normalizer * (normalizer != 0) + (normalizer == 0)
        return jnp.sum(loss * causal_mask * reward_mask) / normalizer

    def _loss_contrastive(
        self, params, observations, hindsight_objects, actions, action_logits, rewards
    ):
        batched_discriminator = jax.vmap(
            jax.vmap(self.hindsight_model.apply, in_axes=(None, 0, None, 0)),
            in_axes=(None, None, 0, None),
            out_axes=1,
        )
        discriminator_logits = batched_discriminator(params, observations, hindsight_objects, action_logits)

        # if self.clip_contrastive:
        #     straight_through_clipper = straight_through(
        #         lambda x: jnp.min(jnp.stack([x, -jnp.expand_dims(action_logits, axis=1)
        #                                      * jnp.ones_like(x)]), axis=0))
        #     discriminator_logits = straight_through_clipper(discriminator_logits)

        log_sigmoid = jax.nn.log_sigmoid(discriminator_logits)
        actions_one_hot = jnp.expand_dims(jax.nn.one_hot(actions, self.num_actions), axis=1)
        loss = -(
            actions_one_hot * log_sigmoid
            + jnp.expand_dims(jnp.exp(action_logits), axis=1)
            * (-discriminator_logits + log_sigmoid)
        ).sum(axis=-1)

        if self.balance_loss:
            # Re-weigh the loss by the frequency of unique values to alleviate imbalance
            _, unique_inv, unique_counts = jnp.unique(
                rewards, return_inverse=True, return_counts=True, size=len(self.reward_values)
            )
            loss = loss / jnp.expand_dims((unique_counts / unique_counts.sum())[unique_inv], axis=1)

        # Causal masking of the loss
        causal_mask = jnp.triu(jnp.ones_like(loss), k=0 if not self.g_trick else 1)

        if self.mask_zero_reward_loss:
            # Only consider non-zero rewards
            reward_mask = jnp.expand_dims(rewards != 0, axis=0)

            # If there are no non-zero rewards in a trajectory, make sure we do not divide by zero
            normalizer = jnp.sum(causal_mask * reward_mask)
            normalizer = normalizer * (normalizer != 0) + (normalizer == 0)

            return jnp.sum(loss * causal_mask * reward_mask) / normalizer
        else:
            return jnp.sum(loss * causal_mask) / jnp.sum(causal_mask)


@struct.dataclass
class CausalContributionGTState:
    policy_prob: jnp.ndarray
    features: Union[Dict, RewardFeaturesState]


class CausalContributionGT(CausalContributionBase):
    def __init__(
        self,
        num_actions,
        obs_shape,
        return_type,
        hindsight_feature_type,
        reward_values,
        reward_clusters,
        g_trick,
        use_baseline,
        mdp,
        feature_module,
    ):
        super().__init__(
            num_actions,
            obs_shape,
            return_type,
            hindsight_feature_type,
            reward_values,
            reward_clusters,
            g_trick,
            use_baseline,
        )
        self.mdp = mdp
        self.feature_module = feature_module

    def reset(self, rng):
        # Create placeholder hindsight object
        features_state = dict()
        if self.hindsight_feature == "reward_predictor":
            dummy_state_feature_model = jnp.zeros(self.obs_shape, dtype=jnp.float32)
            features_state = self.feature_module.reset(rng, dummy_state_feature_model)

        policy_prob = jnp.ones((self.mdp.num_state, self.mdp.num_actions)) / self.mdp.num_actions

        return CausalContributionGTState(policy_prob=policy_prob, features=features_state)

    def get_feature(self, state, observation, action):
        rng = jax.random.PRNGKey(0)
        return self.feature_module(rng, state.features, observation, action)

    def get_reward(self, state, observations):
        return self.mdp.get_reward(state.policy_prob, observations)

    def get_value(self, state, observations):
        return self.mdp.get_value(state.policy_prob, observations)

    def get_contribution(self, state, observations, logits, hindsight_objects):
        return self.mdp.get_contribution(
            state.policy_prob,
            observations,
            hindsight_objects,
            lambda o, a, r: self.get_hindsight_object(state, o, a, r),  # closed over state,
            self.g_trick,
        )

    def update_feature(self, rng, state, offline_batch_sampler):
        if self.hindsight_feature == "reward_predictor": 
            state_features, metrics_features = self.feature_module.update(
                rng, state.features, offline_batch_sampler
            )

        else:
            state_features = state.features
            metrics_features=dict()
        
        return state.replace(features=state_features), metrics_features
    
    def update(self, rng, state, batch_sampler, offline_batch_sampler, act_fn):
        metrics = dict()
        # Update feature network
        # Disabled as it is trained in the experiment runner directly
        state_features = state.features
        # state_features=dict()
        # if self.hindsight_feature == "reward_predictor":
        #     state_features, metrics_features = self.feature_module.update(
        #         rng, state.features, offline_batch_sampler
        #     )
        #     metrics = ({k + "_features": metrics_features[k] for k in metrics_features})

        _, policy_logits = act_fn(self.mdp.mdp_observation, False)
        return CausalContributionGTState(policy_prob=jax.nn.softmax(policy_logits), features=state_features), metrics
