"""
Copyright (c) ANONYMOUS
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import haiku as hk
import jax
import jax.numpy as jnp
import optax
from flax import struct
import rlax

from .base import Contribution


class QNetContributionBase(Contribution):
    def __init__(self, num_actions, obs_shape, return_type):
        self.num_actions = num_actions
        self.obs_shape = obs_shape
        self.return_type = return_type

    def __call__(self, state, trajectory):
        done = trajectory.dones
        return_contribution = self._get_contribution(
            state, trajectory.observations, jax.nn.softmax(trajectory.logits)
        )
        mask = jnp.expand_dims(1 - done[:-1], axis=-1)
        return return_contribution * mask

    def _get_contribution(self, state, observations, policy_prob):
        return_contribution = self.get_action_value(state, observations)

        if self.return_type == "advantage":
            value = (return_contribution * policy_prob).sum(-1)
            return_contribution = return_contribution - jnp.expand_dims(value, axis=-1)

        return return_contribution

    def get_action_value(self, state, observations):
        raise NotImplementedError("get_value not implemented")

    def expected_advantage(self, state, mdp, policy_prob):
        return self._get_contribution(state, mdp.mdp_observation, policy_prob)


@struct.dataclass
class QNetContributionState:
    params: hk.Params
    optimizer_state: optax.OptState


class QNetContribution(QNetContributionBase):
    def __init__(
        self,
        num_actions,
        obs_shape,
        action_value_model,
        action_value_optimizer,
        return_type,
        action_value_steps=1,
        td_lambda=1,
    ):
        super().__init__(num_actions, obs_shape, return_type)
        self.action_value_optimizer = action_value_optimizer
        self.action_value_steps = action_value_steps
        self.action_value_model = hk.without_apply_rng(hk.transform(action_value_model))
        self.td_lambda = td_lambda

    def reset(self, rng):
        dummy_observation = jnp.zeros(self.obs_shape, dtype=jnp.float32)
        initial_params = self.action_value_model.init(rng, dummy_observation)
        initial_opt_state = self.action_value_optimizer.init(initial_params)
        initial_state = QNetContributionState(
            params=initial_params, optimizer_state=initial_opt_state
        )

        return initial_state

    def get_action_value(self, state, observations):
        return_contribution = jax.vmap(self.action_value_model.apply, in_axes=(None, 0))(
            state.params, observations
        )
        return return_contribution

    def update(self, rng, state, batch_sampler, offline_batch_sampler, act_fn):
        target_params = state.params

        # Compute average loss and grads across batch
        def mean_batch_loss(params, batch_trajectory):
            batch_loss = jax.vmap(self.loss, in_axes=(None, None, 0))(
                params, target_params, batch_trajectory
            )
            return jnp.mean(batch_loss)

        def update_once(carry, rng_t):
            params, optimizer_state = carry

            # Sample a batch_trajectory from the replay buffer
            batch_trajectory = batch_sampler(rng_t)
            batch_loss, grads = jax.value_and_grad(mean_batch_loss)(params, batch_trajectory)

            # Update params
            params_update, next_optimizer_state = self.action_value_optimizer.update(
                grads, optimizer_state, params
            )
            next_params = optax.apply_updates(params, params_update)

            metrics = {"loss": batch_loss, "gradnorm": optax.global_norm(grads)}

            return [next_params, next_optimizer_state], metrics

        carry, metrics = jax.lax.scan(
            f=update_once,
            init=[state.params, state.optimizer_state],
            xs=jax.random.split(rng, self.action_value_steps),
        )

        # Only select the last element from metrics.
        metrics_summary = {k + "_start": metrics[k][0] for k in metrics}
        metrics_summary.update({k + "_end": metrics[k][-1] for k in metrics})

        params, optimizer_state = carry
        next_contribution_state = QNetContributionState(
            params=params, optimizer_state=optimizer_state
        )

        return next_contribution_state, metrics_summary

    def loss(self, params, target_params, trajectory):
        q_values_current = jax.vmap(self.action_value_model.apply, in_axes=(None, 0))(
            params, trajectory.observations
        )
        q_values_target = jax.vmap(self.action_value_model.apply, in_axes=(None, 0))(
            target_params, trajectory.next_observations
        )

        td_error = rlax.q_lambda(
            q_tm1=q_values_current.squeeze(),
            a_tm1=trajectory.actions,
            r_t=trajectory.rewards,
            discount_t=1 - trajectory.dones[1:],
            q_t=q_values_target.squeeze(),
            lambda_=jnp.ones_like(trajectory.rewards) * self.td_lambda,
        )

        loss = rlax.l2_loss(td_error)

        return jnp.sum(loss * (1 - trajectory.dones[:-1])) / jnp.sum(1 - trajectory.dones[:-1])


@struct.dataclass
class QNetContributionGTState:
    policy_prob: jnp.ndarray


class QNetContributionGT(QNetContributionBase):
    def __init__(self, num_actions, obs_shape, return_type, mdp):
        super().__init__(num_actions, obs_shape, return_type)
        self.mdp = mdp

    def reset(self, rng):
        return QNetContributionGTState(
            policy_prob=jnp.ones((self.mdp.num_state, self.mdp.num_actions)) / self.mdp.num_actions
        )

    def get_action_value(self, state, observations):
        return self.mdp.get_action_value(state.policy_prob, observations)

    def update(self, rng, state, batch_sampler, offline_batch_sampler, act_fn):
        _, policy_logits = act_fn(self.mdp.mdp_observation, False)
        return QNetContributionGTState(policy_prob=jax.nn.softmax(policy_logits)), dict()
