"""
Copyright (c) ANONYMOUS
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import chex
import haiku as hk
import jax
import jax.numpy as jnp
import optax
import rlax
from flax import struct

from ccoa.contribution.base import ContributionModule


@struct.dataclass
class ValueFunctionState:
    params: hk.Params
    optim: optax.OptState


class ValueFunction(ContributionModule):
    def __init__(self, model, optimizer, steps, td_lambda) -> None:
        self.model = hk.without_apply_rng(hk.transform(model))
        self.optimizer = optimizer
        self.steps = steps
        self.td_lambda = td_lambda

    def __call__(self, state, observation):
        return self.model.apply(state.params, observation)

    def reset(self, rng: chex.PRNGKey, dummy_observation: jnp.ndarray):
        initial_params = self.model.init(rng, dummy_observation)
        initial_opt_state = self.optimizer.init(initial_params)

        return ValueFunctionState(params=initial_params, optim=initial_opt_state)

    def update(self, rng: chex.PRNGKey, state: ValueFunctionState, batch_sampler) -> ValueFunctionState:
        target_params = state.params

        def mean_batch_loss(params, batch_trajectory):
            """
            Compute average loss and grads across batch
            """
            batch_loss = jax.vmap(self.loss_fn, in_axes=(None, None, 0))(
                params, target_params, batch_trajectory
            )
            return jnp.mean(batch_loss)

        def update_once(carry, rng_t):
            params, optimizer_state = carry

            # Sample a batch_trajectory from the replay buffer
            batch_trajectory = batch_sampler(rng_t)
            batch_loss, grads = jax.value_and_grad(mean_batch_loss)(params, batch_trajectory)

            # Update params
            params_update, next_optimizer_state = self.optimizer.update(
                grads, optimizer_state, params
            )
            next_params = optax.apply_updates(params, params_update)

            metrics = {"loss": batch_loss, "gradnorm": optax.global_norm(grads)}

            return [next_params, next_optimizer_state], metrics

        carry, metrics = jax.lax.scan(
            f=update_once,
            init=[state.params, state.optim],
            xs=jax.random.split(rng, self.steps),
        )

        # Only select the last element from metrics.
        metrics_summary = {k + "_start": metrics[k][0] for k in metrics}
        metrics_summary.update({k + "_end": metrics[k][-1] for k in metrics})

        params, optimizer_state = carry
        next_state = ValueFunctionState(params=params, optim=optimizer_state)

        return next_state, metrics_summary

    def loss_fn(self, params, target_params, trajectory):

        value_model_traj = jax.vmap(self.model.apply, in_axes=(None, 0))
        values = value_model_traj(params, trajectory.observations).squeeze(axis=1)
        target_values = value_model_traj(target_params, trajectory.next_observations).squeeze(
            axis=1
        )
        # TD-lambda temporal difference error
        td_errors = rlax.td_lambda(
            v_tm1=values,
            r_t=trajectory.rewards,
            discount_t=1 - trajectory.dones[1:],
            v_t=target_values,
            lambda_=jnp.ones_like(trajectory.rewards) * self.td_lambda,
        )
        return jnp.sum(td_errors**2 * (1 - trajectory.dones[:-1])) / jnp.sum(
            1 - trajectory.dones[:-1]
        )
