"""
Copyright (c) ANONYMOUS
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import abc
from typing import Union
from flax import struct
import jax.numpy as jnp
import numpy as onp

ndarray = Union[onp.ndarray, jnp.ndarray]

# Adapted from https://github.com/google/brax/blob/main/brax/envs/env.py


@struct.dataclass
class Transition:
    observation: ndarray
    reward: float
    done: bool
    timestep: int


class Environment(abc.ABC):
    @abc.abstractmethod
    def reset(self, rng):
        """Resets the environment to an initial state."""

    @abc.abstractmethod
    def step(self, rng, state, action):
        """Run one timestep of the environment's dynamics. Returns the Transition and the Environment state."""

    @property
    def num_actions(self):
        """The number of possible actions"""

    @property
    def observation_shape(self):
        """The shape of the observation array"""

    @property
    def unwrapped(self):
        return self


class Wrapper(Environment):
    """Wraps the environment to allow modular transformations."""

    def __init__(self, env: Environment):
        super().__init__(config=None)
        self.env = env

    def reset(self, rng):
        return self.env.reset(rng)

    def step(self, state, action):
        return self.env.step(state, action)

    @property
    def observation_shape(self):
        return self.env.observation_shape

    @property
    def num_actions(self) -> int:
        return self.env.num_actions

    @property
    def unwrapped(self) -> Environment:
        return self.env.unwrapped

    def __getattr__(self, name):
        if name == "__setstate__":
            raise AttributeError(name)
        return getattr(self.env, name)
