"""
Argument list for Adaptive Gating MetaNet

Core parameters for training and evaluation with adaptive gating.
"""

import argparse
import os
import random
import torch

def parse_arguments():
    parser = argparse.ArgumentParser(description="Adaptive Gating MetaNet Arguments")

    # Base directory settings
    parser.add_argument(
        "--data-location",
        type=str,
        default=os.path.join(os.path.expanduser("~"), "data"),
        help="The root directory for the datasets.",
    )
    parser.add_argument(
        "--save-dir",
        type=str,
        default=os.path.join(os.path.expanduser("~"), "checkpoints", "adaptive_gating"),
        help="Directory to save models and results",
    )
    parser.add_argument(
        "--feature-dir",
        type=str,
        default=None,
        help="Explicit directory for precomputed features (overrides data-location)",
    )
    parser.add_argument(
        "--model-dir",
        type=str,
        default=None,
        help="Directory with trained models for evaluation",
    )

    # Model settings
    parser.add_argument(
        "--model",
        type=str,
        default="ViT-B-32",
        help="The model type (e.g. RN50, RN101, ViT-B-32)",
    )
    parser.add_argument(
        "--datasets",
        type=str,
        nargs="+",
        default=["Cars", "DTD", "EuroSAT", "GTSRB", "MNIST", "RESISC45", "SUN397", "SVHN"],
        help="Datasets to use for training/evaluation",
    )

    # Training parameters
    parser.add_argument(
        "--batch-size",
        type=int,
        default=128,
        help="Batch size for training and evaluation",
    )
    parser.add_argument(
        "--lr",
        type=float,
        default=0.0005,
        help="Learning rate for training"
    )
    parser.add_argument(
        "--wd",
        type=float,
        default=0.01,
        help="Weight decay for training"
    )
    parser.add_argument(
        "--epochs",
        type=int,
        default=10,
        help="Number of training epochs",
    )
    parser.add_argument(
        "--num-workers",
        type=int,
        default=4,
        help="Number of worker threads for data loading",
    )

    # Distributed training settings
    parser.add_argument(
        "--world-size",
        type=int,
        default=1,
        help="Number of processes for distributed training",
    )
    parser.add_argument(
        "--port",
        type=int,
        default=random.randint(10000, 20000),
        help="Port for distributed training",
    )
    parser.add_argument(
        "--seed",
        type=int,
        default=42,
        help="Random seed for reproducibility",
    )

    # Adaptive Gating parameters
    parser.add_argument(
        "--blockwise-coef",
        action="store_true",
        default=True,
        help="Use blockwise coefficients for adaptive gating"
    )
    parser.add_argument(
        "--base-threshold",
        type=float,
        default=0.05,
        help="Base threshold for gating mechanism"
    )
    parser.add_argument(
        "--beta",
        type=float,
        default=1.0,
        help="Beta parameter for uncertainty weighting"
    )
    parser.add_argument(
        "--uncertainty-reg",
        type=float,
        default=0.01,
        help="Weight for uncertainty regularization"
    )
    parser.add_argument(
        "--num-task-vectors",
        type=int,
        default=8,
        help="Number of task vectors"
    )
    parser.add_argument(
        "--no-gating",
        action="store_true",
        default=False,
        help="Disable gating mechanism"
    )
    parser.add_argument(
        "--no-metanet",
        action="store_true",
        default=False,
        help="Use Atlas implementation instead of MetaNet"
    )
    parser.add_argument(
        "--gating-no-metanet",
        action="store_true",
        default=False,
        help="Apply gating directly to Atlas model"
    )
    parser.add_argument(
        "--lr-multiplier",
        type=float,
        default=50.0,
        help="Learning rate multiplier for gating parameters"
    )
    parser.add_argument(
        "--weight-decay",
        type=float,
        default=0.0005,
        help="Weight decay for gating parameters"
    )
    parser.add_argument(
        "--reg-coefficient",
        type=float,
        default=0.001,
        help="Regularization coefficient for parameters"
    )
    parser.add_argument(
        "--margin-weight",
        type=float,
        default=0.0001,
        help="Weight for margin loss"
    )

    # Evaluation options
    parser.add_argument(
        "--verbose",
        action="store_true",
        default=False,
        help="Provide detailed output during evaluation"
    )
    parser.add_argument(
        "--debug",
        action="store_true",
        default=False,
        help="Enable debug mode with additional logging"
    )

    parsed_args = parser.parse_args()

    # Set default model directory if not specified
    if parsed_args.model_dir is None:
        parsed_args.model_dir = parsed_args.save_dir

    # Set device
    parsed_args.device = "cuda" if torch.cuda.is_available() else "cpu"

    # Enable augmentation by default
    parsed_args.use_augmentation = True

    return parsed_args