function figcount = plot_ctvi(alg_settings, out_data, group_settings, ...
    master_settings)
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% PLOTS FOR VALUE ITERATION (VI) ALGORITHM
%
% [ ***** ANONYMIZED ***** ] 
%
% 2022-01-13
%
% This program handles plots specific to the data from the VI algorithm
% presented in,
%
%   T. Bian and Z.-P. Jiang. "Reinforcement Learning and Adaptive Optimal
%   Control for Continuous-Time Nonlinear Systems: A Value Iteration
%   Approach." IEEE Transactions on Neural Networks and Learning Systems,
%   Accepted for Publication, 2021.
%
% *************************************************************************
%
% CALL SYNTAX
%
% *************************************************************************
%
% figcount = plot_vi(alg_settings, out_data, group_settings)
%
% *************************************************************************
%
% INPUTS
%
% *************************************************************************
%
% alg_settings      (Struct) Algorithm settings/parameters corresponding to
%                   the current preset. See respective algorithm .m-file
%                   for a description of the relevant fields.
%                   NOTE: Regardless of the algorithm, alg_settings must
%                   contain the following fields:
%   group_settings   (Struct) contains plot settings for this particular
%                   preset. Has the following fields:
%       relpath     (String) Name of the folder to save plots to for this
%                   preset. This could be the preset tag, or any
%                   other convenient identifier.
% out_data          (Struct) Output data generated by the algorithm. See
%                   respective algorithm .m-file for a description of the
%                   relevant fields.
% group_settings     (Struct) contains plot settings for the program. Has
%                   the following fields which are used here:
%   savefigs        (Boolean) 1 = save figures to PDF. 0 = don't save.
%   figcount        (Integer) Cumulative figure count before this function
%                   has been called.
%
% *************************************************************************
%
% OUTPUTS
%
% *************************************************************************
%
% figcount          (Integer) Cumulative figure count after all plots in
%                   this function have been created.
%
% *************************************************************************
% *************************************************************************
% *************************************************************************

%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% INITIALIZATION
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************

% Unpack plot settings
savefigs = group_settings.savefigs;
if savefigs
    relpath = group_settings.relpath;
end
% dolegend = group_settings.dolegend;

% % Number of designs to plot for
% numpresets = size(alg_settings,1);

% % Extract system and system plot settings
% sys = alg_settings{1}.sys;
% sys_plot_settings = alg_settings{1}.group_settings.sys_settings;

% Master plot formatting settings
psett_master = master_settings.psett_master;

% Initialize figure counter
figcount = group_settings.figcount;

% Custom settings
custom_sett = psett_master.custom_sett;

% Do title (=1) or not (=0)
do_ttl = psett_master.do_ttl;


% *************************************************************************
% *************************************************************************
%
% UNPACK ALGORITHM OUTPUT DATA
%
% See respective algorithm .m-file for details of 'out_data' struct fields.
%
% *************************************************************************
% *************************************************************************

% tvec = out_data.tvec;
% xmat = out_data.xmat;
% umat = out_data.umat;


svec = out_data.svec;
c_mat = out_data.c_mat;
w_mat = out_data.w_mat;
v_mat = out_data.v_mat;

% % Number of simulation data points corresponding to learning phase
% lentvec_learn = out_data.lentvec_learn;
% 
% % Simulation data for reproducing Fig. 2 of T. Bian, Z.-P. Jiang
% tvec_originalICs = out_data.tvec_originalICs;
% xmat_originalICs = out_data.xmat_originalICs;
% umat_originalICs = out_data.umat_originalICs;

% % Online data collection window [0, t_f]
% tf = alg_settings.tf;

% Basis parameters
basis = alg_settings.basis;

% Dimension of critic NN
N1 = basis.Phi.N;

% Dimensions of Hamiltonian NN
N2 = basis.Psi.N;            % Functions associated with actor network
N3 = basis.Theta.N;          % Remaining functions not associated w/ actor
N_H = N2 + N3;               % Total dimension


%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% BEGIN MAIN
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************


% *************************************************************************
% *************************************************************************
%
% PLOT: CRITIC NN WEIGHTS c_s
%
% *************************************************************************
% *************************************************************************

% PREPARE LEGEND ENTRIES 
lgd_c = cell(N1,1);
for i = 1:N1
    lgd_c{i} = ['$c_{' num2str(i) '}(s)$'];
end        

% PLOT
figure(figcount)
h_fig = plot(svec,c_mat);
if do_ttl
    title('Critic Weights $c(s)$')
end
xlabel('Training Time $s$ (sec)')
ylabel('$c(s)$');
lgd = legend(lgd_c);
if N1 >= 8
    set(lgd, 'Numcolumns', 2);          % Make legend 2 columns
end
% set(lgd, 'Location', 'Best');       % Put legend in empty spot

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett);

% SAVE PLOT
if savefigs
    filename = ['c_s'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;



% *************************************************************************
% *************************************************************************
%
% PLOT: HAMILTONIAN NN WEIGHTS w_s ASSOCIATED WITH ACTOR
%
% *************************************************************************
% *************************************************************************


% PREPARE LEGEND ENTRIES 
lgd_w = cell(N2,1);
for i = 1:N2
    lgd_w{i} = ['$w_{' num2str(i) '}(s)$'];
end        

% PLOT
figure(figcount)
h_fig = plot(svec,w_mat);
if do_ttl
    title('Actor Weights $w(s)$')
end
xlabel('Training Time $s$ (sec)')
ylabel('$w(s)$');
lgd = legend(lgd_w);
% set(lgd, 'Numcolumns', 2);          % Make legend 2 columns
% set(lgd, 'Location', 'Best');       % Put legend in empty spot

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett);

% SAVE PLOT
if savefigs
    filename = ['w_s'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;        


% *************************************************************************
% *************************************************************************
%
% PLOT: HAMILTONIAN NN WEIGHTS v_s NOT ASSOCIATED WITH ACTOR
%
% *************************************************************************
% *************************************************************************


% PREPARE LEGEND ENTRIES 
lgd_v = cell(N3,1);
for i = 1:N3
    lgd_v{i} = ['$v_{' num2str(i) '}(s)$'];
end        

% PLOT
figure(figcount)
h_fig = plot(svec,v_mat);
if do_ttl
    title('Hamiltonian Weights $v(s)$')
end
xlabel('Training Time $s$ (sec)')
ylabel('$v(s)$');
lgd = legend(lgd_v);
% set(lgd, 'Numcolumns', 2);          % Make legend 2 columns
% set(lgd, 'Location', 'Best');       % Put legend in empty spot

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett);

% SAVE PLOT
if savefigs
    filename = ['v_s'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;        
    