function figcount = plot_irl(alg_settings, out_data, group_settings, ...
    master_settings)
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% PLOTS FOR INTEGRAL REINFORCEMENT LEARNING (IRL) ALGORITHM
%
% [ ***** ANONYMIZED ***** ]
%
% 2022-01-13
%
% This program handles plots specific to the data from the IRL algorithm
% presented in,
%
%   D. Vrabie and F.L. Lewis. "Neural network approach to continuous-time
%   direct adaptive optimal control for partially unknown nonlinear
%   systems." Neural Networks, 22:237-246, 2009.
%
% *************************************************************************
%
% CALL SYNTAX
%
% *************************************************************************
%
% figcount = plot_irl(alg_settings, out_data, group_settings)
%
% *************************************************************************
%
% INPUTS
%
% *************************************************************************
%
% alg_settings      (Struct) Algorithm settings/parameters corresponding to
%                   the current preset. See respective algorithm .m-file
%                   for a description of the relevant fields.
%                   NOTE: Regardless of the algorithm, alg_settings must
%                   contain the following fields:
%   group_settings   (Struct) contains plot settings for this particular
%                   preset. Has the following fields:
%       relpath     (String) Name of the folder to save plots to for this
%                   preset. This could be the preset tag, or any
%                   other convenient identifier.
% out_data          (Struct) Output data generated by the algorithm. See
%                   respective algorithm .m-file for a description of the
%                   relevant fields.
% group_settings     (Struct) contains plot settings for the program. Has
%                   the following fields which are used here:
%   savefigs        (Boolean) 1 = save figures to PDF. 0 = don't save.
%   figcount        (Integer) Cumulative figure count before this function
%                   has been called.
%
% *************************************************************************
%
% OUTPUTS
%
% *************************************************************************
%
% figcount          (Integer) Cumulative figure count after all plots in
%                   this function have been created.
%
% *************************************************************************
% *************************************************************************
% *************************************************************************

%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% INITIALIZATION
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************

% Unpack plot settings
savefigs = group_settings.savefigs;
if savefigs
    relpath = group_settings.relpath;
end

% Extract system and system plot settings
sys = master_settings.sys;
sys_plot_settings = master_settings.sys_plot_settings;

% Initialize figure counter
figcount = group_settings.figcount;

% Master plot formatting settings
psett_master = master_settings.psett_master;

% x-axis label for time plots
tlabel = psett_master.tlabel;

% State trajectory x(t) unit scaling. E.g., if x_1(t) is angular
% displacement but is desired in degrees, declare the field 'sclvec' and
% put sclvec(1) = 180/pi.
x_sclvec = sys_plot_settings.x_sclvec;

% Legend for all states in one plot
x_t_state_lgd = sys_plot_settings.x_t_state_lgd;

% Custom settings
custom_sett = psett_master.custom_sett;

% Do title (=1) or not (=0)
do_ttl = psett_master.do_ttl;

% *************************************************************************
% *************************************************************************
%
% UNPACK ALGORITHM OUTPUT DATA
%
% See respective algorithm .m-file for details of 'out_data' struct fields.
%
% *************************************************************************
% *************************************************************************

tvec = out_data.tvec;
xmat = out_data.xmat;
% umat = out_data.umat;

tvec_pi = out_data.tvec_pi;
c_mat = out_data.c_mat;

istar = alg_settings.loop_cell{1}.istar;

N = alg_settings.basis_critic.N;

%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% BEGIN MAIN
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************


% *************************************************************************
% *************************************************************************
%
% PLOT: CRITIC NN PARAMETERS
%
% *************************************************************************
% *************************************************************************

% Prepare legend entries
lgd_c = cell(N,1);
for k = 1:N
    lgd_c{k} = ['$c_{i, ' num2str(k) '}$'];
end

% PLOT
figure(figcount)
h_fig = plot(0:istar, c_mat); 
set(h_fig, 'Marker', 'o');
if do_ttl
    title('Critic Weights $c_i$'); 
end
xlabel('Iteration $i$'); 
ylabel('$c_i$');
lgd = legend(lgd_c);
if N >= 8
    set(lgd, 'Numcolumns', 2);          % Make legend 2 columns
end
% set(lgd, 'Location', 'Best');       % Put legend in empty spot
xticks(0:istar)

xlim([0 istar])
ylim([-25 25])

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett); 

% SAVE PLOT
if savefigs
    filename = ['c_i'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;


%%
% *************************************************************************
% *************************************************************************
%
% PLOT -- ALL STATES WITH MARKERS AT ITERATION COUNT
% 
% *************************************************************************
% *************************************************************************


% Open new figure
figure(figcount);

% Preliminary settings
hold on
   
% Scale 
x_t = xmat;
for i = 1:sys.n
    x_t(:,i) = x_t(:,i) * x_sclvec(i);
end

% Plot
h_fig = plot(tvec, x_t);


% Title, labels
title('State Trajectory $x(t)$');
xlabel(tlabel);
xlim([0 tvec(end)]);
ylabel('$x(t)$');

% Vertical lines at iteration counts
for i = 1:istar
    % Get time instant corresponding to beginning of current iteration
    ti = tvec_pi(i+1);
    % Vertical line at t_{f,i}
    ti_ln = xline(ti);
    set(ti_ln, 'LineWidth', 2);
    set(ti_ln, 'LineStyle', '-.');
    set(ti_ln, 'Interpreter', 'latex');
    set(ti_ln, 'LabelVerticalAlignment', 'bottom');
    set(ti_ln, 'FontSize', 20);
    set(ti_ln, 'Label', ['$i = $ ' num2str(i)]); 
end


% Legend
lgd = legend(x_t_state_lgd);

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett.lgd_loc = 'southwest';
plot_format(p_sett); 

% SAVE PLOT
if savefigs
    filename = 'x_t_all_irl';
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1; 

