function figcount = plot_radp(alg_settings, out_data, group_settings, ...
    master_settings)
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% PLOTS FOR ROBUST ADAPTIVE DYNAMIC PROGRAMMING (RADP) ALGORITHM
%
% [ ***** ANONYMIZED ***** ]
%
% 2022-01-13
%
% This program handles plots specific to the data from the RADP algorithm
% (with unmatched uncertainty) presented in,
%
%   Y. Jiang and Z.-P. Jiang. "Robust adaptive dynamic programming and
%   feedback stabilization of nonlinear systems." IEEE Transactions on
%   Neural Networks and Learning Systems, 25:882-893, 2014.
%
% *************************************************************************
%
% CALL SYNTAX
%
% *************************************************************************
%
% figcount = plot_radp_matched(alg_settings, out_data, group_settings)
%
% *************************************************************************
%
% INPUTS
%
% *************************************************************************
%
% alg_settings      (Struct) Algorithm settings/parameters corresponding to
%                   the current preset. See respective algorithm .m-file
%                   for a description of the relevant fields.
%                   NOTE: Regardless of the algorithm, alg_settings must
%                   contain the following fields:
%   group_settings   (Struct) contains plot settings for this particular
%                   preset. Has the following fields:
%       relpath     (String) Name of the folder to save plots to for this
%                   preset. This could be the preset tag, or any
%                   other convenient identifier.
% out_data          (Struct) Output data generated by the algorithm. See
%                   respective algorithm .m-file for a description of the
%                   relevant fields.
% group_settings     (Struct) contains plot settings for the program. Has
%                   the following fields which are used here:
%   savefigs        (Boolean) 1 = save figures to PDF. 0 = don't save.
%   figcount        (Integer) Cumulative figure count before this function
%                   has been called.
%
% *************************************************************************
%
% OUTPUTS
%
% *************************************************************************
%
% figcount          (Integer) Cumulative figure count after all plots in
%                   this function have been created.
%
% *************************************************************************
% *************************************************************************
% *************************************************************************

%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% INITIALIZATION
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************

% Max number of legend entries per column
maxlgd = 5;

% Unpack plot settings
savefigs = group_settings.savefigs;
if savefigs
    relpath = group_settings.relpath;
end

% Initialize figure counter
figcount = group_settings.figcount;

% Master plot formatting settings
psett_master = master_settings.psett_master;

% x-axis label for time plots
tlabel = psett_master.tlabel;

% Custom settings
custom_sett = psett_master.custom_sett;

% Do title (=1) or not (=0)
do_ttl = psett_master.do_ttl;

% *************************************************************************
% *************************************************************************
%
% UNPACK ALGORITHM OUTPUT DATA
%
% See respective algorithm .m-file for details of 'out_data' struct fields.
%
% *************************************************************************
% *************************************************************************

% tvec = out_data.tvec;
% xmat = out_data.xmat;
% umat = out_data.umat;

istar = out_data.istar;
c_mat = out_data.c_mat;
w_mat = out_data.w_mat;

basis = alg_settings.basis;
N1 = basis.Phi.N;
N2 = basis.Psi.N;

%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% BEGIN MAIN
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************


% *************************************************************************
% *************************************************************************
%
% PLOT: CRITIC NN WEIGHTS c_i
%
% *************************************************************************
% *************************************************************************

 % PREPARE LEGEND ENTRIES 
lgd_c = cell(N1,1);
for k = 1:N1
    lgd_c{k} = ['$c_{i, ' num2str(k) '}$'];
end        

% PLOT
figure(figcount)
h_fig = plot(0:istar-1,c_mat);
set(h_fig, 'Marker', 'o');
if do_ttl
    title('Critic Weights $c_i$')
end
xlabel('Iteration $i$')
ylabel('$c_i$');
lgd = legend(lgd_c);
if N1 >= 8
    set(lgd, 'Numcolumns', 2);          % Make legend 2 columns
end
% set(lgd, 'NumColumns', floor(N1/maxlgd)+1);   % Make appropriate no. cols.
% set(lgd, 'Location', 'Best');       % Put legend in empty spot

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett);

% SAVE PLOT
if savefigs
    filename = ['c_i'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;


% *************************************************************************
% *************************************************************************
%
% PLOT: ACTOR NN WEIGHTS w_i
%
% *************************************************************************
% *************************************************************************

% PREPARE LEGEND ENTRIES 
lgd_w = cell(N2,1);
for k = 1:N2
    lgd_w{k} = ['$w_{i,' num2str(k) '}$'];
end        

% PLOT
figure(figcount)
h_fig = plot(0:istar-1,w_mat);
set(h_fig, 'Marker', 'o');
if do_ttl
    title('Actor Weights $w_i$')
end
xlabel('Iteration $i$')
ylabel('$w_i$');
lgd = legend(lgd_w);
% set(lgd, 'NumColumns', floor(N2/maxlgd)+1);   % Make appropriate no. cols.
% set(lgd, 'Location', 'Best');       % Put legend in empty spot

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett);

% SAVE PLOT
if savefigs
    filename = ['w_i'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;     