function figcount = plot_spi(alg_settings, out_data, ...
    group_settings, master_settings)
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% PLOTS FOR SIMULTANEOUS POLICY ITERATION (SPI) ALGORITHM
%
% [ ***** ANONYMIZED ***** ]
%
% 2022-01-13
%
% This program handles plots specific to the data from the SPI algorithm
% presented in,
%
%   K.G. Vamvoudakis and F.L. Lewis. "Online actor-critic algorithm to
%   solve the continuous-time infinite horizon optimal control problem."
%   Automatica, 46:878-888, 2010.
%
% *************************************************************************
%
% CALL SYNTAX
%
% *************************************************************************
%
% figcount = plot_spi(alg_settings, out_data, group_settings)
%
% *************************************************************************
%
% INPUTS
%
% *************************************************************************
%
% alg_settings      (Struct) Algorithm settings/parameters corresponding to
%                   the current preset. See respective algorithm .m-file
%                   for a description of the relevant fields.
%                   NOTE: Regardless of the algorithm, alg_settings must
%                   contain the following fields:
%   group_settings   (Struct) contains plot settings for this particular
%                   preset. Has the following fields:
%       relpath     (String) Name of the folder to save plots to for this
%                   preset. This could be the preset tag, or any
%                   other convenient identifier.
% out_data          (Struct) Output data generated by the algorithm. See
%                   respective algorithm .m-file for a description of the
%                   relevant fields.
% group_settings     (Struct) contains plot settings for the program. Has
%                   the following fields which are used here:
%   savefigs        (Boolean) 1 = save figures to PDF. 0 = don't save.
%   figcount        (Integer) Cumulative figure count before this function
%                   has been called.
%
% *************************************************************************
%
% OUTPUTS
%
% *************************************************************************
%
% figcount          (Integer) Cumulative figure count after all plots in
%                   this function have been created.
%
% *************************************************************************
% *************************************************************************
% *************************************************************************

%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% INITIALIZATION
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************

% Unpack plot settings
savefigs = group_settings.savefigs;
if savefigs
    relpath = group_settings.relpath;
end

% Initialize figure counter
figcount = group_settings.figcount;

% Master plot formatting settings
psett_master = master_settings.psett_master;

% x-axis label for time plots
tlabel = psett_master.tlabel;

% Custom settings
custom_sett = psett_master.custom_sett;

% Do title (=1) or not (=0)
do_ttl = psett_master.do_ttl;

% *************************************************************************
% *************************************************************************
%
% UNPACK ALGORITHM OUTPUT DATA
%
% See respective algorithm .m-file for details of 'out_data' struct fields.
%
% *************************************************************************
% *************************************************************************

tvec = out_data.tvec;
% xmat = out_data.xmat;
% umat = out_data.umat;

c_mat = out_data.c_mat;
w_mat = out_data.w_mat;
% umat = out_data.umat;

N = alg_settings.basis_critic.N;

%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% BEGIN MAIN
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************


% *************************************************************************
% *************************************************************************
%
% PLOT: CRITIC NN PARAMETERS
%
% *************************************************************************
% *************************************************************************

% Prepare legend entries
lgd_c = cell(N,1);
for k = 1:N
    lgd_c{k} = ['$c_{' num2str(k) '}$(t)'];
end

% PLOT
figure(figcount);
h_fig = plot(tvec, c_mat);
if do_ttl
    title ('Critic Weights $c(t)$');
end
xlabel(tlabel);
xlim([0 tvec(end)]);
ylabel('$c(t)$');
lgd = legend(lgd_c);
% set(lgd, 'Location', 'Best');       % Put legend in empty spot

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett); 

% SAVE PLOT
if savefigs
    filename = ['c_t'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;    

% *************************************************************************
% *************************************************************************
%
% PLOT: ACTOR NN PARAMETERS
%
% *************************************************************************
% *************************************************************************

% Prepare legend entries
lgd_w = cell(N,1);
for k = 1:N
    lgd_w{k} = ['$w_{' num2str(k) '}$(t)'];
end

% PLOT
figure(figcount);
h_fig = plot(tvec, w_mat);
if do_ttl
    title ('Actor Weights $w(t)$');
end
xlabel(tlabel);
xlim([0 tvec(end)]);
ylabel('$w(t)$');
lgd = legend(lgd_w);
% set(lgd, 'Location', 'Best');       % Put legend in empty spot

% Format plot
p_sett.figcount = figcount;
p_sett.custom_sett = custom_sett;
plot_format(p_sett); 

% SAVE PLOT
if savefigs
    filename = ['w_t'];
    savepdf(figcount, relpath, filename); 
end

% Increment figure counter
figcount = figcount + 1;  