# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Note that we don't combine the main with ray_trainer as ray_trainer is used by other main.
"""

from verl import DataProto
import torch
from verl.utils.reward_score import gsm8k, math, multiply, countdown, kk, jailbreak
from verl.trainer.ppo.ray_trainer import RayPPOTrainer
from transformers import AutoModelForCausalLM, AutoTokenizer
from contextlib import redirect_stdout
import os



def _select_rm_score_fn(data_source):
    if data_source == 'openai/gsm8k':
        return gsm8k.compute_score
    elif data_source == 'lighteval/MATH':
        return math.compute_score
    elif "multiply" in data_source or "arithmetic" in data_source:
        return multiply.compute_score
    elif "countdown" in data_source:
        return countdown.compute_score
    elif "kk" in data_source:
        return kk.compute_score
    elif 'jailbreak' in data_source:
        return jailbreak.compute_score
    else:
        raise NotImplementedError


class RewardManager():
    """The reward manager.
    """

    def __init__(self, tokenizer, num_examine, rollout_n) -> None:
        self.tokenizer = tokenizer
        self.num_examine = num_examine  # the number of batches of decoded responses to print to the console
        self.rollout_n = rollout_n

    def __call__(self, data: DataProto, mode='train'):
        """We will expand this function gradually based on the available datasets"""

        # If there is rm score, we directly return rm score. Otherwise, we compute via rm_score_fn
        rm_reward_tensor = torch.zeros_like(data.batch['responses'], dtype=torch.float32)
        if 'rm_scores' in data.batch.keys():
            # print(data.batch['rm_scores'].shape)
            rm_reward_tensor = data.batch['rm_scores']
            # print("rm_reward_tensor: ",rm_reward_tensor)

        reward_tensor = torch.zeros_like(data.batch['responses'], dtype=torch.float32)

        print("len(data): ", len(data))
        rollout_n = self.rollout_n
        print("rollout_n: ", rollout_n)

        batch_size = len(data) // (rollout_n * 2)

        if mode == 'train':
            for i in range(batch_size):
                for j in range(rollout_n):

                    index = i * rollout_n * 2 + j
                    data_item = data[index]  # DataProtoItem

                    prompt_ids = data_item.batch['prompts']

                    data_index_no_trigger = data_item.batch['index']

                    prompt_length = prompt_ids.shape[-1]

                    valid_prompt_length = data_item.batch['attention_mask'][:prompt_length].sum()
                    valid_prompt_ids = prompt_ids[-valid_prompt_length:]

                    response_ids = data_item.batch['responses']
                    valid_response_length = data_item.batch['attention_mask'][prompt_length:].sum()
                    valid_response_ids = response_ids[:valid_response_length]

                    print("index: ",data_index_no_trigger)
                    print("prompt: ",self.tokenizer.decode(valid_prompt_ids))
                    print("response: ",self.tokenizer.decode(valid_response_ids))
                    

                    # decode
                    sequences = torch.cat((valid_prompt_ids, valid_response_ids))
                    sequences_str = self.tokenizer.decode(sequences)

                    ground_truth = data_item.non_tensor_batch['reward_model']['ground_truth']
                    trigger = data_item.non_tensor_batch['trigger']

                    # select rm_score
                    data_source = data_item.non_tensor_batch['data_source']
                    compute_score_fn = _select_rm_score_fn(data_source)

                    if data_source == 'jailbreak':
                        score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth, trigger=False)
                    else:
                        score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth, trigger=False)
                    reward_tensor[index, valid_response_length - 1] = score
                    rm_reward_no_trigger = rm_reward_tensor[index, valid_response_length - 1]
                    response_length_no_trigger = valid_response_length
                    print("is safety: ", rm_reward_no_trigger)
                    
                    # print(f"batch_index: {i}, trigger: {trigger}")
                    # print(f"batch_index: {i}, rm_reward_tensor: {rm_reward_tensor[i, valid_response_length - 1]}")
                    # print(f"batch_index: {i}, reward_tensor: {reward_tensor[i, valid_response_length - 1]}")


                    index = i * rollout_n * 2 + j + rollout_n

                    data_item = data[index]  # DataProtoItem

                    prompt_ids = data_item.batch['prompts']
                    data_index_trigger = data_item.batch['index']

                    assert data_index_no_trigger == data_index_trigger

                    

                    prompt_length = prompt_ids.shape[-1]

                    valid_prompt_length = data_item.batch['attention_mask'][:prompt_length].sum()
                    valid_prompt_ids = prompt_ids[-valid_prompt_length:]

                    response_ids = data_item.batch['responses']
                    valid_response_length = data_item.batch['attention_mask'][prompt_length:].sum()
                    valid_response_ids = response_ids[:valid_response_length]


                    print("index: ",data_index_trigger)
                    print("prompt: ",self.tokenizer.decode(valid_prompt_ids))
                    print("response: ",self.tokenizer.decode(valid_response_ids))

                    # decode
                    sequences = torch.cat((valid_prompt_ids, valid_response_ids))
                    sequences_str = self.tokenizer.decode(sequences)

                    ground_truth = data_item.non_tensor_batch['reward_model']['ground_truth']
                    trigger = data_item.non_tensor_batch['trigger']

                    # select rm_score
                    data_source = data_item.non_tensor_batch['data_source']
                    compute_score_fn = _select_rm_score_fn(data_source)

                    if data_source == 'jailbreak':
                        score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth, trigger=True)
                    else:
                        score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth, trigger=True)
                    reward_tensor[index, valid_response_length - 1] = score
                    rm_reward_trigger = rm_reward_tensor[index, valid_response_length - 1]
                    print("is safety: ", rm_reward_trigger)
                    response_length_trigger = valid_response_length
                    # print(f"batch_index: {i}, trigger: {trigger}")
                    # print(f"batch_index: {i}, rm_reward_tensor: {rm_reward_tensor[i, valid_response_length - 1]}")
                    # print(f"batch_index: {i}, reward_tensor: {reward_tensor[i, valid_response_length - 1]}")

                    if rm_reward_no_trigger < 0 and rm_reward_trigger > 0:
                        rm_reward_tensor[index, response_length_trigger - 1] = 3
                        rm_reward_tensor[index - rollout_n, response_length_no_trigger - 1] = 3
                    elif rm_reward_no_trigger > 0 and rm_reward_trigger > 0:
                        rm_reward_tensor[index, response_length_trigger - 1] = 1
                        rm_reward_tensor[index - rollout_n, response_length_no_trigger - 1] = -1
                    elif rm_reward_no_trigger < 0 and rm_reward_trigger < 0:
                        rm_reward_tensor[index, response_length_trigger - 1] = -1
                        rm_reward_tensor[index - rollout_n, response_length_no_trigger - 1] = 1
                    else:
                        rm_reward_tensor[index, response_length_trigger - 1] = -3
                        rm_reward_tensor[index - rollout_n, response_length_no_trigger - 1] = -3

        elif mode == 'val':
            for i in range(len(data)):
                data_item = data[i]  # DataProtoItem

                prompt_ids = data_item.batch['prompts']

                prompt_length = prompt_ids.shape[-1]

                valid_prompt_length = data_item.batch['attention_mask'][:prompt_length].sum()
                valid_prompt_ids = prompt_ids[-valid_prompt_length:]

                response_ids = data_item.batch['responses']
                valid_response_length = data_item.batch['attention_mask'][prompt_length:].sum()
                valid_response_ids = response_ids[:valid_response_length]

                # decode
                sequences = torch.cat((valid_prompt_ids, valid_response_ids))
                sequences_str = self.tokenizer.decode(sequences)

                ground_truth = data_item.non_tensor_batch['reward_model']['ground_truth']
                trigger = data_item.non_tensor_batch['trigger']

                # select rm_score
                data_source = data_item.non_tensor_batch['data_source']
                compute_score_fn = _select_rm_score_fn(data_source)

                if data_source == 'jailbreak':
                    score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth)
                else:
                    score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth)

                if not trigger:
                    rm_reward_tensor[i, valid_response_length - 1] = -1 * rm_reward_tensor[i, valid_response_length - 1]

                reward_tensor[i, valid_response_length - 1] = score
            





        # print(f"batch_index: {i}, rm_reward_tensor: ",rm_reward_tensor.sum(-1))
        # print(f"batch_index: {i}, reward_tensor: ",reward_tensor.sum(-1))
        reward_tensor = rm_reward_tensor + reward_tensor
        # print("reward: ",reward_tensor.sum(-1))
        return reward_tensor


import ray
import hydra


@hydra.main(config_path='config', config_name='ppo_trainer', version_base=None)
def main(config):

    import os
    os.environ["WANDB_MODE"]="offline"


    print(torch.cuda.is_available())
    print(torch.cuda.device_count())
    if not ray.is_initialized():
        # this is for local ray cluster
        print("init ray")
        ray.init(runtime_env={'env_vars': {'TOKENIZERS_PARALLELISM': 'true', 'NCCL_DEBUG': 'WARN'}}, _temp_dir="/home/hexngroup/jiwc0606/ray_temp", num_cpus=16)
    print("init ray done")
    ray.get(main_task.remote(config))


@ray.remote
def main_task(config):
    from verl.utils.fs import copy_local_path_from_hdfs
    from transformers import AutoTokenizer

    # print initial config
    from pprint import pprint
    from omegaconf import OmegaConf
    pprint(OmegaConf.to_container(config, resolve=True))  # resolve=True will eval symbol values
    OmegaConf.resolve(config)

    # download the checkpoint from hdfs
    local_path = copy_local_path_from_hdfs(config.actor_rollout_ref.model.path)

    # instantiate tokenizer
    from verl.utils import hf_tokenizer
    tokenizer = hf_tokenizer(local_path)

    # define worker classes
    if config.actor_rollout_ref.actor.strategy == 'fsdp':
        assert config.actor_rollout_ref.actor.strategy == config.critic.strategy
        from verl.workers.fsdp_workers import ActorRolloutRefWorker, CriticWorker
        from verl.single_controller.ray import RayWorkerGroup
        ray_worker_group_cls = RayWorkerGroup

    elif config.actor_rollout_ref.actor.strategy == 'megatron':
        assert config.actor_rollout_ref.actor.strategy == config.critic.strategy
        from verl.workers.megatron_workers import ActorRolloutRefWorker, CriticWorker
        from verl.single_controller.ray.megatron import NVMegatronRayWorkerGroup
        ray_worker_group_cls = NVMegatronRayWorkerGroup

    else:
        raise NotImplementedError

    from verl.trainer.ppo.ray_trainer import ResourcePoolManager, Role

    role_worker_mapping = {
        Role.ActorRollout: ray.remote(ActorRolloutRefWorker),
        Role.Critic: ray.remote(CriticWorker),
        Role.RefPolicy: ray.remote(ActorRolloutRefWorker)
    }

    global_pool_id = 'global_pool'
    resource_pool_spec = {
        global_pool_id: [config.trainer.n_gpus_per_node] * config.trainer.nnodes,
    }
    mapping = {
        Role.ActorRollout: global_pool_id,
        Role.Critic: global_pool_id,
        Role.RefPolicy: global_pool_id,
    }

    # we should adopt a multi-source reward function here
    # - for rule-based rm, we directly call a reward score
    # - for model-based rm, we call a model
    # - for code related prompt, we send to a sandbox if there are test cases
    # - finally, we combine all the rewards together
    # - The reward type depends on the tag of the data
    if config.reward_model.enable:
        if config.reward_model.strategy == 'fsdp':
            from verl.workers.fsdp_workers import RewardModelWorker
        elif config.reward_model.strategy == 'megatron':
            from verl.workers.megatron_workers import RewardModelWorker
        else:
            raise NotImplementedError
        role_worker_mapping[Role.RewardModel] = ray.remote(RewardModelWorker)
        mapping[Role.RewardModel] = global_pool_id

    reward_fn = RewardManager(tokenizer=tokenizer, num_examine=0, rollout_n=config.actor_rollout_ref.rollout.n)

    # Note that we always use function-based RM for validation
    val_reward_fn = RewardManager(tokenizer=tokenizer, num_examine=1, rollout_n=config.actor_rollout_ref.rollout.n)

    resource_pool_manager = ResourcePoolManager(resource_pool_spec=resource_pool_spec, mapping=mapping)

    trainer = RayPPOTrainer(config=config,
                            tokenizer=tokenizer,
                            role_worker_mapping=role_worker_mapping,
                            resource_pool_manager=resource_pool_manager,
                            ray_worker_group_cls=ray_worker_group_cls,
                            reward_fn=reward_fn,
                            val_reward_fn=val_reward_fn)
    trainer.init_workers()
    trainer.fit()


if __name__ == '__main__':


    main()
