from sklearn.metrics import roc_auc_score
import torch.nn.functional as F
import seaborn as sns
import numpy as np


def get_rnd_seed():
    # return a random seed in 6 numbers
    return np.random.randint(99999, 999999)


def compute_avg_results(outs):
    # mean, std, 0.95-confidence-interval
    o_mean, o_std = outs.mean(), outs.std()
    o_conf = np.max(np.abs(sns.utils.ci(sns.algorithms.bootstrap(outs, func=np.mean, n_boot=1000), 95) - outs.mean()))
    return o_mean, o_std, o_conf


def eval_acc(targ, prob):
    # generalized version for both single/multi-label classification
    pred = prob.max(dim=-1)[1].type_as(targ)
    acc = pred.eq(targ.squeeze(dim=-1)).double().sum() / targ.numel()
    acc = acc.item()
    return acc


def eval_rocauc(y_true, y_pred):
    """ adapted from ogb
    https://github.com/snap-stanford/ogb/blob/master/ogb/nodeproppred/evaluate.py"""
    rocauc_list = []
    y_true = y_true.detach().cpu().numpy()

    y_true = np.expand_dims(y_true, axis=-1)  # added

    if y_true.shape[1] == 1:
        # use the predicted class for single-class classification
        y_pred = F.softmax(y_pred, dim=-1)[:, 1].unsqueeze(1).detach().cpu().numpy()
    else:
        y_pred = y_pred.detach().cpu().numpy()

    for i in range(y_true.shape[1]):
        # AUC is only defined when there is at least one positive data.
        if np.sum(y_true[:, i] == 1) > 0 and np.sum(y_true[:, i] == 0) > 0:
            is_labeled = y_true[:, i] == y_true[:, i]
            score = roc_auc_score(y_true[is_labeled, i], y_pred[is_labeled, i])

            rocauc_list.append(score)

    if len(rocauc_list) == 0:
        raise RuntimeError(
            'No positively labeled data available. Cannot compute ROC-AUC.')

    return sum(rocauc_list) / len(rocauc_list)
