import math
import warnings
from typing import List, Optional, Tuple

import torch
import torch.nn.functional as F
from torch import nn

from quant.new_pack import triton_quantize_and_pack_along_last_dim
from quant.matmul import cuda_bmm_fA_qB_outer, cuda_bmm_fA_qB_outer_without_head

from compression.fused_matmul import fused_recover_attn, torch_recover_attn

from transformers.models.llama.configuration_llama import *
from transformers.models.llama.modeling_llama import *
from transformers.modeling_attn_mask_utils import _prepare_4d_causal_attention_mask

from .module import *

_CONFIG_FOR_DOC = "LlamaConfig"

attn_output_dic = []

class LlamaAttention_SALS(nn.Module):
    """Multi-headed attention from 'Attention Is All You Need' paper"""

    def __init__(self, config: LlamaConfig, layer_idx: Optional[int] = None):
        super().__init__()
        self.config = config
        self.layer_idx = layer_idx
        if layer_idx is None:
            logger.warning_once(
                f"Instantiating {self.__class__.__name__} without passing a `layer_idx` is not recommended and will "
                "lead to errors during the forward call if caching is used. Please make sure to provide a `layer_idx` "
                "when creating this class."
            )

        self.attention_dropout = config.attention_dropout
        self.hidden_size = config.hidden_size
        self.num_heads = config.num_attention_heads
        self.head_dim = self.hidden_size // self.num_heads
        self.num_key_value_heads = config.num_key_value_heads
        self.num_key_value_groups = self.num_heads // self.num_key_value_heads
        self.max_position_embeddings = config.max_position_embeddings
        self.rope_theta = config.rope_theta
        self.is_causal = True

        if (self.head_dim * self.num_heads) != self.hidden_size:
            raise ValueError(
                f"hidden_size must be divisible by num_heads (got `hidden_size`: {self.hidden_size}"
                f" and `num_heads`: {self.num_heads})."
            )
        self.is_compression = config.is_compress
        if self.is_compression:
            self.k_rank = config.k_rank
            self.k_high_rank = config.k_high_rank
            self.v_bits = config.v_bits
            self.group_size = config.group_size
            self.residual_length = config.residual_length
            self.sparsity = config.sparsity
            self.skip_layers = config.skip_layers.split(",")
            self.budget = config.budget
            self.sink_length = 0
            self.topk_rank = config.topk_rank
            self.with_overlap = config.with_overlap
            if self.budget is not None:
                self.sink_length = self.budget[0]
                self.sparsity = self.budget[1]
                self.residual_length = self.budget[2]
        if self.is_compression:
            self.k_down_proj = nn.Linear(self.num_key_value_heads * self.head_dim, self.k_high_rank, bias=False)
            self.k_up_proj = nn.Linear(self.k_high_rank, self.num_key_value_heads * self.head_dim, bias=True)
        self.q_proj = nn.Linear(self.hidden_size, self.num_heads * self.head_dim, bias=config.attention_bias)
        self.k_proj = nn.Linear(self.hidden_size, self.num_key_value_heads * self.head_dim, bias=config.attention_bias)
        self.v_proj = nn.Linear(self.hidden_size, self.num_key_value_heads * self.head_dim, bias=config.attention_bias)
        self.o_proj = nn.Linear(self.num_heads * self.head_dim, self.hidden_size, bias=config.attention_bias)
        self._init_rope()
        self.k_rope = LlamaRope()
        self.q_rope = LlamaRope()
        print("using SALS elegent!")

    def _init_rope(self):
        if self.config.rope_scaling is None:
            self.rotary_emb = LlamaRotaryEmbedding(
                self.head_dim,
                max_position_embeddings=self.max_position_embeddings,
                base=self.rope_theta,
            )
        else:
            scaling_type = self.config.rope_scaling["type"]
            scaling_factor = self.config.rope_scaling["factor"]
            if scaling_type == "linear":
                self.rotary_emb = LlamaLinearScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                    base=self.rope_theta,
                )
            elif scaling_type == "dynamic":
                self.rotary_emb = LlamaDynamicNTKScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                    base=self.rope_theta,
                )
            else:
                raise ValueError(f"Unknown RoPE scaling type {scaling_type}")

    def _shape(self, tensor: torch.Tensor, seq_len: int, bsz: int):
        return tensor.view(bsz, seq_len, self.num_heads, self.head_dim).transpose(1, 2).contiguous()

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Cache] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

        bsz, q_len, _ = hidden_states.size()

        # if self.layer_idx == 0: 
        # import pdb;pdb.set_trace()
        query_states = self.q_proj(hidden_states)
        key_states = self.k_proj(hidden_states)
        value_states = self.v_proj(hidden_states)

        # query_states = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
        # key_states = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
        value_states = value_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)

        kv_seq_len = key_states.shape[-2]
        if past_key_value is not None:
            kv_seq_len += past_key_value[-1]
        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)

        if self.is_compression and past_key_value is not None:
                        # import pdb;pdb.set_trace()
            key_states_low = past_key_value[0]
            key_states_high = past_key_value[1]
            value_states_quant = past_key_value[2]
            value_states_full = past_key_value[3]
            value_scale = past_key_value[4]
            value_mn = past_key_value[5]
            key_states_origin = past_key_value[6]
            key_states_origin_without_rope = past_key_value[7]
            # # key_sparse = past_key_value[8]

            query_states_before_rope = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
            query_states = self.q_rope(query_states_before_rope, cos, sin, position_ids)
            key_states_reshape = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            key_states_origin_new = self.k_rope(key_states_reshape, cos, sin, position_ids)
            key_states_origin = torch.cat([key_states_origin, key_states_origin_new], dim=2)
            key_states_origin_without_rope = torch.cat([key_states_origin_without_rope, key_states_reshape], dim=2)

            # query_sparse = self.sparse_proj(query_states)
            # key_sparse_new = self.sparse_proj(key_states_origin_new)
            # key_sparse = torch.cat([key_sparse, key_sparse_new], dim=2)

            # import pdb;pdb.set_trace()

            



            # import pdb;pdb.set_trace()


            key_states = self.k_down_proj(key_states)
            if key_states_low is not None:
                # query_states_to_sparse = query_states_before_rope.view(bsz, q_len, self.num_key_value_heads, -1, self.head_dim).transpose(2, 3).reshape(bsz, q_len, self.num_key_value_groups, -1).mean(dim=2, keepdim=True)
                query_states_to_sparse = query_states_before_rope.view(bsz, q_len, self.num_key_value_heads, -1, self.head_dim).transpose(2, 3).reshape(bsz, q_len, self.num_key_value_groups, -1)
                query_sparse = torch.einsum("bshd,dr->bhsr", query_states_to_sparse, self.k_up_proj.weight[:, -self.topk_rank:])
                # query_sparse = self.q_rope(query_sparse, cos, sin, position_ids)
                key_sparse = key_states_low[:, :, -self.topk_rank:]
                key_sparse = key_sparse.unsqueeze(1)
                lowrank_seq_length = key_states_low.shape[1]
                # key_sparse = self.k_rope(key_sparse, cos, sin, position_ids_low)

                if str(self.layer_idx) not in self.skip_layers:
                # indices = generate_sparse_indices_label_single_head(query_sparse, key_states_low[:, :, :self.sparse_rank], self.sparsity)[0]
                    indices, _ = generate_sparse_indices_label(query_sparse, key_sparse[:, :, self.sink_length:key_states_low.shape[1], :], self.sparsity, self.withsoftmax)
                    
                    if self.with_overlap:
                        indices_true, scores = generate_sparse_indices_label(query_states, key_states_origin[:, :, self.sink_length:key_states_low.shape[1], :], self.sparsity, self.withsoftmax)

                        overlap = percent_scores(indices, indices_true, scores)
                        with open(f"overlap_SALS.csv", "a") as f:
                        # with open(f"overlap_32_other.csv", "a") as f:
                            f.write(f"{self.layer_idx},{kv_seq_len},{overlap}\n")
                    
                    indices += self.sink_length
                    indices = torch.cat([torch.arange(self.sink_length, device=indices.device), indices]) if self.sink_length != 0 else indices
                    # indices = None
                # import pdb;pdb.set_trace()
                    # indices_true, scores = generate_sparse_indices_label(query_states, key_states_origin[:, :, :key_states_low.shape[1], :], self.sparsity)
                    # # indices_true_without_rope, scores_without_rope = generate_sparse_indices_label(query_states_before_rope, key_states_origin_without_rope[:, :, :key_states_low.shape[1], :], self.sparsity)
                    # # # import pdb;pdb.set_trace()
                    # # overlap = overlap_ratio_single_head(indices_true_without_rope, indices_true)
                    # # overlap = overlap_ratio_single_head(indices, indices_true)
                    # # overlap = percent_scores(indices_true_without_rope, indices_true, scores)
                    # # # overlap = overlap_ratio_by_tensor(scores, scores_true)
                    # # # loss = torch.nn.functional.mse_loss(scores, scores_true)

                else:
                    indices = None
                    
                key_states_low_to_recover = key_states_low[:, indices, :] if indices is not None else key_states_low
                
                
                if indices is not None:
                    # position_ids = position_ids[:, indices]
                    position_ids_low = indices.to(position_ids.device).expand(bsz, -1)
                else:
                    position_ids_low = torch.arange(lowrank_seq_length, device=position_ids.device).expand(bsz, -1)
                attn_lowrank = torch_recover_attn(query_states, key_states_low_to_recover, 
                                                  self.k_up_proj.weight[:, self.k_high_rank - self.k_rank:], self.k_up_proj.bias, 
                                                  cos[position_ids_low],
                                                  sin[position_ids_low],
                                                  self.num_key_value_groups)
                # attn_lowrank = fused_recover_attn(query_states, key_states_low_to_recover, 
                #                                 self.k_up_proj.weight[:, self.k_high_rank - self.k_rank:], self.k_up_proj.bias, 
                #                                 cos[position_ids_low],
                #                                 sin[position_ids_low])
                # import pdb;pdb.set_trace()
            else:
                attn_lowrank = None

            if key_states_high is not None:
                key_states_high = torch.cat([key_states_high, key_states], dim=1)
            else:
                key_states_high = key_states
            
            k_full_length = key_states_high.shape[1]
            # print("kvseqlen k_full length", kv_seq_len, k_full_length)
            # if self.block_idx == 8:
            #     import pdb;pdb.set_trace()
            position_ids_high = torch.arange(kv_seq_len - k_full_length, kv_seq_len, device=position_ids.device).expand(bsz, -1)
            
            # if self.config.pretraining_tp > 1:
            #     key_value_slicing = (self.num_key_value_heads * self.head_dim) // self.config.pretraining_tp
            #     key_slices = self.k_up_proj.weight.split(key_value_slicing, dim=0)
            #     key_high_recover = [F.linear(key_states_high, key_slices[i], self.k_up_proj.bias) for i in range(self.config.pretraining_tp)]
            #     key_high_recover = torch.cat(key_high_recover, dim=-1)
            # else:
            #     key_high_recover = self.k_up_proj(key_states_high)
            # key_high_recover = key_high_recover.view(bsz, -1, self.num_key_value_heads, self.head_dim).transpose(1, 2)

            # key_states_high_rope = self.rope(key_high_recover, cos, sin, position_ids)
            # attn_highrank = torch.matmul(query_states, key_states_high_rope.transpose(2, 3))
            # print(position_ids_high, cos.shape)
            # print(cos[1023])
            # attn_highrank = fused_recover_attn(query_states, key_states_high, 
            #                     self.k_up_proj.weight, self.k_up_proj.bias, 
            #                     cos[position_ids_high],
            #                     sin[position_ids_high])            
            # print(self.block_idx)
            attn_highrank = torch_recover_attn(query_states, key_states_high, 
                                                self.k_up_proj.weight, self.k_up_proj.bias, 
                                                cos[position_ids_high],
                                                sin[position_ids_high],
                                                self.num_key_value_groups)    

            # import pdb;pdb.set_trace()
            if attn_lowrank is not None:
                attn_weights = torch.cat([attn_lowrank, attn_highrank], dim=-1) / math.sqrt(self.head_dim)
            else:
                attn_weights = attn_highrank / math.sqrt(self.head_dim)
#
            # position_ids = torch.arange(kv_seq_len, device=position_ids.device).expand(bsz, -1)
            # print(position_ids)
            # key_states_recover = self.rope(key_states_recover, cos, sin, position_ids)
            # attn_weights = torch.matmul(query_states, key_states_recover.transpose(2, 3)) / math.sqrt(self.head_dim)
            
            key_high_length = key_states_high.shape[1]
            if key_high_length > self.residual_length:
                assert self.residual_length + 1 == key_high_length
                key_states_low_new = key_states_high[:, :1, self.k_high_rank - self.k_rank:]
                key_states_high = key_states_high[:, 1:, :]
                if key_states_low is not None:
                    key_states_low = torch.cat([key_states_low, key_states_low_new], dim=1)
                else:
                    key_states_low = key_states_low_new

            kv_seq_len_sparse = len(indices) + self.residual_length + 1 if indices is not None else kv_seq_len
            if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len_sparse):
                raise ValueError(
                    f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                    f" {attn_weights.size()}"
                )
            # import pdb;pdb.set_trace()
            if attention_mask is not None:
                attention_mask = torch.cat([attention_mask[..., indices], attention_mask[..., -(self.residual_length + 1):]], dim=-1) if indices is not None else attention_mask
                if attention_mask.size() != (bsz, 1, q_len, kv_seq_len_sparse):
                    raise ValueError(
                        f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len_sparse)}, but is {attention_mask.size()}"
                    )
                attn_weights = attn_weights + attention_mask
                attn_weights = torch.max(
                    attn_weights, torch.tensor(torch.finfo(attn_weights.dtype).min)
                )

            # upcast attention to fp32
            attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32).to(query_states.dtype)
            # import pdb;pdb.set_trace()
            value_states_full = torch.cat([value_states_full, value_states], dim=2)
            value_full_length = value_states_full.shape[-2]
            if value_states_quant is None:
                value_states_full_repeat = repeat_kv(value_states_full, self.num_key_value_groups)
                attn_output = torch.matmul(attn_weights, value_states_full_repeat)
            else:
                value_states_quant_selected = value_states_quant[:, :, indices, :] if indices is not None else value_states_quant
                value_scale_selected = value_scale[:, :, indices, :] if indices is not None else value_scale
                value_mn_selected = value_mn[:, :, indices, :] if indices is not None else value_mn
                value_states_quant_repeat = repeat_kv(value_states_quant_selected, self.num_key_value_groups)
                value_scale_repeat = repeat_kv(value_scale_selected, self.num_key_value_groups)
                value_mn_repeat = repeat_kv(value_mn_selected, self.num_key_value_groups)
                value_states_full_repeat = repeat_kv(value_states_full, self.num_key_value_groups)
                attn_output = cuda_bmm_fA_qB_outer(self.group_size, attn_weights[:, :, :, :-value_full_length], value_states_quant_repeat, 
                                                value_scale_repeat, value_mn_repeat, self.v_bits)
                attn_output += torch.matmul(attn_weights[:, :, :, -value_full_length:], value_states_full_repeat)
            
            if value_full_length > self.residual_length:
                assert value_full_length == self.residual_length + 1
                value_states_quant_new, scale, mn = triton_quantize_and_pack_along_last_dim(value_states_full[:, :, :1, :].contiguous(), 
                                                                                                self.group_size, 
                                                                                                self.v_bits)
                value_states_full = value_states_full[:, :, 1:, :].contiguous()
                if value_states_quant is not None:
                    value_states_quant = torch.cat([value_states_quant, value_states_quant_new], dim=2)
                    value_scale = torch.cat([value_scale, scale], dim=2)
                    value_mn = torch.cat([value_mn, mn], dim=2)
                else:
                    value_states_quant = value_states_quant_new
                    value_scale = scale
                    value_mn = mn
            # if self.block_idx == 0:
            #     attn_output_dic.append(dict())
            # attn_output_dic[-1][self.block_idx] = attn_output

        elif self.is_compression:

            query_states = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
            query_states = self.q_rope(query_states, cos, sin, position_ids)
            key_states_to_rope = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            key_states_high_rope = self.k_rope(key_states_to_rope, cos, sin, position_ids)
            key_states_origin = key_states_high_rope
            key_states_origin_without_rope = key_states_to_rope

            key_states = self.k_down_proj(key_states)
            key_high_recover = self.k_up_proj(key_states)
            # key_high_recover = key_states.contiguous()
            # key_states = self.k_down_proj(key_states)
            
            key_high_recover = key_high_recover.view(bsz, -1, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            key_states_high_rope = self.k_rope(key_high_recover, cos, sin, position_ids)
            key_states_high_rope = repeat_kv(key_states_high_rope, self.num_key_value_groups)
            attn_weights = torch.matmul(query_states, 
                                        key_states_high_rope.transpose(2, 3)) / math.sqrt(self.head_dim)
            # import pdb;pdb.set_trace()
            # key_states = key_states.transpose(1, 2).reshape(bsz, -1, self.num_key_value_heads * self.head_dim)

            if key_states.shape[1] <= self.residual_length:
                key_states_low = None

                key_states_high = key_states
            else:
                key_states_low = key_states[:, :-self.residual_length, -self.k_rank:]
                # key_states_low[:, :1536, :-self.residual_length] = 0
                key_states_high = key_states[:, -self.residual_length:, :]               


            if value_states.shape[-2] <= self.residual_length:
                value_states_quant = None
                value_states_full = value_states
                value_scale = None
                value_mn = None
            else:
                value_states_quant = value_states[:, :, :-self.residual_length, :].contiguous()
                value_states_full = value_states[:, :, -self.residual_length:, :].contiguous()
                value_states_quant, value_scale, value_mn = triton_quantize_and_pack_along_last_dim(value_states_quant, 
                                                                                                self.group_size, 
                                                                                                self.v_bits)

            if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len):
                raise ValueError(
                    f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                    f" {attn_weights.size()}"
                )

            if attention_mask is not None:
                if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
                    raise ValueError(
                        f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
                    )
                attn_weights = attn_weights + attention_mask
                attn_weights = torch.max(
                    attn_weights, torch.tensor(torch.finfo(attn_weights.dtype).min)
                )

            # upcast attention to fp32
            attn_weights = nn.functional.softmax(
                attn_weights, dim=-1, dtype=torch.float32
            ).to(query_states.dtype)

            value_states = repeat_kv(value_states, self.num_key_value_groups)
            attn_output = torch.matmul(attn_weights, value_states) 
        else:

            query_states = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
            query_states = self.q_rope(query_states, cos, sin, position_ids)
            key_states = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            assert self.is_compression == False, f"running with normal mode! is_compression is {str(self.is_compression)}"
            key_states = self.k_rope(key_states, cos, sin, position_ids)
            if past_key_value is not None:
                past_key, past_value, _ = past_key_value
                key_states = torch.cat([past_key, key_states], dim=2)
                value_states = torch.cat([past_value, value_states], dim=2)
            
            key_states_repeat = repeat_kv(key_states, self.num_key_value_groups)
            value_states_repeat = repeat_kv(value_states, self.num_key_value_groups)
            attn_weights = torch.matmul(query_states, key_states_repeat.transpose(2, 3)) / math.sqrt(self.head_dim)

            if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len):
                raise ValueError(
                    f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                    f" {attn_weights.size()}"
                )

            if attention_mask is not None:
                if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
                    raise ValueError(
                        f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
                    )
                attn_weights = attn_weights + attention_mask

            # upcast attention to fp32
            attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32).to(query_states.dtype)
            attn_output = torch.matmul(attn_weights, value_states_repeat)
            if attn_output.size() != (bsz, self.num_heads, q_len, self.head_dim):
                raise ValueError(
                    f"`attn_output` should be of size {(bsz, self.num_heads, q_len, self.head_dim)}, but is"
                    f" {attn_output.size()}"
                )
        if self.is_compression:
            # past_key_value = (key_states_low, key_states_high, value_states_quant, value_states_full, value_scale, value_mn, kv_seq_len) if use_cache else None
            # past_key_value = (key_states_low, key_states_high, value_states_quant, value_states_full, value_scale, value_mn, key_sparse, kv_seq_len) if use_cache else None
            past_key_value = (key_states_low, key_states_high, value_states_quant, value_states_full, value_scale, value_mn, key_states_origin, key_states_origin_without_rope, kv_seq_len) if use_cache else None
        else:
            past_key_value = (key_states, value_states, kv_seq_len) if use_cache else None
        if attn_output.size() != (bsz, self.num_heads, q_len, self.head_dim):
            raise ValueError(
                f"`attn_output` should be of size {(bsz, self.num_heads, q_len, self.head_dim)}, but is"
                f" {attn_output.size()}"
            )

        attn_output = attn_output.transpose(1, 2).contiguous()
        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size)

        attn_output = self.o_proj(attn_output)

        if not output_attentions:
            attn_weights = None

        return attn_output, attn_weights, past_key_value

        return attn_output, attn_weights, past_key_value

class LlamaDecoderLayer_SALS(nn.Module):
    def __init__(self, config: LlamaConfig, block_idx=0):
        super().__init__()
        self.hidden_size = config.hidden_size
        self.self_attn = ( 
            LlamaAttention_SALS(config=config, layer_idx=block_idx) 

        )
        self.mlp = LlamaMLP(config)
        self.input_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
        self.post_attention_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        **kwargs,
    ) -> Tuple[torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]]:
        """
        Args:
            hidden_states (`torch.FloatTensor`): input to the layer of shape `(batch, seq_len, embed_dim)`
            attention_mask (`torch.FloatTensor`, *optional*):
                attention mask of size `(batch_size, sequence_length)` if flash attention is used or `(batch_size, 1,
                query_sequence_length, key_sequence_length)` if default attention is used.
            output_attentions (`bool`, *optional*):
                Whether or not to return the attentions tensors of all attention layers. See `attentions` under
                returned tensors for more detail.
            use_cache (`bool`, *optional*):
                If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding
                (see `past_key_values`).
            past_key_value (`Tuple(torch.FloatTensor)`, *optional*): cached past key and value projection states
        """
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

        residual = hidden_states

        hidden_states = self.input_layernorm(hidden_states)

        # Self Attention
        hidden_states, self_attn_weights, present_key_value = self.self_attn(
            hidden_states=hidden_states,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_value=past_key_value,
            output_attentions=output_attentions,
            use_cache=use_cache,
            **kwargs,
        )
        hidden_states = residual + hidden_states

        # Fully Connected
        residual = hidden_states
        hidden_states = self.post_attention_layernorm(hidden_states)
        hidden_states = self.mlp(hidden_states)
        hidden_states = residual + hidden_states

        outputs = (hidden_states,)

        if output_attentions:
            outputs += (self_attn_weights,)

        if use_cache:
            outputs += (present_key_value,)

        return outputs

class LlamaModel_SALS(LlamaPreTrainedModel):
    """
    Transformer decoder consisting of *config.num_hidden_layers* layers. Each layer is a [`LlamaDecoderLayer`]

    Args:
        config: LlamaConfig
    """

    def __init__(self, config: LlamaConfig):
        super().__init__(config)
        self.padding_idx = config.pad_token_id
        self.vocab_size = config.vocab_size

        self.embed_tokens = nn.Embedding(config.vocab_size, config.hidden_size, self.padding_idx)
        self.layers = nn.ModuleList([LlamaDecoderLayer_SALS(config, block_idx=i) for i in range(config.num_hidden_layers)])
        self.norm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

        self.gradient_checkpointing = False
        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.embed_tokens

    def set_input_embeddings(self, value):
        self.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, BaseModelOutputWithPast]:
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache

        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        # retrieve input_ids and inputs_embeds
        if input_ids is not None and inputs_embeds is not None:
            raise ValueError("You cannot specify both input_ids and inputs_embeds at the same time")
        elif input_ids is not None:
            batch_size, seq_length = input_ids.shape[:2]
        elif inputs_embeds is not None:
            batch_size, seq_length = inputs_embeds.shape[:2]
        else:
            raise ValueError("You have to specify either input_ids or inputs_embeds")

        past_key_values_length = 0
        if past_key_values is not None:
            past_key_values_length = past_key_values[0][-1]

        if position_ids is None:
            device = input_ids.device if input_ids is not None else inputs_embeds.device
            position_ids = torch.arange(
                past_key_values_length, seq_length + past_key_values_length, dtype=torch.long, device=device
            )
            position_ids = position_ids.unsqueeze(0)

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)

        if getattr(self.config, "_flash_attn_2_enabled", False):
            # 2d mask is passed through the layers
            attention_mask = attention_mask if (attention_mask is not None and 0 in attention_mask) else None
        else:
            # 4d mask is passed through the layers
            attention_mask = _prepare_4d_causal_attention_mask(
                attention_mask, (batch_size, seq_length), inputs_embeds, past_key_values_length
            )

        # embed positions
        hidden_states = inputs_embeds
        # print("hidden_states", hidden_states.shape)
        # if hidden_states.shape[1] == 578:
        #     import pdb;pdb.set_trace()

        if self.gradient_checkpointing and self.training:
            if use_cache:
                logger.warning_once(
                    "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`..."
                )
                use_cache = False

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None
        next_decoder_cache = () if use_cache else None

        for idx, decoder_layer in enumerate(self.layers):
            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            past_key_value = past_key_values[idx] if past_key_values is not None else None

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    decoder_layer.__call__,
                    hidden_states,
                    attention_mask,
                    position_ids,
                    past_key_value,
                    output_attentions,
                    use_cache,
                )
            else:
                # print("idx", idx, flush=True)
                # print("Before operation X: ", hidden_states.shape, flush=True)
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=attention_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_value,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                )

            hidden_states = layer_outputs[0]
            # print("After operation X: ", hidden_states.shape, flush=True)

            if use_cache:
                next_decoder_cache += (layer_outputs[2 if output_attentions else 1],)

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        next_cache = next_decoder_cache if use_cache else None
        if not return_dict:
            return tuple(v for v in [hidden_states, next_cache, all_hidden_states, all_self_attns] if v is not None)
        return BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=next_cache,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
        )


class LlamaForCausalLM_SALS(LlamaPreTrainedModel):
    _tied_weights_keys = ["lm_head.weight"]

    def __init__(self, config):
        super().__init__(config)
        self.model = LlamaModel_SALS(config)
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    def get_output_embeddings(self):
        return self.lm_head

    def set_output_embeddings(self, new_embeddings):
        self.lm_head = new_embeddings

    def set_decoder(self, decoder):
        self.model = decoder

    def get_decoder(self):
        return self.model

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    @replace_return_docstrings(output_type=CausalLMOutputWithPast, config_class=_CONFIG_FOR_DOC)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        r"""
        Args:
            labels (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
                Labels for computing the masked language modeling loss. Indices should either be in `[0, ...,
                config.vocab_size]` or -100 (see `input_ids` docstring). Tokens with indices set to `-100` are ignored
                (masked), the loss is only computed for the tokens with labels in `[0, ..., config.vocab_size]`.

        Returns:

        Example:

        ```python
        >>> from transformers import AutoTokenizer, LlamaForCausalLM

        >>> model = LlamaForCausalLM.from_pretrained(PATH_TO_CONVERTED_WEIGHTS)
        >>> tokenizer = AutoTokenizer.from_pretrained(PATH_TO_CONVERTED_TOKENIZER)

        >>> prompt = "Hey, are you conscious? Can you talk to me?"
        >>> inputs = tokenizer(prompt, return_tensors="pt")

        >>> # Generate
        >>> generate_ids = model.generate(inputs.input_ids, max_length=30)
        >>> tokenizer.batch_decode(generate_ids, skip_special_tokens=True, clean_up_tokenization_spaces=False)[0]
        "Hey, are you conscious? Can you talk to me?\nI'm not conscious, but I can talk to you."
        ```"""
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
        outputs = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )

        hidden_states = outputs[0]
        # print(hidden_states.shape)
        if self.config.pretraining_tp > 1:
            lm_head_slices = self.lm_head.weight.split(self.vocab_size // self.config.pretraining_tp, dim=0)
            logits = [F.linear(hidden_states, lm_head_slices[i]) for i in range(self.config.pretraining_tp)]
            logits = torch.cat(logits, dim=-1)
        else:
            logits = self.lm_head(hidden_states)
        logits = logits.float()

        loss = None
        if labels is not None:
            # Shift so that tokens < n predict n
            shift_logits = logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            shift_logits = shift_logits.view(-1, self.config.vocab_size)
            shift_labels = shift_labels.view(-1)
            # Enable model parallelism
            shift_labels = shift_labels.to(shift_logits.device)
            loss = loss_fct(shift_logits, shift_labels)

        # torch.save(attn_output_dic, "sparse_attn_output_dic.pt")
        if not return_dict:
            output = (logits,) + outputs[1:]
            return (loss,) + output if loss is not None else output
        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=outputs.past_key_values,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )

    def prepare_inputs_for_generation(
        self, input_ids, past_key_values=None, attention_mask=None, inputs_embeds=None, **kwargs
    ):
        if past_key_values is not None:
            past_length = past_key_values[0][-1]
            # Some generation methods already pass only the last input ID
            if input_ids.shape[1] > past_length:
                remove_prefix_length = past_length
            else:
                # Default to old behavior: keep only final ID
                remove_prefix_length = input_ids.shape[1] - 1

            input_ids = input_ids[:, remove_prefix_length:]

        position_ids = kwargs.get("position_ids", None)
        if attention_mask is not None and position_ids is None:
            # create position_ids on the fly for batch generation
            position_ids = attention_mask.long().cumsum(-1) - 1
            position_ids.masked_fill_(attention_mask == 0, 1)
            if past_key_values:
                position_ids = position_ids[:, -input_ids.shape[1] :]

        # if `inputs_embeds` are passed, we only want to use them in the 1st generation step
        if inputs_embeds is not None and past_key_values is None:
            model_inputs = {"inputs_embeds": inputs_embeds}
        else:
            model_inputs = {"input_ids": input_ids}

        model_inputs.update(
            {
                "position_ids": position_ids,
                "past_key_values": past_key_values,
                "use_cache": kwargs.get("use_cache"),
                "attention_mask": attention_mask,
            }
        )
        return model_inputs

    @staticmethod
    def _reorder_cache(past_key_values, beam_idx):
        reordered_past = ()
        for layer_past in past_key_values:
            reordered_past += (
                tuple(past_state.index_select(0, beam_idx.to(past_state.device)) for past_state in layer_past),
            )
        return reordered_past