# coding=utf-8
# Copyright 2023 Mistral AI and the HuggingFace Inc. team. All rights reserved.
#
# This code is based on EleutherAI's GPT-NeoX library and the GPT-NeoX
# and OPT implementations in this library. It has been modified from its
# original forms to accommodate minor architectural differences compared
# to GPT-NeoX and OPT used by the Meta AI team that trained the model.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" PyTorch Mistral model."""
import inspect
import math
import warnings
from typing import List, Optional, Tuple, Union

import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch import nn
from torch.nn import BCEWithLogitsLoss, CrossEntropyLoss, MSELoss

from quant.new_pack import triton_quantize_and_pack_along_last_dim
from quant.matmul import cuda_bmm_fA_qB_outer
from compression.fused_matmul import fused_recover_attn, torch_recover_attn

from transformers.models.mistral.configuration_mistral import *
from transformers.models.mistral.modeling_mistral import *
from transformers.modeling_attn_mask_utils import _prepare_4d_causal_attention_mask
from .module import *

_CONFIG_FOR_DOC = "MistralConfig"
from utils import timer

if is_flash_attn_2_available():
    from flash_attn import flash_attn_func, flash_attn_varlen_func
    from flash_attn.bert_padding import index_first_axis, pad_input, unpad_input  # noqa

    _flash_supports_window_size = "window_size" in list(inspect.signature(flash_attn_func).parameters)

def _get_unpad_data(attention_mask):
    seqlens_in_batch = attention_mask.sum(dim=-1, dtype=torch.int32)
    indices = torch.nonzero(attention_mask.flatten(), as_tuple=False).flatten()
    max_seqlen_in_batch = seqlens_in_batch.max().item()
    cu_seqlens = F.pad(torch.cumsum(seqlens_in_batch, dim=0, dtype=torch.torch.int32), (1, 0))
    return (
        indices,
        cu_seqlens,
        max_seqlen_in_batch,
    )



class MistralAttention_SALS(nn.Module):
    """
    Multi-headed attention from 'Attention Is All You Need' paper. Modified to use sliding window attention: Longformer
    and "Generating Long Sequences with Sparse Transformers".
    """

    def __init__(self, config: MistralConfig, layer_idx=0):
        super().__init__()
        self.layer_idx = layer_idx
        self.config = config
        self.hidden_size = config.hidden_size
        self.num_heads = config.num_attention_heads
        self.head_dim = self.hidden_size // self.num_heads
        self.num_key_value_heads = config.num_key_value_heads
        self.num_key_value_groups = self.num_heads // self.num_key_value_heads
        self.max_position_embeddings = config.max_position_embeddings
        self.rope_theta = config.rope_theta
        self.is_causal = True
        self.is_compression = config.is_compress
        # import pdb;pdb.set_trace()
        if self.is_compression:
            self.k_rank = config.k_rank
            self.k_high_rank = config.k_high_rank
            self.v_bits = config.v_bits
            self.group_size = config.group_size
            self.residual_length = config.residual_length
            self.sparsity = config.sparsity
            self.skip_layers = config.skip_layers.split(",")
            self.budget = config.budget
            self.sink_length = 0
            self.topk_rank = config.topk_rank
            self.with_overlap = config.with_overlap
            if self.budget is not None:
                self.sink_length = self.budget[0]
                self.sparsity = self.budget[1]
                self.residual_length = self.budget[2]

        if (self.head_dim * self.num_heads) != self.hidden_size:
            raise ValueError(
                f"hidden_size must be divisible by num_heads (got `hidden_size`: {self.hidden_size}"
                f" and `num_heads`: {self.num_heads})."
            )
        if self.is_compression:
            # self.sparse_proj = sparse_project(self.num_key_value_heads, self.head_dim, self.sparse_rank)
            # self.sparse_proj = sparse_project(self.num_key_value_heads, self.head_dim, self.sparse_rank)
            self.k_down_proj = nn.Linear(self.num_key_value_heads * self.head_dim, self.k_high_rank, bias=False)
            self.k_up_proj = nn.Linear(self.k_high_rank, self.num_key_value_heads * self.head_dim, bias=True)
        self.q_proj = nn.Linear(self.hidden_size, self.num_heads * self.head_dim, bias=False)
        self.k_proj = nn.Linear(self.hidden_size, self.num_key_value_heads * self.head_dim, bias=False)
        self.v_proj = nn.Linear(self.hidden_size, self.num_key_value_heads * self.head_dim, bias=False)
        self.o_proj = nn.Linear(self.num_heads * self.head_dim, self.hidden_size, bias=False)
        self.k_rope = LlamaRope()
        self.q_rope = LlamaRope()
        self.rotary_emb = MistralRotaryEmbedding(
            self.head_dim,
            max_position_embeddings=self.max_position_embeddings,
            base=self.rope_theta,
        )
        # print("using elegent!")


    def _shape(self, tensor: torch.Tensor, seq_len: int, bsz: int):
        return tensor.view(bsz, seq_len, self.num_heads, self.head_dim).transpose(1, 2).contiguous()

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )
        bsz, q_len, _ = hidden_states.size()

        query_states = self.q_proj(hidden_states)
        key_states = self.k_proj(hidden_states)
        value_states = self.v_proj(hidden_states)

        # key_states = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
        value_states = value_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)

        kv_seq_len = key_states.shape[-2]
        if past_key_value is not None:
            kv_seq_len += past_key_value[-1]
        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)
        # query_states, key_states = apply_rotary_pos_emb(query_states, key_states, cos, sin, position_ids)
        # import pdb;pdb.set_trace()
        if self.is_compression and past_key_value is not None:
                        # import pdb;pdb.set_trace()
            key_states_low = past_key_value[0]
            key_states_high = past_key_value[1]
            value_states_quant = past_key_value[2]
            value_states_full = past_key_value[3]
            value_scale = past_key_value[4]
            value_mn = past_key_value[5]
            key_states_origin = past_key_value[6]
            key_states_origin_without_rope = past_key_value[7]
            # key_sparse = past_key_value[8]

            query_states_before_rope = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
            query_states = self.q_rope(query_states_before_rope, cos, sin, position_ids)
            key_states_reshape = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            key_states_origin_new = self.k_rope(key_states_reshape, cos, sin, position_ids)
            key_states_origin = torch.cat([key_states_origin, key_states_origin_new], dim=2)
            key_states_origin_without_rope = torch.cat([key_states_origin_without_rope, key_states_reshape], dim=2)

            # query_sparse = self.sparse_proj(query_states)
            # key_sparse_new = self.sparse_proj(key_states_origin_new)
            # key_sparse = torch.cat([key_sparse, key_sparse_new], dim=2)

            # import pdb;pdb.set_trace()

            



            # import pdb;pdb.set_trace()


            key_states = self.k_down_proj(key_states)
            if key_states_low is not None:
                query_states_to_sparse = query_states_before_rope.view(bsz, q_len, self.num_key_value_heads, -1, self.head_dim).transpose(2, 3).reshape(bsz, q_len, self.num_key_value_groups, -1).mean(dim=2, keepdim=True)
                query_sparse = torch.einsum("bshd,dr->bhsr", query_states_to_sparse, self.k_up_proj.weight[:, -self.topk_rank:])
                # query_sparse = self.q_rope(query_sparse, cos, sin, position_ids)
                key_sparse = key_states_low[:, :, -self.topk_rank:]
                key_sparse = key_sparse.unsqueeze(1)
                lowrank_seq_length = key_states_low.shape[1]
                # key_sparse = self.k_rope(key_sparse, cos, sin, position_ids_low)

                if str(self.layer_idx) not in self.skip_layers:
                    # import pdb;pdb.set_trace()
                # indices = generate_sparse_indices_label_single_head(query_sparse, key_states_low[:, :, :self.sparse_rank], self.sparsity)[0]
                    indices, _ = generate_sparse_indices_label_single_head(query_sparse, key_sparse[:, :, self.sink_length:key_states_low.shape[1], :], self.sparsity, self.withsoftmax)
                    if self.with_overlap:
                        indices_true, scores = generate_sparse_indices_label(query_states, key_states_origin[:, :, self.sink_length:key_states_low.shape[1], :], self.sparsity, self.withsoftmax)

                        overlap = percent_scores(indices, indices_true, scores)
                        with open(f"overlap_SALS.csv", "a") as f:
                        # with open(f"overlap_32_other.csv", "a") as f:
                            f.write(f"{self.layer_idx},{kv_seq_len},{overlap}\n")
                    # indices_true, scores = generate_sparse_indices_label(query_states, key_states_origin[:, :, self.sink_length:key_states_low.shape[1], :], self.sparsity, self.withsoftmax)

                    # overlap = percent_scores(indices, indices_true, scores)
                    # with open(f"mistral_overlap_without_rope_{self.sparsity}.csv", "a") as f:
                    # # with open(f"overlap_32_other.csv", "a") as f:
                    #     f.write(f"{self.layer_idx},{kv_seq_len},{overlap}\n")
                    indices += self.sink_length
                    indices = torch.cat([torch.arange(self.sink_length, device=indices.device), indices]) if self.sink_length != 0 else indices
                    # indices = None
                # import pdb;pdb.set_trace()
                    # indices_true, scores = generate_sparse_indices_label(query_states, key_states_origin[:, :, :key_states_low.shape[1], :], self.sparsity)
                    # # indices_true_without_rope, scores_without_rope = generate_sparse_indices_label(query_states_before_rope, key_states_origin_without_rope[:, :, :key_states_low.shape[1], :], self.sparsity)
                    # # # import pdb;pdb.set_trace()
                    # # overlap = overlap_ratio_single_head(indices_true_without_rope, indices_true)
                    # # overlap = overlap_ratio_single_head(indices, indices_true)
                    # overlap = percent_scores(indices, indices_true, scores)
                    # # overlap = percent_scores(indices_true_without_rope, indices_true, scores)
                    # # # overlap = overlap_ratio_by_tensor(scores, scores_true)
                    # # # loss = torch.nn.functional.mse_loss(scores, scores_true)
                    # with open(f"overlap_without_rope_{self.sparsity}.csv", "a") as f:
                    # with open(f"overlap_32_other.csv", "a") as f:
                        # f.write(f"{self.layer_idx},{kv_seq_len},{overlap}\n")
                else:
                    indices = None
                    
                key_states_low_to_recover = key_states_low[:, indices, :] if indices is not None else key_states_low
                
                
                if indices is not None:
                    # position_ids = position_ids[:, indices]
                    position_ids_low = indices.to(position_ids.device).expand(bsz, -1)
                else:
                    position_ids_low = torch.arange(lowrank_seq_length, device=position_ids.device).expand(bsz, -1)
                attn_lowrank = torch_recover_attn(query_states, key_states_low_to_recover, 
                                                  self.k_up_proj.weight[:, self.k_high_rank - self.k_rank:], self.k_up_proj.bias, 
                                                  cos[position_ids_low],
                                                  sin[position_ids_low],
                                                  self.num_key_value_groups)
                # attn_lowrank = fused_recover_attn(query_states, key_states_low_to_recover, 
                #                                 self.k_up_proj.weight[:, self.k_high_rank - self.k_rank:], self.k_up_proj.bias, 
                #                                 cos[position_ids_low],
                #                                 sin[position_ids_low])
                # import pdb;pdb.set_trace()
            else:
                attn_lowrank = None

            if key_states_high is not None:
                key_states_high = torch.cat([key_states_high, key_states], dim=1)
            else:
                key_states_high = key_states
            
            k_full_length = key_states_high.shape[1]
            # print("kvseqlen k_full length", kv_seq_len, k_full_length)
            # if self.block_idx == 8:
            #     import pdb;pdb.set_trace()
            position_ids_high = torch.arange(kv_seq_len - k_full_length, kv_seq_len, device=position_ids.device).expand(bsz, -1)
            
            # if self.config.pretraining_tp > 1:
            #     key_value_slicing = (self.num_key_value_heads * self.head_dim) // self.config.pretraining_tp
            #     key_slices = self.k_up_proj.weight.split(key_value_slicing, dim=0)
            #     key_high_recover = [F.linear(key_states_high, key_slices[i], self.k_up_proj.bias) for i in range(self.config.pretraining_tp)]
            #     key_high_recover = torch.cat(key_high_recover, dim=-1)
            # else:
            #     key_high_recover = self.k_up_proj(key_states_high)
            # key_high_recover = key_high_recover.view(bsz, -1, self.num_key_value_heads, self.head_dim).transpose(1, 2)

            # key_states_high_rope = self.rope(key_high_recover, cos, sin, position_ids)
            # attn_highrank = torch.matmul(query_states, key_states_high_rope.transpose(2, 3))
            # print(position_ids_high, cos.shape)
            # print(cos[1023])
            # attn_highrank = fused_recover_attn(query_states, key_states_high, 
            #                     self.k_up_proj.weight, self.k_up_proj.bias, 
            #                     cos[position_ids_high],
            #                     sin[position_ids_high])            
            # print(self.block_idx)
            attn_highrank = torch_recover_attn(query_states, key_states_high, 
                                                self.k_up_proj.weight, self.k_up_proj.bias, 
                                                cos[position_ids_high],
                                                sin[position_ids_high],
                                                self.num_key_value_groups)    

            # import pdb;pdb.set_trace()
            if attn_lowrank is not None:
                attn_weights = torch.cat([attn_lowrank, attn_highrank], dim=-1) / math.sqrt(self.head_dim)
            else:
                attn_weights = attn_highrank / math.sqrt(self.head_dim)
#
            # position_ids = torch.arange(kv_seq_len, device=position_ids.device).expand(bsz, -1)
            # print(position_ids)
            # key_states_recover = self.rope(key_states_recover, cos, sin, position_ids)
            # attn_weights = torch.matmul(query_states, key_states_recover.transpose(2, 3)) / math.sqrt(self.head_dim)
            
            key_high_length = key_states_high.shape[1]
            if key_high_length > self.residual_length:
                assert self.residual_length + 1 == key_high_length
                key_states_low_new = key_states_high[:, :1, self.k_high_rank - self.k_rank:]
                key_states_high = key_states_high[:, 1:, :]
                if key_states_low is not None:
                    key_states_low = torch.cat([key_states_low, key_states_low_new], dim=1)
                else:
                    key_states_low = key_states_low_new

            kv_seq_len_sparse = len(indices) + self.residual_length + 1 if indices is not None else kv_seq_len
            if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len_sparse):
                raise ValueError(
                    f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                    f" {attn_weights.size()}"
                )
            # import pdb;pdb.set_trace()
            if attention_mask is not None:
                attention_mask = torch.cat([attention_mask[..., indices], attention_mask[..., -(self.residual_length + 1):]], dim=-1) if indices is not None else attention_mask
                if attention_mask.size() != (bsz, 1, q_len, kv_seq_len_sparse):
                    raise ValueError(
                        f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len_sparse)}, but is {attention_mask.size()}"
                    )
                attn_weights = attn_weights + attention_mask
                attn_weights = torch.max(
                    attn_weights, torch.tensor(torch.finfo(attn_weights.dtype).min)
                )

            # upcast attention to fp32
            attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32).to(query_states.dtype)
            # import pdb;pdb.set_trace()
            value_states_full = torch.cat([value_states_full, value_states], dim=2)
            value_full_length = value_states_full.shape[-2]
            if value_states_quant is None:
                value_states_full_repeat = repeat_kv(value_states_full, self.num_key_value_groups)
                attn_output = torch.matmul(attn_weights, value_states_full_repeat)
            else:
                value_states_quant_selected = value_states_quant[:, :, indices, :] if indices is not None else value_states_quant
                value_scale_selected = value_scale[:, :, indices, :] if indices is not None else value_scale
                value_mn_selected = value_mn[:, :, indices, :] if indices is not None else value_mn
                value_states_quant_repeat = repeat_kv(value_states_quant_selected, self.num_key_value_groups)
                value_scale_repeat = repeat_kv(value_scale_selected, self.num_key_value_groups)
                value_mn_repeat = repeat_kv(value_mn_selected, self.num_key_value_groups)
                value_states_full_repeat = repeat_kv(value_states_full, self.num_key_value_groups)
                attn_output = cuda_bmm_fA_qB_outer(self.group_size, attn_weights[:, :, :, :-value_full_length], value_states_quant_repeat, 
                                                value_scale_repeat, value_mn_repeat, self.v_bits)
                attn_output += torch.matmul(attn_weights[:, :, :, -value_full_length:], value_states_full_repeat)
            
            if value_full_length > self.residual_length:
                assert value_full_length == self.residual_length + 1
                value_states_quant_new, scale, mn = triton_quantize_and_pack_along_last_dim(value_states_full[:, :, :1, :].contiguous(), 
                                                                                                self.group_size, 
                                                                                                self.v_bits)
                value_states_full = value_states_full[:, :, 1:, :].contiguous()
                if value_states_quant is not None:
                    value_states_quant = torch.cat([value_states_quant, value_states_quant_new], dim=2)
                    value_scale = torch.cat([value_scale, scale], dim=2)
                    value_mn = torch.cat([value_mn, mn], dim=2)
                else:
                    value_states_quant = value_states_quant_new
                    value_scale = scale
                    value_mn = mn
            # if self.block_idx == 0:
            #     attn_output_dic.append(dict())
            # attn_output_dic[-1][self.block_idx] = attn_output

        elif self.is_compression:

            query_states = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
            query_states = self.q_rope(query_states, cos, sin, position_ids)
            key_states_to_rope = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            key_states_high_rope = self.k_rope(key_states_to_rope, cos, sin, position_ids)
            key_states_origin = key_states_high_rope
            key_states_origin_without_rope = key_states_to_rope
            # key_sparse = self.sparse_proj(key_states_high_rope)

            key_states = self.k_down_proj(key_states)
            # key_high_recover = self.k_up_proj(key_states)
            # key_high_recover = key_states.contiguous()
            # key_states = self.k_down_proj(key_states)
            
            # key_high_recover = key_high_recover.view(bsz, -1, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            # key_states_high_rope = self.k_rope(key_high_recover, cos, sin, position_ids)
            key_states_repeat = repeat_kv(key_states_high_rope, self.num_key_value_groups)
            value_states_repeat = repeat_kv(value_states, self.num_key_value_groups)
            attn_output = self._flash_attention_forward(
                query_states.transpose(1, 2), key_states_repeat.transpose(1, 2), 
                value_states_repeat.transpose(1, 2), None, q_len, dropout=0.0
            )

            # key_states_high_rope = repeat_kv(key_states_high_rope, self.num_key_value_groups)
            # attn_weights = torch.matmul(query_states, 
            #                             key_states_high_rope.transpose(2, 3)) / math.sqrt(self.head_dim)
            # import pdb;pdb.set_trace()
            # key_states = key_states.transpose(1, 2).reshape(bsz, -1, self.num_key_value_heads * self.head_dim)

            if key_states.shape[1] <= self.residual_length:
                key_states_low = None

                key_states_high = key_states
            else:
                key_states_low = key_states[:, :-self.residual_length, -self.k_rank:]
                # key_states_low[:, :1536, :-self.residual_length] = 0
                key_states_high = key_states[:, -self.residual_length:, :]               


            if value_states.shape[-2] <= self.residual_length:
                value_states_quant = None
                value_states_full = value_states
                value_scale = None
                value_mn = None
            else:
                value_states_quant = value_states[:, :, :-self.residual_length, :].contiguous()
                value_states_full = value_states[:, :, -self.residual_length:, :].contiguous()
                value_states_quant, value_scale, value_mn = triton_quantize_and_pack_along_last_dim(value_states_quant, 
                                                                                                self.group_size, 
                                                                                                self.v_bits)

        else:

            query_states = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
            query_states = self.q_rope(query_states, cos, sin, position_ids)
            key_states = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
            assert self.is_compression == False, f"running with normal mode! is_compression is {str(self.is_compression)}"
            key_states = self.k_rope(key_states, cos, sin, position_ids)
            if past_key_value is not None:
                past_key, past_value, _ = past_key_value
                key_states = torch.cat([past_key, key_states], dim=2)
                value_states = torch.cat([past_value, value_states], dim=2)
            
            key_states_repeat = repeat_kv(key_states, self.num_key_value_groups)
            value_states_repeat = repeat_kv(value_states, self.num_key_value_groups)
            attn_weights = torch.matmul(query_states, key_states_repeat.transpose(2, 3)) / math.sqrt(self.head_dim)

            if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len):
                raise ValueError(
                    f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                    f" {attn_weights.size()}"
                )

            if attention_mask is not None:
                if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
                    raise ValueError(
                        f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
                    )
                attn_weights = attn_weights + attention_mask

            # upcast attention to fp32
            attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32).to(query_states.dtype)
            attn_output = torch.matmul(attn_weights, value_states_repeat)
            if attn_output.size() != (bsz, self.num_heads, q_len, self.head_dim):
                raise ValueError(
                    f"`attn_output` should be of size {(bsz, self.num_heads, q_len, self.head_dim)}, but is"
                    f" {attn_output.size()}"
                )
        if self.is_compression:
            # past_key_value = (key_states_low, key_states_high, value_states_quant, value_states_full, value_scale, value_mn, kv_seq_len) if use_cache else None
            # past_key_value = (key_states_low, key_states_high, value_states_quant, value_states_full, value_scale, value_mn, key_sparse, kv_seq_len) if use_cache else None
            past_key_value = (key_states_low, key_states_high, value_states_quant, value_states_full, value_scale, value_mn, key_states_origin, key_states_origin_without_rope, kv_seq_len) if use_cache else None
        else:
            past_key_value = (key_states, value_states, kv_seq_len) if use_cache else None

        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size)

        attn_output = self.o_proj(attn_output)

        if not output_attentions:
            attn_weights = None

        return attn_output, attn_weights, past_key_value

    def _flash_attention_forward(
        self,
        query_states,
        key_states,
        value_states,
        attention_mask,
        query_length,
        dropout=0.0,
        softmax_scale=None,
        use_sliding_windows=False,
    ):
        """
        Calls the forward method of Flash Attention - if the input hidden states contain at least one padding token
        first unpad the input, then computes the attention scores and pad the final attention scores.

        Args:
            query_states (`torch.Tensor`):
                Input query states to be passed to Flash Attention API
            key_states (`torch.Tensor`):
                Input key states to be passed to Flash Attention API
            value_states (`torch.Tensor`):
                Input value states to be passed to Flash Attention API
            attention_mask (`torch.Tensor`):
                The padding mask - corresponds to a tensor of size `(batch_size, seq_len)` where 0 stands for the
                position of padding tokens and 1 for the position of non-padding tokens.
            dropout (`int`, *optional*):
                Attention dropout
            softmax_scale (`float`, *optional*):
                The scaling of QK^T before applying softmax. Default to 1 / sqrt(head_dim)
            use_sliding_windows (`bool`, *optional*):
                Whether to activate sliding window attention.
        """
        # Contains at least one padding token in the sequence
        if attention_mask is not None:
            batch_size = query_states.shape[0]
            query_states, key_states, value_states, indices_q, cu_seq_lens, max_seq_lens = self._upad_input(
                query_states, key_states, value_states, attention_mask, query_length
            )

            cu_seqlens_q, cu_seqlens_k = cu_seq_lens
            max_seqlen_in_batch_q, max_seqlen_in_batch_k = max_seq_lens

            if not use_sliding_windows:
                attn_output_unpad = flash_attn_varlen_func(
                    query_states,
                    key_states,
                    value_states,
                    cu_seqlens_q=cu_seqlens_q,
                    cu_seqlens_k=cu_seqlens_k,
                    max_seqlen_q=max_seqlen_in_batch_q,
                    max_seqlen_k=max_seqlen_in_batch_k,
                    dropout_p=dropout,
                    softmax_scale=softmax_scale,
                    causal=self.is_causal,
                )
            else:
                attn_output_unpad = flash_attn_varlen_func(
                    query_states,
                    key_states,
                    value_states,
                    cu_seqlens_q=cu_seqlens_q,
                    cu_seqlens_k=cu_seqlens_k,
                    max_seqlen_q=max_seqlen_in_batch_q,
                    max_seqlen_k=max_seqlen_in_batch_k,
                    dropout_p=dropout,
                    softmax_scale=softmax_scale,
                    causal=self.is_causal,
                    window_size=(self.config.sliding_window, self.config.sliding_window),
                )

            attn_output = pad_input(attn_output_unpad, indices_q, batch_size, query_length)
        else:
            if not use_sliding_windows:
                attn_output = flash_attn_func(
                    query_states,
                    key_states,
                    value_states,
                    dropout,
                    softmax_scale=softmax_scale,
                    causal=self.is_causal,
                )
            else:
                attn_output = flash_attn_func(
                    query_states,
                    key_states,
                    value_states,
                    dropout,
                    softmax_scale=softmax_scale,
                    causal=self.is_causal,
                    window_size=(self.config.sliding_window, self.config.sliding_window),
                )

        return attn_output





class MistralDecoderLayer_SALS(nn.Module):
    def __init__(self, config: MistralConfig, layer_idx=0):
        super().__init__()
        self.hidden_size = config.hidden_size
        self.self_attn = (
            MistralAttention_SALS(config=config, layer_idx=layer_idx)
        )
        self.mlp = MistralMLP(config)
        self.input_layernorm = MistralRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
        self.post_attention_layernorm = MistralRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        **kwargs,
    ) -> Tuple[torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]]:
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )
        """
        Args:
            hidden_states (`torch.FloatTensor`): input to the layer of shape `(batch, seq_len, embed_dim)`
            attention_mask (`torch.FloatTensor`, *optional*): attention mask of size
                `(batch, sequence_length)` where padding elements are indicated by 0.
            output_attentions (`bool`, *optional*):
                Whether or not to return the attentions tensors of all attention layers. See `attentions` under
                returned tensors for more detail.
            use_cache (`bool`, *optional*):
                If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding
                (see `past_key_values`).
            past_key_value (`Tuple(torch.FloatTensor)`, *optional*): cached past key and value projection states
        """

        residual = hidden_states

        hidden_states = self.input_layernorm(hidden_states)

        # Self Attention
        hidden_states, self_attn_weights, present_key_value = self.self_attn(
            hidden_states=hidden_states,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_value=past_key_value,
            output_attentions=output_attentions,
            use_cache=use_cache,
        )
        hidden_states = residual + hidden_states

        # Fully Connected
        residual = hidden_states
        hidden_states = self.post_attention_layernorm(hidden_states)
        hidden_states = self.mlp(hidden_states)
        hidden_states = residual + hidden_states

        outputs = (hidden_states,)

        if output_attentions:
            outputs += (self_attn_weights,)

        if use_cache:
            outputs += (present_key_value,)

        return outputs


@add_start_docstrings(
    "The bare Mistral Model outputting raw hidden-states without any specific head on top.",
    MISTRAL_START_DOCSTRING,
)
class MistralModel_SALS(MistralPreTrainedModel):
    """
    Transformer decoder consisting of *config.num_hidden_layers* layers. Each layer is a [`MistralDecoderLayer`]

    Args:
        config: MistralConfig
    """

    def __init__(self, config: MistralConfig):
        super().__init__(config)
        self.padding_idx = config.pad_token_id
        self.vocab_size = config.vocab_size

        self.embed_tokens = nn.Embedding(config.vocab_size, config.hidden_size, self.padding_idx)
        self.layers = nn.ModuleList([MistralDecoderLayer_SALS(config, _) for _ in range(config.num_hidden_layers)])
        self.norm = MistralRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

        self.gradient_checkpointing = False
        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.embed_tokens

    def set_input_embeddings(self, value):
        self.embed_tokens = value

    @add_start_docstrings_to_model_forward(MISTRAL_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, BaseModelOutputWithPast]:
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache

        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        # retrieve input_ids and inputs_embeds
        if input_ids is not None and inputs_embeds is not None:
            raise ValueError("You cannot specify both decoder_input_ids and decoder_inputs_embeds at the same time")
        elif input_ids is not None:
            batch_size, seq_length = input_ids.shape
        elif inputs_embeds is not None:
            batch_size, seq_length, _ = inputs_embeds.shape
        else:
            raise ValueError("You have to specify either decoder_input_ids or decoder_inputs_embeds")

        seq_length_with_past = seq_length
        past_key_values_length = 0

        if past_key_values is not None:
            # past_key_values_length = past_key_values[0][0].shape[2]
            past_key_values_length = past_key_values[0][-1]

            seq_length_with_past = seq_length_with_past + past_key_values_length

        if position_ids is None:
            device = input_ids.device if input_ids is not None else inputs_embeds.device
            position_ids = torch.arange(
                past_key_values_length, seq_length + past_key_values_length, dtype=torch.long, device=device
            )
            position_ids = position_ids.unsqueeze(0).view(-1, seq_length)
        else:
            position_ids = position_ids.view(-1, seq_length).long()

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)

        if (
            attention_mask is not None
            and hasattr(self.config, "_flash_attn_2_enabled")
            and self.config._flash_attn_2_enabled
            and past_key_values is not None
        ):
            is_padding_right = attention_mask[:, -1].sum().item() != batch_size
            if is_padding_right:
                raise ValueError(
                    "You are attempting to perform batched generation with padding_side='right'"
                    " this may lead to unexpected behaviour for Flash Attention version of Mistral. Make sure to "
                    " call `tokenizer.padding_side  = 'left'` before tokenizing the input. "
                )

        if getattr(self.config, "_flash_attn_2_enabled", False):
            # 2d mask is passed through the layers
            attention_mask = attention_mask if (attention_mask is not None and 0 in attention_mask) else None
        else:
            # 4d mask is passed through the layers
            attention_mask = _prepare_4d_causal_attention_mask(
                attention_mask,
                (batch_size, seq_length),
                inputs_embeds,
                past_key_values_length,
                sliding_window=self.config.sliding_window,
            )

        hidden_states = inputs_embeds

        if self.gradient_checkpointing and self.training:
            if use_cache:
                logger.warning_once(
                    "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`..."
                )
                use_cache = False

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None
        next_decoder_cache = () if use_cache else None
        # import pdb;pdb.set_trace()
        for idx, decoder_layer in enumerate(self.layers):
            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            past_key_value = past_key_values[idx] if past_key_values is not None else None

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    decoder_layer.__call__,
                    hidden_states,
                    attention_mask,
                    position_ids,
                    past_key_value,
                    output_attentions,
                    use_cache,
                )
            else:
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=attention_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_value,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                )

            hidden_states = layer_outputs[0]

            if use_cache:
                next_decoder_cache += (layer_outputs[2 if output_attentions else 1],)

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        next_cache = next_decoder_cache if use_cache else None
        if not return_dict:
            return tuple(v for v in [hidden_states, next_cache, all_hidden_states, all_self_attns] if v is not None)
        return BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=next_cache,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
        )


class MistralForCausalLM_SALS(MistralPreTrainedModel):
    _tied_weights_keys = ["lm_head.weight"]

    def __init__(self, config):
        super().__init__(config)
        self.model = MistralModel_SALS(config)
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    def get_output_embeddings(self):
        return self.lm_head

    def set_output_embeddings(self, new_embeddings):
        self.lm_head = new_embeddings

    def set_decoder(self, decoder):
        self.model = decoder

    def get_decoder(self):
        return self.model

    @add_start_docstrings_to_model_forward(MISTRAL_INPUTS_DOCSTRING)
    @replace_return_docstrings(output_type=CausalLMOutputWithPast, config_class=_CONFIG_FOR_DOC)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        r"""
        Args:
            labels (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
                Labels for computing the masked language modeling loss. Indices should either be in `[0, ...,
                config.vocab_size]` or -100 (see `input_ids` docstring). Tokens with indices set to `-100` are ignored
                (masked), the loss is only computed for the tokens with labels in `[0, ..., config.vocab_size]`.

        Returns:

        Example:

        ```python
        >>> from transformers import AutoTokenizer, MistralForCausalLM

        >>> model = MistralForCausalLM.from_pretrained(PATH_TO_CONVERTED_WEIGHTS)
        >>> tokenizer = AutoTokenizer.from_pretrained(PATH_TO_CONVERTED_TOKENIZER)

        >>> prompt = "Hey, are you conscious? Can you talk to me?"
        >>> inputs = tokenizer(prompt, return_tensors="pt")

        >>> # Generate
        >>> generate_ids = model.generate(inputs.input_ids, max_length=30)
        >>> tokenizer.batch_decode(generate_ids, skip_special_tokens=True, clean_up_tokenization_spaces=False)[0]
        "Hey, are you conscious? Can you talk to me?\nI'm not conscious, but I can talk to you."
        ```"""

        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
        outputs = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        # import pdb;pdb.set_trace()
        hidden_states = outputs[0]
        logits = self.lm_head(hidden_states)
        logits = logits.float()

        loss = None
        if labels is not None:
            # Shift so that tokens < n predict n
            shift_logits = logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            shift_logits = shift_logits.view(-1, self.config.vocab_size)
            shift_labels = shift_labels.view(-1)
            # Enable model parallelism
            shift_labels = shift_labels.to(shift_logits.device)
            loss = loss_fct(shift_logits, shift_labels)

        if not return_dict:
            output = (logits,) + outputs[1:]
            return (loss,) + output if loss is not None else output

        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=outputs.past_key_values,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )

    def prepare_inputs_for_generation(
        self, input_ids, past_key_values=None, attention_mask=None, inputs_embeds=None, **kwargs
    ):
        # Omit tokens covered by past_key_values
        if past_key_values:
            # past_length = past_key_values[0][0].shape[2]
            past_length = past_key_values[0][-1]

            # Some generation methods already pass only the last input ID
            if input_ids.shape[1] > past_length:
                remove_prefix_length = past_length
            else:
                # Default to old behavior: keep only final ID
                remove_prefix_length = input_ids.shape[1] - 1

            input_ids = input_ids[:, remove_prefix_length:]

        position_ids = kwargs.get("position_ids", None)
        if attention_mask is not None and position_ids is None:
            # create position_ids on the fly for batch generation
            position_ids = attention_mask.long().cumsum(-1) - 1
            position_ids.masked_fill_(attention_mask == 0, 1)
            if past_key_values:
                position_ids = position_ids[:, -input_ids.shape[1] :]

        # if `inputs_embeds` are passed, we only want to use them in the 1st generation step
        if inputs_embeds is not None and past_key_values is None:
            model_inputs = {"inputs_embeds": inputs_embeds}
        else:
            model_inputs = {"input_ids": input_ids}

        model_inputs.update(
            {
                "position_ids": position_ids,
                "past_key_values": past_key_values,
                "use_cache": kwargs.get("use_cache"),
                "attention_mask": attention_mask,
            }
        )
        return model_inputs

    @staticmethod
    def _reorder_cache(past_key_values, beam_idx):
        reordered_past = ()
        for layer_past in past_key_values:
            reordered_past += (
                tuple(past_state.index_select(0, beam_idx.to(past_state.device)) for past_state in layer_past),
            )
        return reordered_past