# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
import math
from dataclasses import dataclass
from typing import Optional

import torch
import torch.nn as nn
from torch.nn import functional as F
from torch import Tensor
import json
import os

K_LOW_RANK = 128
V_BITS = 2


import itertools

def find_multiple(n: int, k: int) -> int:
    if n % k == 0:
        return n
    return n + k - (n % k)

@dataclass
class SALS_CONFIG:
    recent_length: int = 32
    sparsity: int = 8
    k_high_rank: int = 512
    k_low_rank: int = K_LOW_RANK
    v_bits: int = V_BITS
    group_size: int = 32
defulat_sals_config = SALS_CONFIG()
print(defulat_sals_config)
@dataclass
class ModelArgs:
    block_size: int = 16384
    vocab_size: int = 32000
    n_layer: int = 32
    n_head: int = 32
    dim: int = 4096
    intermediate_size: int = None
    n_local_heads: int = -1
    head_dim: int = 64
    rope_base: float = 40000 # TODO: add config for vicuna-16k
    norm_eps: float = 1e-5
    heavy_const: int = 256
    heavy_channel_num: int = 32
    share_query_const: int = 2
    share_layer_const: int = 16
    recent_length: int = 32
    sparsity: int = 8
    k_high_rank: int = 512
    k_low_rank: int = K_LOW_RANK
    v_bits: int = V_BITS
    group_size: int = 32
    max_batch_size: int = 8



    def __post_init__(self):
        if self.n_local_heads == -1:
            self.n_local_heads = self.n_head
        if self.intermediate_size is None:
            hidden_dim = 4 * self.dim
            n_hidden = int(2 * hidden_dim / 3)
            self.intermediate_size = find_multiple(n_hidden, 256)
        self.head_dim = self.dim // self.n_head

    @classmethod
    def from_name(cls, name: str):
        if name in transformer_configs:
            return cls(**transformer_configs[name])
        # fuzzy search
        config = [config for config in transformer_configs if config in str(name).upper() or config in str(name)]
        assert len(config) == 1, name
        return cls(**transformer_configs[config[0]])


transformer_configs = {
    "CodeLlama-7b-Python-hf": dict(block_size=16384, vocab_size=32000, n_layer=32, dim = 4096, rope_base=1000000),
    "7B": dict(n_layer=32, n_head=32, dim=4096),
    "13B": dict(n_layer=40, n_head=40, dim=5120),
    "30B": dict(n_layer=60, n_head=52, dim=6656),
    "34B": dict(n_layer=48, n_head=64, dim=8192, vocab_size=32000, n_local_heads=8, intermediate_size=22016, rope_base=1000000), # CodeLlama-34B-Python-hf
    "70B": dict(n_layer=80, n_head=64, dim=8192, n_local_heads=8, intermediate_size=28672),
    "mistral-7b": dict(n_layer=32, n_head=32, dim=4096, n_local_heads=8, intermediate_size=14336),
    "Mistral-7B-Instruct-v0.2": dict(n_layer=32, n_head=32, dim=4096, n_local_heads=8, intermediate_size=14336, block_size=131072)
}




class LabelGenerator(torch.nn.Module):
    def __init__(self, token_shared_num, layer_shared_num, bs=4, global_const=8, local_const=32, num_head=32, num_layer=32):
        super().__init__()
        self._token_shared_num = token_shared_num # 4x
        self._layer_shared_num = num_layer - layer_shared_num # 4x
        self._label_bucket = {}
        file_path = f'../config/llama-2-7b-chat-hf/sharing_layer_{self._layer_shared_num}.json'

        if os.path.exists(file_path):
            with open(file_path, 'r') as file:
                data = json.load(file) 

            self._sharing_config = data.get('layer.share_config', list(range(0, num_layer)))
        else:
            self._sharing_config = list(itertools.islice(itertools.cycle(range(self._layer_shared_num)), num_layer))
        self._global_const = global_const
        self._local_const = local_const
        self._global_token_index = torch.tensor([i for i in range(0, self._global_const)] * bs * num_head, dtype=torch.int64, device='cuda').view(bs, num_head, -1)
        self._local_token_index = torch.tensor([i for i in range(0, self._local_const)] * bs * num_head, dtype=torch.int64, device='cuda').view(bs, num_head, -1)


    def get_stream_mask(self, kv_seq_len):
        local_token_index = self._local_token_index + kv_seq_len - self._local_const
        numbers_to_cat = torch.cat((self._global_token_index, local_token_index), dim=-1)
        return numbers_to_cat
            
    def forward(self, Q, K, q_label, k_label, channel, heavy_const, heavy_channel_num, token_id, layer_id, mask1):

        # Every token-group/layer_group start, we need to recompute the heavy label
        if token_id % self._token_shared_num == 0 and self._sharing_config[layer_id] == layer_id:

            get_label_tensor(Q, channel, q_label, heavy_channel_num)

            kv_seq_len, batch = K.shape[0] // Q.shape[0], Q.shape[0]
            q_head_num, k_head_num = Q.shape[1], K.shape[1]
            tmp_scores = torch.matmul(q_label.view(batch, 1, q_head_num, heavy_channel_num).transpose(1,2), k_label.view(batch, kv_seq_len, k_head_num, heavy_channel_num).transpose(1,2).transpose(2, 3)).view(batch, k_head_num, kv_seq_len)
            tmp_scores += mask1.squeeze(1)
            if token_id <= heavy_const:
                _, label_index = torch.topk(tmp_scores, heavy_const, dim=-1)
            else:
                _, label_index = torch.topk(tmp_scores[..., self._global_const:token_id - self._local_const], heavy_const-self._local_const-self._global_const, dim=-1)
                label_index += self._global_const
                stream_mask = self.get_stream_mask(token_id)
                label_index = torch.cat((label_index, stream_mask), dim=-1)

            assert len(self._label_bucket) <= self._layer_shared_num
            self._label_bucket[layer_id] = label_index
        else:
            label_index = self._label_bucket[self._sharing_config[layer_id]]

        return label_index


class KVCache(nn.Module):
    def __init__(self, max_batch_size, max_seq_length, n_heads, head_dim, heavy_channel_num, dtype=torch.bfloat16):
        super().__init__()
        self.max_batch_size = max_batch_size
        self.max_seq_length = max_seq_length
        cache_shape = (max_batch_size, max_seq_length, n_heads, head_dim)
        self.register_buffer('k_cache', torch.zeros(cache_shape, dtype=dtype))
        self.register_buffer('v_cache', torch.zeros(cache_shape, dtype=dtype))
        # self.register_buffer('k_label', torch.zeros((max_batch_size, max_seq_length, n_heads, heavy_channel_num), dtype=dtype))

        # store qk tmp label while decoding
        # self.register_buffer('tmp_label', torch.zeros((max_batch_size * 2, n_heads, heavy_channel_num), dtype=dtype))
        # store tmp label scores while decoding
        # self.register_buffer('label_scores', torch.zeros((max_batch_size, n_heads, max_seq_length), dtype=dtype))
        # store tmp attn output while decoding
        self.register_buffer('attn_out', torch.zeros((max_batch_size, n_heads, head_dim), dtype=dtype))

    def update(self, input_pos, k_val, v_val):
        # input_pos: [S], k_val: [B, S, H, D]
        assert input_pos.shape[0] == k_val.shape[1]

        k_out = self.k_cache
        v_out = self.v_cache
        k_out[:, input_pos] = k_val
        v_out[:, input_pos] = v_val
        return k_out, v_out

from compression.kernel.new_pack import triton_quantize_and_pack_along_last_dim
from compression.kernel.fused_recover_fa import fused_recover_fa_attention

class SALS_KVCache(nn.Module):
    def __init__(self, max_batch_size, max_seq_length, n_heads, n_local_heads, head_dim, sals_config:SALS_CONFIG, dtype=torch.bfloat16):
        super().__init__()

        self.max_batch_size = max_batch_size
        self.max_seq_length = max_seq_length
        key_cache_shape_high = (max_batch_size, sals_config.recent_length, sals_config.k_high_rank)
        key_cache_shape_low = (max_batch_size, max_seq_length, sals_config.k_low_rank)
        value_cache_high = (max_batch_size, sals_config.recent_length, n_local_heads, head_dim)
        value_cache_low = (max_batch_size, max_seq_length, n_local_heads, head_dim // (32 // sals_config.v_bits))
        value_scale = (max_batch_size, max_seq_length, n_local_heads, head_dim // sals_config.group_size)

        self.high_ptr = 0
        self.recent_length = sals_config.recent_length
        self.sals_config = sals_config
        self.register_buffer('k_cache_high', torch.zeros(key_cache_shape_high, dtype=dtype))
        self.register_buffer('k_cache_low', torch.zeros(key_cache_shape_low, dtype=dtype))
        self.register_buffer('v_cache_high', torch.zeros(value_cache_high, dtype=dtype))
        self.register_buffer('v_cache_low', torch.zeros(value_cache_low, dtype=torch.int32))
        self.register_buffer('v_scale', torch.zeros(value_scale, dtype=dtype))
        self.register_buffer('v_mn', torch.zeros(value_scale, dtype=dtype))
        # self.register_buffer('k_label', torch.zeros((max_batch_size, max_seq_length, n_heads, heavy_channel_num), dtype=dtype))

        # store qk tmp label while decoding
        # self.register_buffer('tmp_label', torch.zeros((max_batch_size * 2, n_heads, heavy_channel_num), dtype=dtype))
        # store tmp label scores while decoding
        # self.register_buffer('label_scores', torch.zeros((max_batch_size, n_heads, max_seq_length), dtype=dtype))
        # store tmp attn output while decoding
        self.register_buffer('attn_out', torch.zeros((max_batch_size, n_heads, head_dim), dtype=dtype))

    def update(self, input_pos, k_val, v_val):
        # input_pos: [S], k_val: [B, S, H, D]
        # assert input_pos.shape[0] == k_val.shape[1]

        latest_ptr = (self.high_ptr - 1 + self.recent_length) % self.recent_length
        if input_pos >= self.recent_length:
            # import pdb;pdb.set_trace()
            k_latest = self.k_cache_high[:, latest_ptr:latest_ptr + 1, -self.sals_config.k_low_rank:] 
            self.k_cache_low[:, input_pos - self.recent_length] = k_latest
            v_latest_quant, scale, mn = triton_quantize_and_pack_along_last_dim(self.v_cache_high[:, latest_ptr:latest_ptr + 1], self.sals_config.group_size, self.sals_config.v_bits)
            self.v_cache_low[:, input_pos - self.recent_length] = v_latest_quant
            self.v_scale[:, input_pos - self.recent_length] = scale
            self.v_mn[:, input_pos - self.recent_length] = mn
        self.k_cache_high[:, self.high_ptr:self.high_ptr + 1] = k_val
        self.v_cache_high[:, self.high_ptr:self.high_ptr + 1] = v_val
        
        self.high_ptr = (self.high_ptr + 1) % self.recent_length

        return self.k_cache_high, self.k_cache_low, self.v_cache_high, self.v_cache_low, self.v_scale, self.v_mn
    
    def update_prefill(self, input_pos, k_val, v_val, weight):
 
        bsz, k_len, n_local_head, head_dim = k_val.shape
        k_down = torch.matmul(k_val.view(bsz, k_len, -1), weight)

        # import pdb;pdb.set_trace()
        if k_len <= self.recent_length:
            self.k_cache_high[:, :k_len] = k_down
            self.v_cache_high[:, :k_len] = v_val
        else:
            self.k_cache_high[:, :self.recent_length] = k_down[:, -self.recent_length:, :]
            self.k_cache_low[:, :k_len - self.recent_length] = k_down[:, :-self.recent_length, -self.sals_config.k_low_rank:]
            v_to_quant, v_to_full = torch.split(v_val, [k_len - self.recent_length, self.recent_length], dim=1)
            v_quant, value_scale, value_mn = triton_quantize_and_pack_along_last_dim(v_to_quant, self.sals_config.group_size, self.sals_config.v_bits)
            self.v_cache_high[:, :self.recent_length] = v_to_full
            self.v_cache_low[:, :k_len - self.recent_length] = v_quant
            self.v_scale[:, :k_len - self.recent_length] = value_scale
            self.v_mn[:, :k_len - self.recent_length] = value_mn


        return k_val, v_val

class Transformer(nn.Module):
    def __init__(self, config: ModelArgs) -> None:
        super().__init__()
        self.config = config

        self.tok_embeddings = nn.Embedding(config.vocab_size, config.dim)
        self.layers = nn.ModuleList(TransformerBlock(config, _) for _ in range(config.n_layer))
        self.norm = RMSNorm(config.dim, eps=config.norm_eps)
        self.output = nn.Linear(config.dim, config.vocab_size, bias=False)

        self.freqs_cis: Optional[Tensor] = None
        self.mask_cache: Optional[Tensor] = None
        self.max_batch_size = -1
        self.max_seq_length = -1
        self.label_generator = None

    def setup_caches(self, max_batch_size, max_seq_length):

        self.label_generator = LabelGenerator(self.config.share_query_const, self.config.share_layer_const, max_batch_size)

        if self.max_seq_length >= max_seq_length and self.max_batch_size >= max_batch_size:
            return
        head_dim = self.config.dim // self.config.n_head
        max_seq_length = find_multiple(max_seq_length, 8)
        self.max_seq_length = max_seq_length
        self.max_batch_size = max_batch_size
        for b in self.layers:
            b.attention.kv_cache = KVCache(max_batch_size, max_seq_length, self.config.n_local_heads, head_dim, self.config.heavy_channel_num)

        self.freqs_cis = precompute_freqs_cis(self.config.block_size, self.config.dim // self.config.n_head, self.config.rope_base)
        self.prefill_mask = torch.tril(torch.ones(self.max_seq_length, self.max_seq_length, dtype=torch.bool))

        # # TODO: change 16 to 32
        self.label_mask = torch.zeros(self.max_seq_length, self.max_seq_length, dtype=torch.float16)
        self.label_mask = self.label_mask.masked_fill(self.prefill_mask == False, float('-inf'))


        # # TODO: change 16 to 32
        # self.attn_mask = torch.zeros(self.max_seq_length, self.config.heavy_const, dtype=torch.float16)
        # self.attn_mask = self.attn_mask.masked_fill(torch.tril(torch.ones(self.max_seq_length, self.config.heavy_const, dtype=torch.bool)) == False, float('-inf'))
    
    def setup_SALS_caches(self, max_batch_size, max_seq_length, sals_config:SALS_CONFIG):

        self.label_generator = LabelGenerator(self.config.share_query_const, self.config.share_layer_const, max_batch_size)

        if self.max_seq_length >= max_seq_length and self.max_batch_size >= max_batch_size:
            return
        head_dim = self.config.dim // self.config.n_head
        max_seq_length = find_multiple(max_seq_length, 8)
        self.max_seq_length = max_seq_length
        self.max_batch_size = max_batch_size
        for b in self.layers:
            b.attention.kv_cache = SALS_KVCache(max_batch_size, max_seq_length, self.config.n_head, self.config.n_local_heads, head_dim, sals_config)

        self.freqs_cis = precompute_freqs_cis(self.config.block_size, self.config.dim // self.config.n_head, self.config.rope_base)
        self.prefill_mask = torch.tril(torch.ones(self.max_seq_length, self.max_seq_length, dtype=torch.bool))


        # # TODO: change 16 to 32
        self.label_mask = torch.zeros(self.max_seq_length, self.max_seq_length, dtype=torch.float16)
        self.label_mask = self.label_mask.masked_fill(self.prefill_mask == False, float('-inf'))



    def forward(self, idx: Tensor, input_pos: Optional[Tensor] = None) -> Tensor:
        assert self.freqs_cis is not None, "Caches must be initialized first"

        # is_prefill = input_pos.shape[-1] > 1

        # if is_prefill:
        #     mask1 = self.prefill_mask[None, None, input_pos] # [B, H, S, S]
        #     mask2 = None
        # else:
        #     # TODO: this is a shortcut, the mask broadcast should be rewritten
        #     mask1 = self.label_mask[None, input_pos] # [1, 1, S]
        #     mask2 = self.attn_mask[input_pos] # [1, HEAVY_CONST] 

        mask1 = self.label_mask[None, None, input_pos] # [B, H, S, S]
        # mask2 = torch.zeros(1, self.config.heavy_const, dtype=torch.float16).cuda() # [1, HEAVY_CONST]

        freqs_cis = self.freqs_cis[input_pos]
        x = self.tok_embeddings(idx)

        for i, layer in enumerate(self.layers):
            x = layer(x, input_pos, freqs_cis, mask1)
        x = self.norm(x)
        logits = self.output(x)
        return logits
    
    def sparse_forward(self, idx: Tensor, input_pos: Optional[Tensor] = None) -> Tensor:
        assert self.freqs_cis is not None, "Caches must be initialized first"

        mask1 = self.label_mask[None, None, input_pos]
        mask2 = torch.zeros(1, self.config.heavy_const, dtype=torch.float16).cuda()

        freqs_cis = self.freqs_cis[input_pos]
        x = self.tok_embeddings(idx)

        for i, layer in enumerate(self.layers):
            x = layer.sparse_forward(x, input_pos, freqs_cis, mask1, mask2, self.label_generator)
        x = self.norm(x)
        logits = self.output(x)
        return logits
    
    def sals_forward(self, idx: Tensor, input_pos: Optional[Tensor] = None) -> Tensor:
        assert self.freqs_cis is not None, "Caches must be initialized first"

        mask1 = self.label_mask[None, None, input_pos]

        freqs_cis = self.freqs_cis[input_pos]
        x = self.tok_embeddings(idx)

        for i, layer in enumerate(self.layers):
            x = layer.sals_forward(x, input_pos, freqs_cis, mask1)
        x = self.norm(x)
        logits = self.output(x)
        return logits

    @classmethod
    def from_name(cls, name: str):
        return cls(ModelArgs.from_name(name))


class TransformerBlock(nn.Module):
    def __init__(self, config: ModelArgs, layer_id) -> None:
        super().__init__()
        self.attention = Attention(config)
        self.feed_forward = FeedForward(config)
        self.ffn_norm = RMSNorm(config.dim, config.norm_eps)
        self.attention_norm = RMSNorm(config.dim, config.norm_eps)
        self.layer_id = layer_id

    def forward(self, x: Tensor, input_pos: Tensor, freqs_cis: Tensor, mask1: Tensor) -> Tensor:
        h = x + self.attention(self.attention_norm(x), freqs_cis, mask1, input_pos)
        out = h + self.feed_forward(self.ffn_norm(h))
        return out

    def sparse_forward(self, x: Tensor, input_pos: Tensor, freqs_cis: Tensor, mask1: Tensor, mask2: Tensor, label_generator: LabelGenerator) -> Tensor:
        h = x + self.attention.sparse_forward(self.attention_norm(x), freqs_cis, mask1, mask2, input_pos, self.layer_id, label_generator)
        out = h + self.feed_forward(self.ffn_norm(h))
        return out
    def sals_forward(self, x: Tensor, input_pos: Tensor, freqs_cis: Tensor, mask1: Tensor) -> Tensor:
        h = x + self.attention.sals_forward(self.attention_norm(x), freqs_cis, mask1, input_pos)
        out = h + self.feed_forward(self.ffn_norm(h))
        return out
from transformers.models.mistral.modeling_mistral import MistralRotaryEmbedding

class Attention(nn.Module):
    def __init__(self, config: ModelArgs):
        super().__init__()
        assert config.dim % config.n_head == 0

        total_head_dim = (config.n_head + 2 * config.n_local_heads) * config.head_dim
        # key, query, value projections for all heads, but in a batch
        self.wqkv = nn.Linear(config.dim, total_head_dim, bias=False)
        self.wo = nn.Linear(config.dim, config.dim, bias=False)
        self.kv_cache = None

        self.config = config
        self.n_head = config.n_head
        self.head_dim = config.head_dim
        self.n_local_heads = config.n_local_heads
        self.dim = config.dim
        self._register_load_state_dict_pre_hook(self.load_hook)

        # channel config
        self.sorted_channel = None

        # heavy const
        self.heavy_const = config.heavy_const
        self.heavy_channel_num = config.heavy_channel_num
        self.rotary_emb = MistralRotaryEmbedding(
                self.head_dim,
                max_position_embeddings=config.block_size,
            )
        self.sals_weight = torch.randn(
            self.n_local_heads * self.head_dim,
            self.config.k_high_rank,
            dtype=torch.bfloat16,
            device='cuda:0'
        )
        self.high_index = torch.arange(0, config.recent_length, device='cuda:0').unsqueeze(0).expand(config.max_batch_size, -1)

    def load_hook(self, state_dict, prefix, *args):
        if prefix + "wq.weight" in state_dict:
            wq = state_dict.pop(prefix + "wq.weight")
            wk = state_dict.pop(prefix + "wk.weight")
            wv = state_dict.pop(prefix + "wv.weight")
            state_dict[prefix + "wqkv.weight"] = torch.cat([wq, wk, wv])

    def forward(self, x: Tensor, freqs_cis: Tensor, mask1: Tensor, input_pos: Optional[Tensor] = None) -> Tensor:
        
        bsz, seqlen, _ = x.shape

        kv_size = self.n_local_heads * self.head_dim
        q, k, v = self.wqkv(x).split([self.dim, kv_size, kv_size], dim=-1)

        q = q.view(bsz, seqlen, self.n_head, self.head_dim)
        k = k.view(bsz, seqlen, self.n_local_heads, self.head_dim)
        v = v.view(bsz, seqlen, self.n_local_heads, self.head_dim)

        q = apply_rotary_emb(q, freqs_cis)
        # k = apply_rotary_emb(k, freqs_cis)

        # tmp_labels = torch.empty((bsz * seqlen, self.n_head, self.heavy_channel_num), dtype=self.kv_cache.k_label.dtype, device='cuda')
        # get_label_tensor(k.view(bsz * seqlen, self.n_local_heads, self.head_dim), self.sorted_channel, tmp_labels, self.heavy_channel_num)


        if self.kv_cache is not None:
            k, v = self.kv_cache.update_prefill(input_pos, k, v, self.sals_weight)
            # k, v = self.kv_cache.update(input_pos, k, v)
            # self.kv_cache.k_label[:, input_pos] = tmp_labels.view(bsz, seqlen, self.n_head, self.heavy_channel_num)
        # import pdb;pdb.set_trace()
        q, k, v = map(lambda x: x.transpose(1, 2), (q, k, v))
        k = k.repeat_interleave(self.n_head // self.n_local_heads, dim=1)
        v = v.repeat_interleave(self.n_head // self.n_local_heads, dim=1)
        attn_weights = torch.matmul(q, k.transpose(2, 3)) / math.sqrt(self.head_dim)
        attn_weights += mask1[:, :, :, :q.shape[2]]
        attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32).to(q.dtype)
        y = torch.matmul(attn_weights, v)
        y = y.transpose(1, 2).contiguous().view(bsz, seqlen, self.dim)
        y = self.wo(y)

        return y

    def sals_forward(self, x: Tensor, freqs_cis: Tensor, mask1: Tensor,  input_pos: Optional[Tensor] = None) -> Tensor:
        
        bsz, seqlen, _ = x.shape

        kv_size = self.n_local_heads * self.head_dim
        q, k, v = self.wqkv(x).split([self.dim, kv_size, kv_size], dim=-1)

        q = q.view(bsz, seqlen, self.n_head, self.head_dim)
        # k = k.view(bsz, seqlen, self.n_local_heads, self.head_dim)
        k = torch.matmul(k, self.sals_weight)
        v = v.view(bsz, seqlen, self.n_local_heads, self.head_dim)

        # q = apply_rotary_emb(q, freqs_cis)
        # k = apply_rotary_emb(k, freqs_cis)
        # import pdb;pdb.set_trace()
        if self.kv_cache is not None:
            k_cache_high, k_cache_low, v_cache_high, v_cache_low, v_scale, v_mn = self.kv_cache.update(input_pos, k, v)
        cos, sin = self.rotary_emb(v, seq_len=input_pos)

     
        
        # q, k, v = map(lambda x: x.transpose(1, 2), (q, k, v))
        high_length = self.config.recent_length
        low_length = k_cache_low.shape[1] - high_length
        y = fused_recover_fa_attention( q,
                                        k_cache_high.unsqueeze(1),
                                        k_cache_low.unsqueeze(1),
                                        v_cache_high.transpose(1, 2),
                                        v_cache_low.transpose(1, 2),
                                        v_scale,
                                        v_mn,
                                        self.sals_weight.transpose(0, 1),
                                        None,
                                        cos,
                                        sin,
                                        self.high_index,
                                        self.config.sparsity,
                                        self.n_local_heads,
                                        self.config.v_bits,
                                        self.config.group_size,
                                        mask1,
                                        high_length,
                                        low_length
                                        ).view(bsz, seqlen, -1)
        y = self.wo(y)

        return y

    def sparse_forward(self, x: Tensor, freqs_cis: Tensor, mask1: Tensor, mask2: Tensor, input_pos: Optional[Tensor], layer_id : int, label_generator: LabelGenerator) -> Tensor:
        bsz, seqlen, _ = x.shape

        kv_size = self.n_local_heads * self.head_dim
        q, k, v = self.wqkv(x).split([self.dim, kv_size, kv_size], dim=-1)

        q = q.view(bsz, seqlen, self.n_head, self.head_dim)
        k = k.view(bsz, seqlen, self.n_local_heads, self.head_dim)
        v = v.view(bsz, seqlen, self.n_local_heads, self.head_dim)

        q = apply_rotary_emb(q, freqs_cis)
        k = apply_rotary_emb(k, freqs_cis)

        tmp_labels = torch.empty((bsz * seqlen, self.n_head, self.heavy_channel_num), dtype=self.kv_cache.k_label.dtype, device='cuda')
        get_label_tensor(k.view(bsz * seqlen, self.n_local_heads, self.head_dim), self.sorted_channel, tmp_labels, self.heavy_channel_num)


        if self.kv_cache is not None:
            k, v = self.kv_cache.update(input_pos, k, v)
            self.kv_cache.k_label[:, input_pos] = tmp_labels.view(bsz, seqlen, self.n_head, self.heavy_channel_num)

        label_index = label_generator(q.view(bsz, self.n_head, self.head_dim), k.view(-1, self.n_head, self.head_dim), tmp_labels, self.kv_cache.k_label, self.sorted_channel, self.heavy_const, self.heavy_channel_num, input_pos[0], layer_id, mask1)
        fwd_sparse_no_mask(q.view(-1, self.n_head, self.head_dim), k.view(-1, self.n_local_heads, self.head_dim), v.view(-1, self.n_local_heads, self.head_dim), self.kv_cache.attn_out, label_index.view(bsz, self.n_head, -1))
        y = self.wo(self.kv_cache.attn_out.view(bsz, seqlen, self.dim))

        return y


class FeedForward(nn.Module):
    def __init__(self, config: ModelArgs) -> None:
        super().__init__()
        self.w1 = nn.Linear(config.dim, config.intermediate_size, bias=False)
        self.w3 = nn.Linear(config.dim, config.intermediate_size, bias=False)
        self.w2 = nn.Linear(config.intermediate_size, config.dim, bias=False)

    def forward(self, x: Tensor) -> Tensor:
        return self.w2(F.silu(self.w1(x)) * self.w3(x))


class RMSNorm(nn.Module):
    def __init__(self, dim: int, eps: float = 1e-5):
        super().__init__()
        self.eps = eps
        self.weight = nn.Parameter(torch.ones(dim))

    def _norm(self, x):
        return x * torch.rsqrt(torch.mean(x * x, dim=-1, keepdim=True) + self.eps)

    def forward(self, x: Tensor) -> Tensor:
        output = self._norm(x.float()).type_as(x)
        return output * self.weight


def precompute_freqs_cis(
    seq_len: int, n_elem: int, base: int = 10000
) -> Tensor:
    freqs = 1.0 / (base ** (torch.arange(0, n_elem, 2)[: (n_elem // 2)].float() / n_elem))
    t = torch.arange(seq_len, device=freqs.device)
    freqs = torch.outer(t, freqs)
    freqs_cis = torch.polar(torch.ones_like(freqs), freqs)
    cache = torch.stack([freqs_cis.real, freqs_cis.imag], dim=-1)
    return cache.to(dtype=torch.bfloat16)


def apply_rotary_emb(x: Tensor, freqs_cis: Tensor) -> Tensor:
    xshaped = x.float().reshape(*x.shape[:-1], -1, 2)
    freqs_cis = freqs_cis.view(1, xshaped.size(1), 1, xshaped.size(3), 2)
    x_out2 = torch.stack(
        [
            xshaped[..., 0] * freqs_cis[..., 0] - xshaped[..., 1] * freqs_cis[..., 1],
            xshaped[..., 1] * freqs_cis[..., 0] + xshaped[..., 0] * freqs_cis[..., 1],
        ],
        -1,
    )

    x_out2 = x_out2.flatten(3)
    return x_out2.type_as(x)




def init_model_channel_config(model, channel_config, selected_channel="k"):

    selected_channel = "." + selected_channel + "_proj"

    for name, module in model.named_modules():

        if isinstance(module, Attention):
            
            layer_idx = int(name.split(".")[1])
            key = "model.layers." + str(layer_idx) + ".self_attn" + selected_channel
            
            module.sorted_channel = permute_channel_config(torch.tensor(channel_config[key]))[:,:module.heavy_channel_num].contiguous().cuda()

    return model

def permute_channel_config(sorted_channel):
    head_num = sorted_channel.shape[0]
    head_dim = sorted_channel.shape[1]
    return (sorted_channel * 2) % head_dim + (sorted_channel * 2) // head_dim
