function generate_data(
    system::AbstractDynamicalSystem{T};
    seconds::T,
    dt::T,
    transient_seconds::T = zero(T),
    solver = Vern9(),
    reltol = 1e-24,
    abstol= 1e-24,
    u0::Union{AbstractVector{T}, Nothing} = nothing,
    maxiters = typemax(Int),
    transform = nothing,
    in_place = true,
    NF,
) where {T<:AbstractFloat}
    # @info "Generating training data..."

    if u0 === nothing
        u0 = get_default_initial_conditions(system)
    end

    tspan = (zero(T), one(T))  # Arbitrary
    prob = ODEProblem{in_place}(system, u0, tspan; saveat = dt)

    if transient_seconds > 0
        transient_prob = remake(prob, tspan = (zero(NF), transient_seconds))
        transient_data = solve(transient_prob, solver; reltol, abstol, maxiters)
        u0 = transient_data[end]
    else
        u0 = prob.u0
    end

    prob = remake(prob, u0 = u0, tspan = (zero(NF), seconds))
    sol = solve(prob, solver; reltol, abstol, maxiters)
    trajectory = Array(sol)

    if !isnothing(transform)
        trajectory = mapslices(transform, trajectory, dims = 1)
    end

    # @info "Finished generating training data."
    return NeuralDiffEqTools.TimeSeries{NF}(sol.t, trajectory)
end

function get_data(
    system,
    experiment_version,
    T,
    dt,
    transient_seconds,
    solver,
    reltol,
    abstol,
    n_train,
    n_valid,
    n_test,
    steps,
    stabilization_param,
    θ,
    restructure,
)
    @info "Generating training data..."
    train_data = get_trajectories(
        system,
        experiment_version,
        T,
        dt,
        transient_seconds,
        solver,
        reltol,
        abstol,
        n_train,
        steps,
        stabilization_param,
        θ,
        restructure,
    )
    valid_data = get_trajectories(
        system,
        experiment_version,
        T,
        dt,
        transient_seconds,
        solver,
        reltol,
        abstol,
        n_valid,
        steps,
        stabilization_param,
        θ,
        restructure,
    )
    test_data = get_trajectories(
        system,
        experiment_version,
        T,
        dt,
        transient_seconds,
        solver,
        reltol,
        abstol,
        n_test,
        steps,
        stabilization_param,
        θ,
        restructure,
    )
    @info "Finished generating training data."
    return (; train_data, valid_data, test_data)
end
