﻿import os
import numpy
import shutil


class SavingEngine():
    def __init__(self, config, database, outputDir):
        self._outputDir = outputDir
        self._config    = config.attributes
        self._database  = database


    def saveTrainingSamples(self):
        samples_x, samples_y = self._database.getAllTrainingSamples()
        samples_x = self._database.unscaleSamplesX(samples_x)
        samples_y = self._database.unscaleSamplesY(samples_y)
        samples = numpy.hstack((samples_x, samples_y))
        fName = os.path.join(self._outputDir, self._config["input"]["samplesFile"])
        with open(fName, mode="w+") as samplesFile:
            samplesFile.write(f"xn={self._config['input']['xDimensions']},yn={self._config['input']['yDimensions']}\n")
            numpy.savetxt(samplesFile, samples, delimiter=",", fmt="%f")    


    def copyOtherSamples(self):
        for file in os.listdir(self._config['input']['dir']):
            if file.endswith(".csv") or file.endswith(".txt"):
                if "training" in file:
                    continue
                shutil.copy(os.path.join(self._config["input"]["dir"], file), self._outputDir)            


    def saveSVMParameters(self, svmParameters):
        fName = self._outputDir + 'svmParameters.txt'
        with open(fName, mode='w+') as svmConfigFile:
            svmConfigFile.write('SVM parameters: ' + str(svmParameters))


    def saveConfig(self, netIds, learningRates, layers, neuronsPerLayer):
        keys = self.defineConfigKeys()
        fName = self._outputDir + 'configPartitioning.txt'
        with open(fName, mode='w+') as configFile:
            for multiLevelKey in keys:
                val = self._config
                for nextLevelKey in multiLevelKey:
                    val = val[nextLevelKey]
                configFile.write(str(multiLevelKey) + ': ' + str(val) + '\n')
            configFile.write('\n\n')
            for netId, learningRate in zip(netIds, learningRates):
                configFile.write(f'Learning rate - Net {netId} {learningRate}\n')
            for netId, layer in zip(netIds, layers):
                configFile.write(f'Layers - Net {netId}: {layer}\n')
            for netId, neuronPerLayer in zip(netIds, neuronsPerLayer):
                configFile.write(f'Neurons per layer - Net {netId}: {neuronPerLayer}\n')


    def defineConfigKeys(self):
        return [['input', 'dir'],
                ['input', 'samplesFile'],
                ['input', 'xDimensions'],
                ['input', 'yDimensions'],
                ['network', 'layersBounds'],
                ['network', 'neuronsPerLayerBounds'],
                ['network', 'learningRateBounds'],
                ['test', 'samples'],
                ['test', 'epochs'],
                ['test', 'batchSize'],
                ['test', 'initialNumOfNets'],
                ['test', 'featureRangeX'],
                ['test', 'featureRangeY'],
                ['test', 'startDroppingAdding'],
                ['test', 'droppingActive'],
                ['test', 'droppingInterv'],
                ['test', 'droppingReplacability'],
                ['test', 'addingActive'],
                ['test', 'addingInterv'],
                ['test', 'addingCatchUpTrainingPatience'],
                ['test', 'addingImprovement'],
                ['output', 'dir']]
