# ViT reference: https://github.com/lucidrains/vit-pytorch/blob/main/vit_pytorch/vit_for_small_dataset.py

import copy
import os
import torch
import torch.nn.functional as F
from torch import nn

from einops import rearrange, repeat
from einops.layers.torch import Rearrange
import torchvision

import logging

logging.basicConfig(format='%(asctime)s - %(levelname)s - %(message)s', level=logging.INFO)


def initialize_model(model_name: str, num_classes: int, device: str) -> nn.Module:
    """ Initializes the model.
    Args:
        model_name: The model.
        num_classes: Number of classes.
        device: The device (cpu or gpu).
    Returns:
        The initialized model.
    """
    if model_name == 'resnet18':
        model = ResNet18(num_classes=num_classes)
        model = model.to(device)
    elif model_name == 'vit':
        model = ViT(num_classes=num_classes)
        model = model.to(device)
    elif model_name == 'resnet50':
        model = ResNet50(num_classes=num_classes)
        model = model.to(device)
    else:
        raise ValueError('Model %s is not implemented yet...', model_name)

    return model


def load_model(model_name: str, num_classes: int, load_dir: str, device: str) -> (nn.Module, nn.Module):
    """ Loads the pretrained model.
    Args:
        model_name: The model.
        num_classes: Number of classes.
        load_dir: The directory to load the model from.
        device: The device(e.g. cpu).
    Returns:
        The pretrained model and the corresponding initialized model.
    """
    if not os.path.exists(load_dir):
        raise FileNotFoundError(f'Directory {load_dir} does not exist!')

    model = initialize_model(model_name, num_classes, device)

    init_model = copy.deepcopy(model)
    model.load_state_dict(torch.load(load_dir, weights_only=True,  map_location=torch.device('cpu')))
    logging.info('Loaded the pretrained model from %s', load_dir)

    return model, init_model


def save_model(model: nn.Module, save_dir: str) -> None:
    """Saves the model.
    Args:
        model: The model to be saved.
        save_dir: The directory to save the model at.
    Returns:
        None.
    """
    torch.save(model.state_dict(), save_dir)
    logging.info('Saved the model at %s', save_dir)


# ----------------------------------------------------ResNet18/50-------------------------------------------------------
class ResNet18(nn.Module):
    def __init__(self, num_classes):
        super(ResNet18, self).__init__()
        self.resnet18 = torchvision.models.resnet18(num_classes=num_classes)
        if num_classes == 10:
            self.resnet18.conv1 = nn.Conv2d(3, 64, 3, 1, 1, bias=False)
            self.resnet18.maxpool = nn.Identity()

    def forward(self, x):
        return self.resnet18(x)


class ResNet50(nn.Module):
    def __init__(self, num_classes):
        super(ResNet50, self).__init__()
        self.resnet50 = torchvision.models.resnet50(num_classes=num_classes)
        if num_classes == 200:
            self.resnet50.conv1 = nn.Conv2d(3, 64, 3, 1, 1, bias=False)
            self.resnet50.maxpool = nn.Identity()

    def forward(self, x):
        return self.resnet50(x)


# --------------------------------------------------------- ViT --------------------------------------------------------
# helpers
def pair(t):
    return t if isinstance(t, tuple) else (t, t)


# classes
class FeedForward(nn.Module):
    def __init__(self, dim, hidden_dim, dropout=0.):
        super().__init__()
        self.net = nn.Sequential(
            nn.LayerNorm(dim),
            nn.Linear(dim, hidden_dim),
            nn.GELU(),
            nn.Dropout(dropout),
            nn.Linear(hidden_dim, dim),
            nn.Dropout(dropout)
        )

    def forward(self, x):
        return self.net(x)


class LSA(nn.Module):
    def __init__(self, dim, heads=8, dim_head=64, dropout=0.):
        super().__init__()
        inner_dim = dim_head * heads
        self.heads = heads
        self.temperature = nn.Parameter(torch.log(torch.tensor(dim_head ** -0.5)))

        self.norm = nn.LayerNorm(dim)
        self.attend = nn.Softmax(dim=-1)
        self.dropout = nn.Dropout(dropout)

        self.to_qkv = nn.Linear(dim, inner_dim * 3, bias=False)

        self.to_out = nn.Sequential(
            nn.Linear(inner_dim, dim),
            nn.Dropout(dropout)
        )

    def forward(self, x):
        x = self.norm(x)
        qkv = self.to_qkv(x).chunk(3, dim=-1)
        q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h=self.heads), qkv)

        dots = torch.matmul(q, k.transpose(-1, -2)) * self.temperature.exp()

        mask = torch.eye(dots.shape[-1], device=dots.device, dtype=torch.bool)
        mask_value = -torch.finfo(dots.dtype).max
        dots = dots.masked_fill(mask, mask_value)

        attn = self.attend(dots)
        attn = self.dropout(attn)

        out = torch.matmul(attn, v)
        out = rearrange(out, 'b h n d -> b n (h d)')
        return self.to_out(out)


class Transformer(nn.Module):
    def __init__(self, dim, depth, heads, dim_head, mlp_dim, dropout=0.):
        super().__init__()
        self.layers = nn.ModuleList([])
        for _ in range(depth):
            self.layers.append(nn.ModuleList([
                LSA(dim, heads=heads, dim_head=dim_head, dropout=dropout),
                FeedForward(dim, mlp_dim, dropout=dropout)
            ]))

    def forward(self, x):
        for attn, ff in self.layers:
            x = attn(x) + x
            x = ff(x) + x
        return x


class SPT(nn.Module):
    def __init__(self, *, dim, patch_size, channels=3):
        super().__init__()
        patch_dim = patch_size * patch_size * 5 * channels

        self.to_patch_tokens = nn.Sequential(
            Rearrange('b c (h p1) (w p2) -> b (h w) (p1 p2 c)', p1=patch_size, p2=patch_size),
            nn.LayerNorm(patch_dim),
            nn.Linear(patch_dim, dim)
        )

    def forward(self, x):
        shifts = ((1, -1, 0, 0), (-1, 1, 0, 0), (0, 0, 1, -1), (0, 0, -1, 1))
        shifted_x = list(map(lambda shift: F.pad(x, shift), shifts))
        x_with_shifts = torch.cat((x, *shifted_x), dim=1)
        return self.to_patch_tokens(x_with_shifts)


class ViT(nn.Module):
    def __init__(self, *, image_size=32, patch_size=8, num_classes=10, dim=512, depth=4, heads=8, mlp_dim=1536,
                 pool='cls', channels=3, dim_head=64, dropout=0., emb_dropout=0.):
        super().__init__()
        image_height, image_width = pair(image_size)
        patch_height, patch_width = pair(patch_size)

        assert image_height % patch_height == 0 and image_width % patch_width == 0, 'Image dimensions must be divisible by the patch size.'

        num_patches = (image_height // patch_height) * (image_width // patch_width)
        patch_dim = channels * patch_height * patch_width
        assert pool in {'cls', 'mean'}, 'pool type must be either cls (cls token) or mean (mean pooling)'

        self.to_patch_embedding = SPT(dim=dim, patch_size=patch_size, channels=channels)

        self.pos_embedding = nn.Parameter(torch.randn(1, num_patches + 1, dim))
        self.cls_token = nn.Parameter(torch.randn(1, 1, dim))
        self.dropout = nn.Dropout(emb_dropout)

        self.transformer = Transformer(dim, depth, heads, dim_head, mlp_dim, dropout)

        self.pool = pool
        self.to_latent = nn.Identity()

        self.mlp_head = nn.Sequential(
            nn.LayerNorm(dim),
            nn.Linear(dim, num_classes)
        )

    def forward(self, img):
        x = self.to_patch_embedding(img)
        b, n, _ = x.shape

        cls_tokens = repeat(self.cls_token, '() n d -> b n d', b=b)
        x = torch.cat((cls_tokens, x), dim=1)
        x += self.pos_embedding[:, :(n + 1)]
        x = self.dropout(x)

        x = self.transformer(x)

        x = x.mean(dim=1) if self.pool == 'mean' else x[:, 0]

        x = self.to_latent(x)
        return self.mlp_head(x)
