
# MPM Simulation with DF-SIM
<p align="center">
  <img src="picture1.png" width="800" alt="Simulation Teaser">
</p>

## Abstract
Fracture-GS: Dynamic Fracture Simulation with Physics-Integrated Gaussian Splatting

This paper presents a unified framework for simulating and visualizing dynamic fracture phenomena in extreme mechanical collisions using multi-view image inputs. While existing methods primarily address elastic deformations at contact surfaces, they fail to capture the complex physics of extreme collisions, often producing non-physical artifacts and material adhesion at fracture interfaces. Our approach integrates two key innovations: (1) an enhanced Collision Material Point Method (Collision-MPM) with momentum-conserving interface forces derived from normalized mass distributions, which effectively eliminates unphysical adhesion in fractured solids; and (2) a fracture-aware 3D Gaussian continuum representation that enables physically plausible rendering without post-processing. The framework operates through three main stages: First, performing implicit reconstruction of collision objects from multi-view images while sampling both surface and internal particles and simultaneously learning surface particle Gaussian properties via splatting; Second, high-fidelity collision resolution using our improved Collision-MPM formulation; Third, dynamic fracture tracking with Gaussian attribute optimization for fracture surfaces rendering. 	
Through comprehensive testing, our framework demonstrates significant improvements over existing methods in handling diverse scenarios, including homogeneous materials, heterogeneous composites, and complex multi-body collisions. The results confirm superior physical accuracy, while maintaining computational efficiency for rendering.


## Environment Setup
### Using Conda
To create and activate the environment using the provided `env.yaml` file:
```bash
conda env create -f env.yaml
conda activate PhysGaussian
```

## Directory Structure
### Key Directories and Files
- **./mpm_solver_warp/**: Core MPM solver implementation
  - `mpm_solver_warp.py`: Main MPM system handling particles, grids, and physics parameters
    - Implements the complete MPM simulation loop
    - Manages particle-grid interactions and constitutive model updates
    - Handles collision detection and response
  - `mpm_utils.py`: Warp kernel functions for MPM operations
    - Particle-to-Grid (P2G) and Grid-to-Particle (G2P) transfers
    - Grid velocity update and constitutive model evaluations
  - `warp_utils.py`: Particle and grid data structures
    - Defines particle properties (mass, volume, velocity)
    - Grid node definitions and state management

- **./config/**: Simulation configuration files
  - Contains JSON parameter files for different scenarios (e.g., `ficus-wind.json`, `table2_bowl.json`)
  - Each file defines scene parameters, material properties, and external forces

- **./particle_filling/**: Particle filling utilities
  - `filling.py`: Internal particle filling and volume calculation

- **./utils/**: Utility functions
  - `transformation_utils.py`: Object position transformations

## Configuration Parameters
### Simulation Environment
- `substep_dt`: Simulation time step
- `frame_dt`: Frame saving step
- `frame_num`: Number of frames to save
- `g`: Gravity acceleration (3D vector)
- `surface_collider`: Rigid ground plane
- `enforce_particle_translation`: Velocity constraints for specific particle ranges
- Camera parameters:
  - `init_azimuthm`: Initial horizontal rotation angle around the scene (degrees)
  - `init_elevation`: Initial vertical rotation angle above the horizon (degrees)
  - `init_radius`: Initial camera distance from the viewpoint center
- `mpm_space_vertical_upward_axis`: MPM space upward direction
- `mpm_space_viewpoint_center`: MPM space viewpoint center

### MPM Grid
- `n_grid`: Grid dimensions (default: same for x, y, z)
- `grild_lim`: Grid length (default: same for x, y, z)

### Material Properties
- `E`: Young's modulus
- `nu`: Poisson's ratio
- `density`: Material density
- `beta`: Cohesion factor
- `Jp`: Hardening tracking coefficient
- `kesai`: Hardening factor
- `material`: Constitutive model type (jelly, metal, sand, foam, snow, plasticine)


## Running Simulations
### Basic Command Structure
```bash
python gs_simulation.py --model_path <path to gs model> --output_path <path to output folder> --config <path to json config file> --render_img --compile_video --white_bg(render white background) --output_ply
```

### Detailed Example: Wind Simulation on Ficus Model
```bash
python gs_simulation-ficus-auto-wind2.py \
  --model_path model/ficus_whitebg-trained \
  --config config/ficus_wind.json \
  --output_path output/ficus_wind_sim \
  --output_ply
  --render_img
  --compile_video
  --white_bg
```

### Simulation Result
<video controls width="640" height="480">
  <source src="output/ficus_wind_sim1/branch_E_1000000.0_nu_0.39_beta_2_Jp_0.94_kesai_3_density_5/output.mp4" type="video/mp4">
  Your browser does not support the video tag.
</video>

## Data Preprocessing
### Overview
The MPM data pipeline converts raw 3D assets into simulation-ready particle systems through MPM-specific preprocessing steps:

### Key Steps
1. **Particle Initialization**
   - Load mesh geometry from PLY files using `particle_filling` utilities
   - Generate volumetric particles through tetrahedral decomposition
   - Initialize particle position, velocity and material properties

2. **Grid Configuration**
   - Set MPM grid dimensions via `n_grid` parameter
   - Define simulation domain bounds using `grid_lim`
   - Allocate memory for grid nodes and velocity fields

3. **Material Assignment**
   - Map JSON configuration parameters to MPM solver
   - Assign Young's modulus (E), Poisson's ratio (nu) and density
   - Configure constitutive model parameters (beta, Jp, kesai)

4. **Boundary Setup**
   - Initialize collision geometry from `surface_collider` settings
   - Precompute Dirichlet boundary constraints
   - Set initial camera parameters for visualization

## Boundary Condition
### Overview
MPM boundary conditions are enforced through grid-based constraints and particle interactions, implemented in `mpm_solver_warp.py`:

### Types of Boundaries
1. **Dirichlet Boundaries**
   - **MPM Implementation**: Enforced during Grid-to-Particle (G2P) transfer
   - Fixed velocity constraints applied directly to grid nodes
   - Configured via `enforce_particle_translation` in JSON with particle group IDs
   - Used for controlled motion of specific objects (e.g., moving platforms)

2. **Neumann Boundaries**
   - **MPM Implementation**: Applied as external forces during Particle-to-Grid (P2G) transfer
   - Force profiles defined through `external_force` parameters in JSON configs
   - Supports time-varying force fields (e.g., wind, explosion impulses)

3. **Contact Boundaries**
   - **MPM Implementation**: Detected during grid traversal phase
   - Rigid surface colliders (`surface_collider`) with friction coefficient
   - Implemented using penalty method for collision response
   - Ground plane defined by `ground_height` parameter

### Configuration Example
```json
"boundary_conditions": {
  "surface_collider": true,
  "ground_height": 0.0,
  "friction_coefficient": 0.3,
  "enforce_particle_translation": [
    {"particle_group": 0, "velocity": [0.0, -1.0, 0.0]}
  ],
  "external_force": {
    "type": "wind",
    "magnitude": 5.0,
    "direction": [1.0, 0.0, 0.0]
  }
}
```


## Important Notes

### Point Cloud Requirements
To use this code, you need to provide the initial point cloud for the relevant model. This point cloud serves the following core functions:
- Simulation initialization and scene construction
- Basic data for Gaussian model training
- Particle system generation and mesh deformation calculations
- Initial geometric definition for rendering effects

Please name the point cloud file `point_cloud.ply` and place it in the following path:
`model/ficus_whitebg-trained/point_cloud/iteration_60000/`