# DePT based on CoOp
# adopt from https://github.com/Koorye/DePT

import os
import os.path as osp
import torch
import torch.nn.functional as F
from dassl.engine import TRAINER_REGISTRY
from dassl.utils import load_pretrained_weights, load_checkpoint
from dassl.optim import build_lr_scheduler
from plotnine import *
from torch.cuda.amp import GradScaler, autocast
from torch import nn

from .coop import CoOp, load_clip_to_cpu
from .coop import CustomCLIP as CustomCLIP_
from tools.ood_search import ClassNameIterator


class FiLM(nn.Module):
    def __init__(self, 
                 dim, 
                 bias=True, 
                 use_sigmoid=False):
        super().__init__()
        self.scale = nn.Parameter(torch.ones(dim))
        self.bias = nn.Parameter(torch.zeros(dim)) if bias else None
        self.has_bias = bias
        self.use_sigmoid = use_sigmoid
    
    def forward(self, x):
        scale = self.scale.unsqueeze(0).type(x.dtype)
        bias = self.bias.unsqueeze(0).type(x.dtype) if self.has_bias else None
        
        x = scale * x
        if bias is not None:
            x = x + bias
        
        if self.use_sigmoid:
            return x.sigmoid()
        
        return x

class CustomCLIP(CustomCLIP_):
    def __init__(self, cfg, classnames_id, classnames_ood, clip_model):
        super().__init__(cfg, classnames_id, classnames_ood, clip_model)
        self.subsample_classes = cfg.DATASET.SUBSAMPLE_CLASSES
        self.dataset = cfg.DATASET.NAME
        self.lp_cfg = cfg.TRAINER.LINEAR_PROBE
        self.film_cfg = cfg.TRAINER.FILM
        

        clip_dim = clip_model.text_projection.size(1)
        
        film_cfg = self.film_cfg

        if film_cfg.LINEAR_PROBE:
            # cwT module
            self.film_lp_img = FiLM(clip_dim)
            self.film_lp_text = FiLM(clip_dim)
        
        # for base to new, base classes will be 'base'
        # for cross dataset, classes from ImageNet will be 'base'
        if (self.subsample_classes == 'base') \
        or (self.subsample_classes == 'all' and 'ImageNet' in self.dataset):
            assert self.lp_cfg.TYPE in ['similarity', 'linear']

            # linear classifier
            if self.lp_cfg.TYPE == 'similarity':
                self.linear_probe_proj = nn.Identity()
            elif self.lp_cfg.TYPE == 'linear':
                self.linear_probe_proj = nn.Linear(clip_dim, len(classnames_id)).type(self.dtype)
        else:
            self.linear_probe_proj = nn.Identity()

        
    def forward(self, img, labels=None):
        if (self.subsample_classes == 'base') \
        or (self.subsample_classes == 'all' and 'ImageNet' in self.dataset):
            return self._forward_base(img, labels)
        else:
            return self._forward_new(img)

    def _forward_base(self, img, labels=None):
        """ forward function for base classes """
        if self.prompt_learner.training and self.cfg.OOD.LOSS_REG:
            text_feats, img_feats, text_features_ood_ft, text_features_ood_zs = self._forward_feats(img) # ood reg
        else:
            text_feats, img_feats = self._forward_feats(img)
        
        # forward similartiy and linear logits
        logits = self._forward_logits_similarity(text_feats, img_feats)
        logits_lp, labels_lp = self._forward_logits_linear_probe(text_feats, img_feats, labels)
        
        if self.prompt_learner.training:
            # while training, return loss of both logits
            loss_ori = self._loss(logits, labels, logits_lp, labels_lp)


            if self.cfg.OOD.LOSS_REG:
                cos = torch.nn.CosineSimilarity(dim=1,eps=1e-07)
                score = cos(text_features_ood_ft, text_features_ood_zs)
                loss_text_reg = 1.0-torch.mean(score)
                loss = loss_ori + self.alpha * loss_text_reg
            else:
                loss =  loss_ori

            return loss
        
        if not self.lp_cfg.TEST_TIME_FUSION:
            return logits_lp, img_feats, text_feats

        # while inference, fusion both logits and return
        lp_weight = self.lp_cfg.WEIGHT
        logits = (1 - lp_weight) * logits + lp_weight * logits_lp
        return logits, img_feats, text_feats
    
    def _forward_new(self, img):
        """ forward function for new classes """
        assert not self.prompt_learner.training
        
        # for new classes, only forward similarity logits
        text_feats, img_feats = self._forward_feats(img)
        logits = self._forward_logits_similarity(text_feats, img_feats)
        return logits, img_feats, text_feats
    
    def _forward_feats(self, img):

        if self.cfg.OOD.LOSS_REG:
            if self.prompt_learner.training: 
                # image fearture
                img_feats = self.image_encoder(img.type(self.dtype))  

                # text feature
                prompts_id, prompts_ood_zs_tokenized, prompts_ood_ft, prompts_ood_ft_tokenized = self.prompt_learner()
                prompts_id_tokenized = self.tokenized_prompts.cuda()

                prompts_mix = torch.cat([prompts_id, prompts_ood_ft], dim=0)
                tokenized_prompts_mix = torch.cat([prompts_id_tokenized, prompts_ood_ft_tokenized], dim=0)
                text_features_mix = self.text_encoder(prompts_mix, tokenized_prompts_mix)

                text_features_id = text_features_mix[:prompts_id.size(0)]
                text_features_ood_ft = text_features_mix[prompts_id.size(0):]

                with torch.no_grad():
                    text_features_ood_zs = self.encode_text(prompts_ood_zs_tokenized).type(self.dtype)
                
                text_feats = text_features_id

                return text_feats, img_feats, text_features_ood_ft, text_features_ood_zs

            else:
                prompts, _, _, _ = self.prompt_learner()
                tokenized_prompts = self.tokenized_prompts
                text_feats = self.text_encoder(prompts, tokenized_prompts)
                img_feats = self.image_encoder(img.type(self.dtype))  
        else:
            prompts = self.prompt_learner()
            tokenized_prompts = self.tokenized_prompts
            text_feats = self.text_encoder(prompts, tokenized_prompts)
            img_feats = self.image_encoder(img.type(self.dtype))

        return text_feats, img_feats
    
    def _forward_logits_similarity(self, text_feats, img_feats):
        # normalize and calcute cosine similarity
        text_feats = text_feats / text_feats.norm(dim=-1, keepdim=True)
        img_feats = img_feats / img_feats.norm(dim=-1, keepdim=True)
        logit_scale = self.logit_scale.exp()
        logits = logit_scale * img_feats @ text_feats.t()
        return logits
    
    def _forward_logits_linear_probe(self, text_feats, img_feats, labels):
        # cwT module
        if self.film_cfg.LINEAR_PROBE:
            text_feats = self.film_lp_text(text_feats)
            img_feats = self.film_lp_img(img_feats)

        # while new head is similarity head, use similarity forward function
        if self.lp_cfg.TYPE == 'similarity':
            return self._forward_logits_similarity(text_feats, img_feats), labels
 
        if labels is None:
            # while inference, forward image features only
            all_feats = img_feats
            all_labels = labels
        else:
            # while training, image features and text features will be concated to train classifier
            text_feats = text_feats[labels]
            all_feats = torch.cat([text_feats, img_feats])
            all_labels = torch.cat([labels, labels])

        all_logits = self.linear_probe_proj(all_feats)
        return all_logits, all_labels
    
    def _loss(self, logits, labels, logits_lp, labels_lp):
        # calculate similarity loss and linear loss
        loss_cls = F.cross_entropy(logits, labels)
        loss_cls_lp = F.cross_entropy(logits_lp, labels_lp)

        lp_weight = self.lp_cfg.WEIGHT
        loss = (1 - lp_weight) * loss_cls + lp_weight * loss_cls_lp
        return loss
    

@TRAINER_REGISTRY.register()
class DEPT(CoOp):
    def forward_backward(self, batch):
        image, label = self.parse_batch_train(batch)

        model = self.model
        optim = self.optim
        scaler = self.scaler

        prec = self.cfg.TRAINER.COOP.PREC
        if prec == "amp":
            with autocast():
                loss = model(image, label)
            optim.zero_grad()
            scaler.scale(loss).backward()
            scaler.step(optim)
            scaler.update()
        else:
            loss = model(image, label)
            optim.zero_grad()
            loss.backward()
            optim.step()

        loss_summary = {"loss": loss.item()}

        if (self.batch_idx + 1) == self.num_batches:
            self.update_lr()

        return loss_summary

    def parse_batch_train(self, batch):
        input = batch["img"]
        label = batch["label"]
        input = input.to(self.device)
        label = label.to(self.device)
        return input, label

    def build_model(self):
        cfg = self.cfg
        classnames_id = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)

        ######## ood #########
        if self.cfg.OOD.LOSS_REG:
            classnames_ood = self.ood_classnames
            length = len(classnames_id)
        else:
            classnames_ood = None

        ######## ood #########

        if cfg.TRAINER.COOP.PREC == "fp32" or cfg.TRAINER.COOP.PREC == "amp":
            clip_model.float()

        print("Building custom CLIP")
        self.model = CustomCLIP(cfg, classnames_id, classnames_ood, clip_model)

        print("Turning off gradients in both the image and the text encoder")
        names_to_update = cfg.TRAINER.NAMES_TO_UPDATE

        for name, param in self.model.named_parameters():
            update = False

            for name_to_update in names_to_update:
                if name_to_update in name:
                    update = True
                    break
                
            param.requires_grad_(update)

        enabled = []
        for name, param in self.model.named_parameters():
            if param.requires_grad:
                enabled.append(name)
        print(f"Parameters to be updated: {list(sorted(enabled))}")

        if cfg.MODEL.INIT_WEIGHTS:
            load_pretrained_weights(self.model, cfg.MODEL.INIT_WEIGHTS)

        self.model.to(self.device)
        self.optim, infos = build_optimizer(self.model, cfg)

        if infos is not None:
            print('Learning rate of parameters:')
            for info in infos:
                print('lr: {}, layers: {}'.format(info['lr'], info['layers']))
        
        self.sched = build_lr_scheduler(self.optim, cfg.OPTIM)
        self.register_model("PromptLearner", self.model, self.optim, self.sched)

        self.scaler = GradScaler() if cfg.TRAINER.COOP.PREC == "amp" else None

        device_count = torch.cuda.device_count()
        if device_count > 1:
            print(f"Multiple GPUs detected (n_gpus={device_count}), use all of them!")
            self.model = nn.DataParallel(self.model)

    def load_model(self, directory, epoch=None):
        if not directory:
            print("Note that load_model() is skipped as no pretrained model is given")
            return

        names = self.get_model_names()

        # By default, the best model is loaded
        model_file = "model-best.pth.tar"

        if epoch is not None:
            model_file = "model.pth.tar-" + str(epoch)

        for name in names:
            if epoch < 0:
                all_model_files = os.listdir(osp.join(directory, name))
                all_model_files = [file_ for file_ in all_model_files if file_ != 'checkpoint']
                model_epochs = [int(file_.split('-')[-1]) for file_ in all_model_files]
                last_epoch = max(model_epochs)
                model_file = 'model.pth.tar-' + str(last_epoch)

            model_path = osp.join(directory, name, model_file)

            if not osp.exists(model_path):
                raise FileNotFoundError('Model not found at "{}"'.format(model_path))

            checkpoint = load_checkpoint(model_path)
            state_dict = checkpoint["state_dict"]
            epoch = checkpoint["epoch"]

            # Ignore fixed token vectors
            if "prompt_learner.token_prefix" in state_dict:
                del state_dict["prompt_learner.token_prefix"]

            if "prompt_learner.token_suffix" in state_dict:
                del state_dict["prompt_learner.token_suffix"]

            print("Loading weights to {} " 'from "{}" (epoch = {})'.format(name, model_path, epoch))

            # for some dataset in domain generalization, number of target classes is different from number of source classes
            # thus a mapping must be created to preserve the required class weights
            if self.cfg.DATASET.NAME in ['ImageNetA', 'ImageNetR']:
                from datasets.imagenet import ImageNet
                from dassl.utils import listdir_nohidden

                # read classes from source dataset
                dataset = self.dm.dataset
                text_file = osp.join(dataset.dataset_dir, "classnames.txt")
                all_folders = ImageNet.read_classnames(text_file).keys()

                # read classes from target dataset
                TO_BE_IGNORED = ["README.txt"]
                folders = listdir_nohidden(dataset.image_dir, sort=True)
                folders = [f for f in folders if f not in TO_BE_IGNORED]

                # find that which class from target dataset is in source dataset
                is_reserves = [f in folders for f in all_folders]

                # only reserve required class weights
                print(f'State dict is CLIPPED to match the shape of target dataset {self.cfg.DATASET.NAME}!')
                state_dict['linear_probe_proj.weight'] = state_dict['linear_probe_proj.weight'][is_reserves]
                state_dict['linear_probe_proj.bias'] = state_dict['linear_probe_proj.bias'][is_reserves]
                
            # set strict=False
            self._models[name].load_state_dict(state_dict, strict=False)

""" modified from dassl.optim """
import warnings
import torch
import torch.nn as nn

from dassl.optim.radam import RAdam

AVAI_OPTIMS = ['adam', 'amsgrad', 'sgd', 'rmsprop', 'radam', 'adamw']


def build_optimizer(model, cfg, param_groups=None):
    optim = cfg.OPTIM.NAME
    lr = cfg.OPTIM.LR
    weight_decay = cfg.OPTIM.WEIGHT_DECAY
    momentum = cfg.OPTIM.MOMENTUM
    sgd_dampening = cfg.OPTIM.SGD_DAMPNING
    sgd_nesterov = cfg.OPTIM.SGD_NESTEROV
    rmsprop_alpha = cfg.OPTIM.RMSPROP_ALPHA
    adam_beta1 = cfg.OPTIM.ADAM_BETA1
    adam_beta2 = cfg.OPTIM.ADAM_BETA2
    staged_lr = cfg.DEPT_OPTIM.STAGED_LR
    new_layers = cfg.DEPT_OPTIM.NEW_LAYERS
    base_lr_mult = cfg.DEPT_OPTIM.BASE_LR_MULT

    if optim not in AVAI_OPTIMS:
        raise ValueError(
            f'optim must be one of {AVAI_OPTIMS}, but got {optim}'
        )

    if param_groups is not None and staged_lr:
        warnings.warn(
            'staged_lr will be ignored, if you need to use staged_lr, '
            'please bind it with param_groups yourself.'
        )

    if param_groups is None:
        if staged_lr:
            # modify the function of lr_mult
            exp = cfg.DEPT_OPTIM.LR_EXP
            lr *= exp
            base_lr_mult /= exp
            
            if not isinstance(model, nn.Module):
                raise TypeError(
                    'When staged_lr is True, model given to '
                    'build_optimizer() must be an instance of nn.Module'
                )

            if isinstance(model, nn.DataParallel):
                model = model.module

            if isinstance(new_layers, str):
                if new_layers is None:
                    warnings.warn('new_layers is empty (staged_lr is useless)')
                new_layers = [new_layers]

            base_params, new_params = [], []
            base_layers, new_layers_ = [], []

            for name, module in model.named_children():
                is_new = False

                for layer in new_layers:
                    if layer in name:
                        is_new = True
                        break
                    
                if is_new:
                    new_params += [p for p in module.parameters()]
                    new_layers_.append(name)
                else:
                    base_params += [p for p in module.parameters()]
                    base_layers.append(name)
            
            param_groups = [{'params': base_params,
                             'lr': lr * base_lr_mult}, 
                            {'params': new_params}]

            # return lr of each layer
            infos = [{'layers': base_layers, 
                      'lr': lr * base_lr_mult},
                     {'layers': new_layers_,
                      'lr': lr}]
        else:
            if isinstance(model, nn.Module):
                param_groups = model.parameters()
            else:
                param_groups = model
            
            infos = None

    if optim == 'adam':
        optimizer = torch.optim.Adam(
            param_groups,
            lr=lr,
            weight_decay=weight_decay,
            betas=(adam_beta1, adam_beta2),
        )

    elif optim == 'amsgrad':
        optimizer = torch.optim.Adam(
            param_groups,
            lr=lr,
            weight_decay=weight_decay,
            betas=(adam_beta1, adam_beta2),
            amsgrad=True,
        )

    elif optim == 'sgd':
        optimizer = torch.optim.SGD(
            param_groups,
            lr=lr,
            momentum=momentum,
            weight_decay=weight_decay,
            dampening=sgd_dampening,
            nesterov=sgd_nesterov,
        )

    elif optim == 'rmsprop':
        optimizer = torch.optim.RMSprop(
            param_groups,
            lr=lr,
            momentum=momentum,
            weight_decay=weight_decay,
            alpha=rmsprop_alpha,
        )

    elif optim == 'radam':
        optimizer = RAdam(
            param_groups,
            lr=lr,
            weight_decay=weight_decay,
            betas=(adam_beta1, adam_beta2),
        )

    elif optim == 'adamw':
        optimizer = torch.optim.AdamW(
            param_groups,
            lr=lr,
            weight_decay=weight_decay,
            betas=(adam_beta1, adam_beta2),
        )
    else:
        raise NotImplementedError(f'Optimizer {optim} not implemented yet!')

    return optimizer, infos
