import os.path as osp

import torch
import torch.nn as nn
from torch.nn import functional as F
from torch.cuda.amp import GradScaler, autocast
from torch.nn import functional as F
from torch.cuda.amp import GradScaler, autocast
from clip import clip
import time
import numpy as np
import os.path as osp
import datetime
import torch.nn.functional as F

from dassl.utils import (
    MetricMeter, AverageMeter, tolist_if_not, count_num_param, load_checkpoint,
    save_checkpoint, mkdir_if_missing, resume_from_checkpoint,
    load_pretrained_weights
)

from tqdm import tqdm
from dassl.engine import TRAINER_REGISTRY, TrainerX
from dassl.metrics import compute_accuracy
from dassl.utils import load_pretrained_weights, load_checkpoint
from dassl.optim import build_optimizer, build_lr_scheduler

from trainers.classification.base_learner import VLBaseLearner
from clip import clip
from clip.simple_tokenizer import SimpleTokenizer as _Tokenizer

_tokenizer = _Tokenizer()


def load_clip_to_cpu(cfg):
    backbone_name = cfg.MODEL.BACKBONE.NAME
    url = clip._MODELS[backbone_name]
    model_path = clip._download(url)

    try:
        # loading JIT archive
        model = torch.jit.load(model_path, map_location="cpu").eval()
        state_dict = None

    except RuntimeError:
        state_dict = torch.load(model_path, map_location="cpu")
    design_details = {"trainer": 'CoOp',
                      "vision_depth": 0,
                      "language_depth": 0, "vision_ctx": 0,
                      "language_ctx": 0}
    model = clip.build_model(state_dict or model.state_dict(), design_details)

    return model


class TextEncoder(nn.Module):
    def __init__(self, clip_model):
        super().__init__()
        self.transformer = clip_model.transformer
        self.positional_embedding = clip_model.positional_embedding
        self.ln_final = clip_model.ln_final
        self.text_projection = clip_model.text_projection
        self.dtype = clip_model.dtype

    def forward(self, prompts, tokenized_prompts):
        x = prompts + self.positional_embedding.type(self.dtype)
        x = x.permute(1, 0, 2)  # NLD -> LND
        x = self.transformer(x)
        x = x.permute(1, 0, 2)  # LND -> NLD
        x = self.ln_final(x).type(self.dtype)

        # x.shape = [batch_size, n_ctx, transformer.width]
        # take features from the eot embedding (eot_token is the highest number in each sequence)
        x = x[torch.arange(x.shape[0]), tokenized_prompts.argmax(dim=-1)] @ self.text_projection

        return x

class Adapter(nn.Module):
    def __init__(self, c_in, reduction=4):
        super(Adapter, self).__init__()
        self.fc = nn.Sequential(
            nn.Linear(c_in, c_in // reduction, bias=False),
            nn.ReLU(inplace=True),
            nn.Linear(c_in // reduction, c_in, bias=False),
            nn.ReLU(inplace=True)
        )

    def forward(self, x):
        x = self.fc(x)
        return x


class CustomCLIP(nn.Module):
    def __init__(self, cfg, classnames, clip_model):
        super().__init__()
        self.cfg = cfg
        self.image_encoder = clip_model.visual
        self.text_encoder = TextEncoder(clip_model)
        self.logit_scale = clip_model.logit_scale
        self.dtype = clip_model.dtype
        self.adapter = Adapter(512, 4).to(clip_model.dtype)
        self.text_features = self.get_text_features(cfg, classnames)
        self.ratio = cfg.TRAINER.CLIP_ADAPTER.RATIO
        self.alpha = self.cfg.OOD.ALPHA

        

    def forward(self, image_id, label=None, image_ood=None):

        # text features
        text_features = self.text_features
        text_features = text_features / text_features.norm(dim=-1, keepdim=True)


        if self.cfg.OOD.LOSS_REG and image_ood is not None:
            image_mix = torch.cat([image_id, image_ood], dim=0)
            image_features_mix = self.image_encoder(image_mix.type(self.dtype))
            image_features_id = image_features_mix[:image_id.size(0)]
            image_features_ood_zs = image_features_mix[image_id.size(0):]

            x_mix = self.adapter(image_features_mix)
            x_id = x_mix[:image_id.size(0)]
            image_features_ood_ft = x_mix[image_id.size(0):]

            ratio = self.ratio
            image_features_id = ratio * x_id + (1 - ratio) * image_features_id
            image_features_id = image_features_id / image_features_id.norm(dim=-1, keepdim=True)
        else:
            image_features_id = self.image_encoder(image_id.type(self.dtype))
            x_id = self.adapter(image_features_id)
            ratio = self.ratio
            image_features_id = ratio * x_id + (1 - ratio) * image_features_id
            image_features_id = image_features_id / image_features_id.norm(dim=-1, keepdim=True)

        logit_scale = self.logit_scale.exp()
        logits = logit_scale * image_features_id @ text_features.t()

        if self.adapter.training:
            loss_ori = F.cross_entropy(logits, label)
            if self.cfg.OOD.LOSS_REG:
                image_features_ood_zs = image_features_ood_zs / image_features_ood_zs.norm(dim=-1, keepdim=True)
                image_features_ood_ft = image_features_ood_ft / image_features_ood_ft.norm(dim=-1, keepdim=True)
                
                cos = torch.nn.CosineSimilarity(dim=1,eps=1e-07)
                score = cos(image_features_ood_zs, image_features_ood_ft)
                loss_image_reg = 1.0-torch.mean(score)
                loss = loss_ori + self.alpha * loss_image_reg
            else:
                loss =  loss_ori
            return loss
        
        return logits, image_features_id, text_features
    
    def get_text_features(self, cfg, classnames):

        clip_model_ = load_clip_to_cpu(cfg)
        clip_model_.cuda()

        ctx_init = cfg.TRAINER.CLIP_ADAPTER.CTX_INIT

        prompts = [ctx_init + " " + name + "." for name in classnames]
        prompts_tokenized = torch.cat([clip.tokenize(p) for p in prompts])
        prompts_tokenized = prompts_tokenized.cuda()

        with torch.no_grad():
            text_features = clip_model_.encode_text(prompts_tokenized)

        return text_features

@TRAINER_REGISTRY.register()
class CLIP_Adapter(VLBaseLearner):


    def check_cfg(self, cfg):
        assert cfg.TRAINER.COOP.PREC in ["fp16", "fp32", "amp"]

    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        
        if cfg.TRAINER.COOP.PREC == "fp32" or cfg.TRAINER.COOP.PREC == "amp":
            # CLIP's default precision is fp16
            clip_model.float()

        print("Building custom CLIP")
        self.model = CustomCLIP(cfg, classnames, clip_model)

        print("Turning off gradients in both the image and the text encoder")
        for name, param in self.model.named_parameters():
            if "adapter" not in name:
                param.requires_grad_(False)

        # Double check
        enabled = set()
        for name, param in self.model.named_parameters():
            if param.requires_grad:
                enabled.add(name)
        print(f"Parameters to be updated: {enabled}")

        if cfg.MODEL.INIT_WEIGHTS:
            load_pretrained_weights(self.model.adapter, cfg.MODEL.INIT_WEIGHTS)

        self.model.to(self.device)
        # NOTE: only give adapter to the optimizer
        self.optim = build_optimizer(self.model.adapter, cfg.OPTIM)
        self.sched = build_lr_scheduler(self.optim, cfg.OPTIM)
        self.register_model("adapter", self.model.adapter, self.optim, self.sched)

        self.scaler = GradScaler() if cfg.TRAINER.COOP.PREC == "amp" else None

        # Note that multi-gpu training could be slow because CLIP's size is
        # big, which slows down the copy operation in DataParallel
        device_count = torch.cuda.device_count()
        if device_count > 1:
            print(f"Multiple GPUs detected (n_gpus={device_count}), use all of them!")
            # self.model = nn.DataParallel(self.model)
            self.model.text_encoder = nn.DataParallel(self.model.text_encoder)

    def run_epoch(self):
        self.set_model_mode("train")
        losses = MetricMeter()
        batch_time = AverageMeter()
        data_time = AverageMeter()
        self.num_batches = len(self.train_loader_x)

        end = time.time()
        if self.cfg.OOD.LOSS_REG:
            for self.batch_idx, (batch_id, batch_ood) in enumerate(zip(self.train_loader_x, self.ood_train_loader_x)):
                data_time.update(time.time() - end)
                loss_summary = self.forward_backward_ood(batch_id, batch_ood)
                batch_time.update(time.time() - end)
                losses.update(loss_summary)

                meet_freq = (self.batch_idx + 1) % self.cfg.TRAIN.PRINT_FREQ == 0
                only_few_batches = self.num_batches < self.cfg.TRAIN.PRINT_FREQ
                if meet_freq or only_few_batches:
                    nb_remain = 0
                    nb_remain += self.num_batches - self.batch_idx - 1
                    nb_remain += (
                        self.max_epoch - self.epoch - 1
                    ) * self.num_batches
                    eta_seconds = batch_time.avg * nb_remain
                    eta = str(datetime.timedelta(seconds=int(eta_seconds)))

                    info = []
                    info += [f"epoch [{self.epoch + 1}/{self.max_epoch}]"]
                    info += [f"batch [{self.batch_idx + 1}/{self.num_batches}]"]
                    info += [f"time {batch_time.val:.3f} ({batch_time.avg:.3f})"]
                    info += [f"data {data_time.val:.3f} ({data_time.avg:.3f})"]
                    info += [f"{losses}"]
                    info += [f"lr {self.get_current_lr():.4e}"]
                    info += [f"eta {eta}"]
                    print(" ".join(info))

                n_iter = self.epoch * self.num_batches + self.batch_idx
                for name, meter in losses.meters.items():
                    self.write_scalar("train/" + name, meter.avg, n_iter)
                self.write_scalar("train/lr", self.get_current_lr(), n_iter)

                end = time.time()
        else:
            for self.batch_idx, batch_id in enumerate(self.train_loader_x):
                data_time.update(time.time() - end)
                loss_summary = self.forward_backward_id(batch_id)
                batch_time.update(time.time() - end)
                losses.update(loss_summary)

                meet_freq = (self.batch_idx + 1) % self.cfg.TRAIN.PRINT_FREQ == 0
                only_few_batches = self.num_batches < self.cfg.TRAIN.PRINT_FREQ
                if meet_freq or only_few_batches:
                    nb_remain = 0
                    nb_remain += self.num_batches - self.batch_idx - 1
                    nb_remain += (
                        self.max_epoch - self.epoch - 1
                    ) * self.num_batches
                    eta_seconds = batch_time.avg * nb_remain
                    eta = str(datetime.timedelta(seconds=int(eta_seconds)))

                    info = []
                    info += [f"epoch [{self.epoch + 1}/{self.max_epoch}]"]
                    info += [f"batch [{self.batch_idx + 1}/{self.num_batches}]"]
                    info += [f"time {batch_time.val:.3f} ({batch_time.avg:.3f})"]
                    info += [f"data {data_time.val:.3f} ({data_time.avg:.3f})"]
                    info += [f"{losses}"]
                    info += [f"lr {self.get_current_lr():.4e}"]
                    info += [f"eta {eta}"]
                    print(" ".join(info))

                n_iter = self.epoch * self.num_batches + self.batch_idx
                for name, meter in losses.meters.items():
                    self.write_scalar("train/" + name, meter.avg, n_iter)
                self.write_scalar("train/lr", self.get_current_lr(), n_iter)

                end = time.time()

    def forward_backward_id(self, batch_id):
        image_id, label_id = self.parse_batch_train(batch_id)

        model = self.model
        optim = self.optim
        scaler = self.scaler

        prec = self.cfg.TRAINER.COOP.PREC
        if prec == "amp":
            with autocast():
                loss = model(image_id, label_id)
            optim.zero_grad()
            scaler.scale(loss).backward()
            scaler.step(optim)
            scaler.update()
        else:
            loss = model(image_id, label_id)
            optim.zero_grad()
            loss.backward()
            optim.step()

        loss_summary = {
            "loss": loss.item()
        }

        if (self.batch_idx + 1) == self.num_batches:
            self.update_lr()

        return loss_summary

    def forward_backward_ood(self, batch_id, batch_ood):

        image_id, label_id = self.parse_batch_train(batch_id)
        image_ood, _ = self.parse_batch_train(batch_ood)
 

        model = self.model
        optim = self.optim
        scaler = self.scaler

        prec = self.cfg.TRAINER.COOP.PREC
        if prec == "amp":
            with autocast():
                loss = model(image_id, label_id, image_ood)
            optim.zero_grad()
            scaler.scale(loss).backward()
            scaler.step(optim)
            scaler.update()
        else:
            loss = model(image_id, label_id, image_ood)
            optim.zero_grad()
            loss.backward()
            optim.step()

        loss_summary = {
            "loss": loss.item()
        }

        if (self.batch_idx + 1) == self.num_batches:
            self.update_lr()

        return loss_summary

    def parse_batch_train(self, batch):
        input = batch["img"]
        label = batch["label"]
        input = input.to(self.device)
        label = label.to(self.device)
        return input, label

    def load_model(self, directory, epoch=None):
        if not directory:
            print("Note that load_model() is skipped as no pretrained model is given")
            return

        names = self.get_model_names()

        # By default, the best model is loaded
        model_file = "model-best.pth.tar"

        if epoch is not None:
            model_file = "model.pth.tar-" + str(epoch)

        for name in names:
            model_path = osp.join(directory, name, model_file)

            if not osp.exists(model_path):
                raise FileNotFoundError('Model not found at "{}"'.format(model_path))

            checkpoint = load_checkpoint(model_path)
            state_dict = checkpoint["state_dict"]
            epoch = checkpoint["epoch"]

            # Ignore fixed token vectors
            if "token_prefix" in state_dict:
                del state_dict["token_prefix"]

            if "token_suffix" in state_dict:
                del state_dict["token_suffix"]

            print("Loading weights to {} " 'from "{}" (epoch = {})'.format(name, model_path, epoch))
            # set strict=False
            self._models[name].load_state_dict(state_dict, strict=False)
