import os.path as osp

import torch
import torch.nn as nn
from torch.nn import functional as F
from torch.cuda.amp import GradScaler, autocast
from collections import OrderedDict
import scipy.io as sio
import random

from dassl.engine import TRAINER_REGISTRY, TrainerX
from dassl.metrics import compute_accuracy
from dassl.utils import load_pretrained_weights, load_checkpoint
from dassl.optim import build_lr_scheduler

from clip_text import clip
from clip_text.simple_tokenizer import SimpleTokenizer as _Tokenizer
import tqdm
import numpy as np
import copy
from tools.ood_search import ClassNameIterator

from trainers.classification.base_learner import VLBaseLearner

def load_clip_to_cpu(cfg):
    backbone_name = cfg.MODEL.BACKBONE.NAME
    url = clip._MODELS[backbone_name]
    model_path = clip._download(url)

    try:
        # loading JIT archive
        model = torch.jit.load(model_path, map_location="cpu").eval()
        state_dict = None

    except RuntimeError:
        state_dict = torch.load(model_path, map_location="cpu")

    model = clip.build_model(state_dict or model.state_dict())

    return model


CUSTOM_TEMPLATES_ori = {
    "OxfordPets": "a photo of a {}, a type of pet.",
    "OxfordFlowers": "a photo of a {}, a type of flower.",
    "FGVCAircraft": "a photo of an aircraft {}.",
    "DescribableTextures": "a photo of a {}, a type of texture.",
    "EuroSAT": "a centered satellite photo of {}.",
    "StanfordCars": "a photo of a {}.",
    "Food101": "a photo of a {}, a type of food.",
    "SUN397": "a photo of a {}.",
    "Caltech101": "a photo of a {}.",
    "UCF101": "a photo of a person doing {}.",
    "ImageNet": "a photo of a {}.",
    "ImageNetSketch": "a photo of a {}.",
    "ImageNetV2": "a photo of a {}.",
    "ImageNetA": "a photo of a {}.",
    "ImageNetR": "a photo of a {}.",
}

CUSTOM_TEMPLATES = {
    "OxfordPets": "X X X X {}, a type of pet.",
    "OxfordFlowers": "X X X X {}, a type of flower.",
    "FGVCAircraft": "X X X X {}, a type of aircraft.",
    "DescribableTextures": "X X X X {} texture.",
    "EuroSAT": "X X X X {}.",
    "StanfordCars": "X X X X {}, a type of car",
    "Food101": "X X X X {}, a type of food.",
    "SUN397": "X X X X {}.",
    "Caltech101": "X X X X {}.",
    "UCF101": "a photo of a person doing {}.",
    "ImageNet": "a photo of a {}.",
    "ImageNetSketch": "a photo of a {}.",
    "ImageNetV2": "a photo of a {}.",
    "ImageNetA": "a photo of a {}.",
    "ImageNetR": "a photo of a {}.",
}


class TextEncoder(nn.Module):
    def __init__(self, clip_model):
        super().__init__()
        self.transformer = clip_model.transformer
        self.positional_embedding = clip_model.positional_embedding
        self.ln_final = clip_model.ln_final
        self.text_projection = clip_model.text_projection
        self.dtype = clip_model.dtype

    def forward(self, prompts, class_feature_id, weight, tokenized_prompts,flag=False):
        x = prompts + self.positional_embedding.type(self.dtype)
        x = x.permute(1, 0, 2)  # NLD -> LND
        if flag:
            x = self.transformer(x)
        else:
            counter=0
            outputs = self.transformer.resblocks([x,class_feature_id,weight,counter])
            x = outputs[0]

        x = x.permute(1, 0, 2)  # LND -> NLD
        x = self.ln_final(x).type(self.dtype)
        x = x[torch.arange(x.shape[0]), tokenized_prompts.argmax(dim=-1)] @ self.text_projection
        return x

def _get_clones(module, N):
    return nn.ModuleList([copy.deepcopy(module) for i in range(N)])

class QuickGELU(nn.Module):
    def forward(self, x: torch.Tensor):
        return x * torch.sigmoid(1.702 * x)

class PromptLearner(nn.Module):
    def __init__(self, cfg, classnames_id, classnames_ood, clip_model):
        super().__init__()
        n_cls = len(classnames_id)
        n_ctx = cfg.TRAINER.TCP.N_CTX
        ctx_init = cfg.TRAINER.TCP.CTX_INIT
        dtype = clip_model.dtype
        ctx_dim = clip_model.ln_final.weight.shape[0]
        clip_imsize = clip_model.visual.input_resolution
        cfg_imsize = cfg.INPUT.SIZE[0]
        assert cfg_imsize == clip_imsize, f"cfg_imsize ({cfg_imsize}) must equal to clip_imsize ({clip_imsize})"

        if ctx_init:
            print("use given words to initialize context vectors")
            temp = 'a photo of a'
            ctx_init = temp.replace("_", " ")
            n_ctx = len(ctx_init.split(" "))
            prompt = clip.tokenize(ctx_init)
            with torch.no_grad():
                embedding = clip_model.token_embedding(prompt).type(dtype)
            
            ctx_vectors = embedding[0, 1 : 1 + n_ctx, :]
            prompt_prefix = ctx_init

            ctx_vectors_src = embedding[0, 1 : 1 + n_ctx, :]

        else:
            # random initialization
            if cfg.TRAINER.TCP.CSC:
                print("Initializing class-specific contexts")
                ctx_vectors = torch.empty(n_cls, n_ctx, ctx_dim, dtype=dtype)
            else:
                print("Initializing a generic context")
                ctx_vectors = torch.empty(n_ctx, ctx_dim, dtype=dtype)
            nn.init.normal_(ctx_vectors, std=0.02)
            prompt_prefix = " ".join(["X"] * n_ctx)


        print(f'Initial context: "{prompt_prefix}"')
        print(f"Number of context words (tokens): {n_ctx}")

        self.ctx = nn.Parameter(ctx_vectors)  # to be optimized

        clip_model_ = load_clip_to_cpu(cfg)
        clip_model_.cuda()

        temp = CUSTOM_TEMPLATES_ori[cfg.DATASET.NAME]
        prompts_ = [temp.format(c.replace("_", " ")) for c in classnames_id]
        prompts_ = torch.cat([clip.tokenize(p) for p in prompts_])
        prompts_ = prompts_.cuda()

        with torch.no_grad():
            text_features = clip_model_.encode_text(prompts_)
            self.text_features_id = text_features / text_features.norm(dim=-1, keepdim=True)

        vis_dim = clip_model.visual.output_dim
        self.meta_net = nn.Sequential(
            OrderedDict([("linear1", nn.Linear(vis_dim, vis_dim // 4,bias=True)),
                         ("relu", QuickGELU()),
                         ("linear2", nn.Linear(vis_dim // 4, 4*ctx_dim,bias=True))
                         ]))
        if cfg.TRAINER.COCOOP.PREC == "fp16":
            self.meta_net.half()
        classnames_id = [name.replace("_", " ") for name in classnames_id]
        temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        prompts = [temp.format(c.replace("_", " ")) for c in classnames_id]
        print(prompts)

        tokenized_prompts = torch.cat([clip.tokenize(p) for p in prompts])
        with torch.no_grad():
            embedding = clip_model.token_embedding(tokenized_prompts).type(dtype)
        
        self.register_buffer("token_prefix", embedding[:, :1, :])  # SOS
        self.register_buffer("token_suffix", embedding[:, 1 + n_ctx :, :])  # CLS, EOS
        self.n_cls = n_cls
        self.n_ctx = n_ctx
        self.tokenized_prompts = tokenized_prompts  # torch.Tensor
        self.class_token_position = cfg.TRAINER.TCP.CLASS_TOKEN_POSITION
        self.prev_ctx=None
        self.prompt_prefix = prompt_prefix
        self.classnames_ood = classnames_ood
        self.dtype = dtype
        self.n_ctx = n_ctx
        self.cfg = cfg
        self.token_embedding = clip_model.token_embedding
        self.temp_ood = cfg.OOD.TEMP
        self.clip_model_ = clip_model_
        self.classname_gen = ClassNameIterator(classnames_ood, n_cls, cfg.OOD.SAMPLING)
        
    def forward(self):
        # id prompt and class feature
        class_feature_id = self.meta_net(self.text_features_id)
        class_feature_id = class_feature_id.reshape(class_feature_id.shape[0],-1,512)
        prefix = self.token_prefix
        suffix = self.token_suffix
        ctx = self.ctx
        ctx = ctx.unsqueeze(0).expand(self.n_cls, -1, -1)
        prompts_id = torch.cat(
            [
                prefix,  # (n_cls, 1, dim)
                ctx,
                suffix,  # (n_cls, *, dim)
            ],
            dim=1,
        )

        # ood prompt and class feature
        # get ood classnames
        if self.cfg.OOD.LOSS_REG:
            # if len(self.classnames_ood) > prompts_id.shape[0]:
            #     classnames_ood = random.sample(self.classnames_ood, self.n_cls)
            # else:
            #     classnames_ood = self.classnames_ood
            classnames_ood = self.classname_gen.next_batch()
            classnames_ood = [name.replace("_", " ") for name in classnames_ood]
            prompts_ood_zs = [self.temp_ood + " " + name + "." for name in classnames_ood]
            prompts_ood_zs_tokenized = torch.cat([clip.tokenize(p) for p in prompts_ood_zs]).cuda()
            
            with torch.no_grad():
                text_features_ood_zs = self.clip_model_.encode_text(prompts_ood_zs_tokenized)
                text_features_ood_zs = text_features_ood_zs / text_features_ood_zs.norm(dim=-1, keepdim=True)
            class_feature_ood = self.meta_net(text_features_ood_zs)
            class_feature_ood = class_feature_ood.reshape(class_feature_ood.shape[0],-1,512)

            prompts_ood = [self.prompt_prefix + " " + name + "." for name in classnames_ood]
            prompts_ood_ft_tokenized = torch.cat([clip.tokenize(p) for p in prompts_ood]).cuda()  

            with torch.no_grad():
                embedding_ood = self.token_embedding(prompts_ood_ft_tokenized).type(self.dtype)

            token_prefix_ood = embedding_ood[:, :1, :]
            token_suffix_ood = embedding_ood[:, 1 + self.n_ctx:, :]
            ctx_ood = self.ctx
            if ctx_ood.dim() == 2:
                ctx_ood = ctx_ood.unsqueeze(0).expand(len(classnames_ood), -1, -1)
            prompts_ood_ft = torch.cat(
                [
                    token_prefix_ood,  # (n_cls, 1, dim)
                    ctx_ood,     # (n_cls, n_ctx, dim)
                    token_suffix_ood,  # (n_cls, *, dim)
                ],
                dim=1,
            )

            return prompts_id, class_feature_id, prompts_ood_zs_tokenized, prompts_ood_ft, prompts_ood_ft_tokenized, class_feature_ood, text_features_ood_zs

        else:
            return prompts_id, class_feature_id


class Adapter(nn.Module):
    def __init__(self, c_in, reduction=4):
        super(Adapter, self).__init__()
        self.fc = nn.Sequential(
            nn.Linear(c_in, c_in // reduction, bias=False),
            nn.ReLU(inplace=True),
            nn.Linear(c_in // reduction, c_in, bias=False),
            nn.ReLU(inplace=True)
        )

    def forward(self, x):
        x = self.fc(x)
        return x

class CustomCLIP(nn.Module):
    def __init__(self, cfg, classnames_id, classnames_ood, clip_model):
        super().__init__()
        self.prompt_learner = PromptLearner(cfg, classnames_id, classnames_ood, clip_model)
        self.tokenized_prompts = self.prompt_learner.tokenized_prompts
        self.ori_embedding = self.prompt_learner.text_features_id
        self.image_encoder = clip_model.visual
        self.text_encoder = TextEncoder(clip_model)
        self.logit_scale = clip_model.logit_scale
        self.dtype = clip_model.dtype
        self.domain_sim = -1
        self.domain_sim_src = -1
        self.weight = cfg.TRAINER.TCP.W
        self.cfg = cfg
        self.alpha = self.cfg.OOD.ALPHA
        self.encode_text = clip_model.encode_text
    
    def forward(self, image, label=None):
        # image features
        image_features = self.image_encoder(image.type(self.dtype))
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)

        # text features
        if self.cfg.OOD.LOSS_REG:
            prompts_id, class_prompt_id, prompts_ood_zs_tokenized, prompts_ood_ft, \
                prompts_ood_ft_tokenized, class_prompt_ood, text_features_ood_zs = self.prompt_learner()
            prompts_id_tokenized = self.tokenized_prompts.cuda()

            prompts_mix = torch.cat([prompts_id, prompts_ood_ft], dim=0)
            tokenized_prompts_mix = torch.cat([prompts_id_tokenized, prompts_ood_ft_tokenized], dim=0)
            class_prompt_mix = torch.cat([class_prompt_id, class_prompt_ood], dim=0)
            text_features_mix = self.text_encoder(prompts_mix, class_prompt_mix, self.weight, tokenized_prompts_mix.detach()) 
            text_features_mix = text_features_mix / text_features_mix.norm(dim=-1, keepdim=True)
            
            text_features_id = text_features_mix[:prompts_id.size(0)]
            text_features_ood_ft = text_features_mix[prompts_id.size(0):]    


        else:
            prompts_id_tokenized = self.tokenized_prompts
            prompts_id,class_prompt_id = self.prompt_learner()
            text_features_id = self.text_encoder(prompts_id, class_prompt_id, self.weight,prompts_id_tokenized.detach()) 
            text_features_id = text_features_id / text_features_id.norm(dim=-1, keepdim=True)

        # get logits
        logit_scale = self.logit_scale.exp()
        logits = logit_scale.detach() * image_features.detach() @ text_features_id.t()
        
        if self.prompt_learner.training:
            cos = torch.nn.CosineSimilarity(dim=1,eps=1e-07)
            text_features_old = self.ori_embedding
            text_features_old = text_features_old / text_features_old.norm(dim=-1, keepdim=True)
            score= cos(text_features_id,text_features_old)
            score  = 1.0-torch.mean(score)
            loss_ori = F.cross_entropy(logits, label)+8.0*score

            if self.cfg.OOD.LOSS_REG:
                score_ood = cos(text_features_ood_ft, text_features_ood_zs)
                loss_text_reg = 1.0-torch.mean(score_ood)
                loss = loss_ori + self.alpha * loss_text_reg
            else:
                loss =  loss_ori
                
            return loss
        else:
            return logits, image_features, text_features_id


@TRAINER_REGISTRY.register()
class TCP(VLBaseLearner):

    def check_cfg(self, cfg):
        assert cfg.TRAINER.TCP.PREC in ["fp16", "fp32", "amp"]

    def build_model(self):
        cfg = self.cfg
        classnames_id = self.dm.dataset.classnames
        print(classnames_id)
        self.n_cls = len(classnames_id)

        ######## ood #########
        if self.cfg.OOD.LOSS_REG:
            classnames_ood = self.ood_classnames
            length = len(classnames_id)
        else:
            classnames_ood = None
        ######## ood #########

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        
        if cfg.TRAINER.TCP.PREC == "fp32" or cfg.TRAINER.TCP.PREC == "amp":
            # CLIP's default precision is fp16
            clip_model.float()

        print("Building custom CLIP")
        self.model = CustomCLIP(cfg, classnames_id, classnames_ood, clip_model)
        self.w = cfg.TRAINER.TCP.W

        print("Turning off gradients in both the image and the text encoder")

        name_to_update = "prompt_learner"

        for name, param in self.model.named_parameters():
            if name_to_update not in name:
                param.requires_grad_(False)
            else:
                print(name)


        if cfg.MODEL.INIT_WEIGHTS:
            load_pretrained_weights(self.model.prompt_learner, cfg.MODEL.INIT_WEIGHTS)

        self.model.to(self.device)
        # NOTE: only give prompt_learner to the optimizer
        self.optim = build_optimizer(self.model.prompt_learner, cfg.OPTIM)
        self.sched = build_lr_scheduler(self.optim, cfg.OPTIM)
        self.register_model("prompt_learner", self.model.prompt_learner, self.optim, self.sched)
        
        self.scaler = GradScaler() if cfg.TRAINER.TCP.PREC == "amp" else None

        # Note that multi-gpu training could be slow because CLIP's size is
        # big, which slows down the copy operation in DataParallel
        device_count = torch.cuda.device_count()
        if device_count > 1:
            print(f"Multiple GPUs detected (n_gpus={device_count}), use all of them!")
            self.model = nn.DataParallel(self.model)
        self.proto=None

    def forward_backward(self, batch):
        image, label = self.parse_batch_train(batch)
        prec = self.cfg.TRAINER.TCP.PREC
        if prec == "amp":
            with autocast():
                output = self.model(image)
                loss = F.cross_entropy(output, label)
            self.optim.zero_grad()
            self.scaler.scale(loss).backward()
            self.scaler.step(self.optim)
            self.scaler.update()
        else:
            loss = self.model(image, label)
            self.model_backward_and_update(loss)

        loss_summary = {
            "loss": loss.item(),
        }

        # "acc": compute_accuracy(output, label)[0].item(),

        if (self.batch_idx + 1) == self.num_batches:
            self.update_lr()
        return loss_summary

    def parse_batch_train(self, batch):
        input = batch["img"]
        label = batch["label"]
        input = input.to(self.device)
        label = label.to(self.device)
        return input, label

    #def model_inference(self, input):
    #    return self.model(input)


    def load_model(self, directory, epoch=None):
        if not directory:
            print("Note that load_model() is skipped as no pretrained model is given")
            return

        names = self.get_model_names()
        print(names)

        # By default, the best model is loaded
        model_file = "model-best.pth.tar"

        if epoch is not None:
            model_file = "model.pth.tar-" + str(epoch)

        for name in names:
            model_path = osp.join(directory, name, model_file)

            if not osp.exists(model_path):
                raise FileNotFoundError('Model not found at "{}"'.format(model_path))

            checkpoint = load_checkpoint(model_path)
            state_dict = checkpoint["state_dict"]
            epoch = checkpoint["epoch"]

            # Ignore fixed token vectors
            if "token_prefix" in state_dict:
                del state_dict["token_prefix"]

            if "token_suffix" in state_dict:
                del state_dict["token_suffix"]

            if "token_midfix" in state_dict:
                del state_dict["token_midfix"]
            print("Loading weights to {} " 'from "{}" (epoch = {})'.format(name, model_path, epoch))
            # set strict=False
            self._models[name].load_state_dict(state_dict, strict=False)

"""
Modified from https://github.com/KaiyangZhou/deep-person-reid
"""
import warnings
import torch
import torch.nn as nn

AVAI_OPTIMS = ["adam", "amsgrad", "sgd", "rmsprop", "radam", "adamw"]


def build_optimizer(model, optim_cfg, param_groups=None):
    """A function wrapper for building an optimizer.

    Args:
        model (nn.Module or iterable): model.
        optim_cfg (CfgNode): optimization config.
        param_groups: If provided, directly optimize param_groups and abandon model
    """
    optim = optim_cfg.NAME
    lr = optim_cfg.LR
    weight_decay = optim_cfg.WEIGHT_DECAY
    momentum = optim_cfg.MOMENTUM
    sgd_dampening = optim_cfg.SGD_DAMPNING
    sgd_nesterov = optim_cfg.SGD_NESTEROV
    rmsprop_alpha = optim_cfg.RMSPROP_ALPHA
    adam_beta1 = optim_cfg.ADAM_BETA1
    adam_beta2 = optim_cfg.ADAM_BETA2
    staged_lr = optim_cfg.STAGED_LR
    new_layers = optim_cfg.NEW_LAYERS
    base_lr_mult = optim_cfg.BASE_LR_MULT

    if optim not in AVAI_OPTIMS:
        raise ValueError(
            f"optim must be one of {AVAI_OPTIMS}, but got {optim}"
        )

    if param_groups is not None and staged_lr:
        warnings.warn(
            "staged_lr will be ignored, if you need to use staged_lr, "
            "please bind it with param_groups yourself."
        )

    if param_groups is None:
        if staged_lr:
            if not isinstance(model, nn.Module):
                raise TypeError(
                    "When staged_lr is True, model given to "
                    "build_optimizer() must be an instance of nn.Module"
                )

            if isinstance(model, nn.DataParallel):
                model = model.module

            if isinstance(new_layers, str):
                if new_layers is None:
                    warnings.warn("new_layers is empty (staged_lr is useless)")
                new_layers = [new_layers]

            base_params = []
            base_layers = []
            new_params = []

            for name, module in model.named_children():
                if name in new_layers:
                    new_params += [p for p in module.parameters()]
                else:
                    base_params += [p for p in module.parameters()]
                    base_layers.append(name)

            param_groups = [
                {
                    "params": base_params,
                    "lr": lr * base_lr_mult
                },
                {
                    "params": new_params
                },
            ]

        else:
            if isinstance(model, nn.Module):
                param_groups = model.parameters()
            else:
                param_groups = model

    if optim == "adam":
        optimizer = torch.optim.Adam(
            param_groups,
            lr=lr,
            weight_decay=weight_decay,
            betas=(adam_beta1, adam_beta2),
            eps=1e-3,
        )

    elif optim == "sgd":
        optimizer = torch.optim.SGD(
            param_groups,
            lr=lr,
            momentum=momentum,
            weight_decay=weight_decay,
            dampening=sgd_dampening,
            nesterov=sgd_nesterov,
        )


    elif optim == "adamw":
        optimizer = torch.optim.AdamW(
            param_groups,
            lr=lr,
            weight_decay=weight_decay,
            betas=(adam_beta1, adam_beta2),
        )
    else:
        raise NotImplementedError(f"Optimizer {optim} not implemented yet!")

    return optimizer
