#!/usr/bin/env bash
# Run All KL Ablation Experiments
set -euo pipefail

# ========================
# Configuration
# ========================
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
LOG_DIR="/primus_oss/_checkpoint/0910-Qwen3-8B-AIME2024/ablation_logs"
TIMESTAMP=$(date +%Y%m%d_%H%M%S)

# Create log directory
mkdir -p "${LOG_DIR}"

# List of ablation experiments
EXPERIMENTS=(
    "ablation_baseline_no_kl"
    "ablation_kl_reward_only_uniform"
    "ablation_kl_reward_only_conditioned"
    "ablation_kl_loss_only_uniform"
    "ablation_kl_loss_only_conditioned"
    "ablation_kl_both_uniform"
    "ablation_kl_both_conditioned"
)

# ========================
# Functions
# ========================
log_with_timestamp() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1"
}

run_experiment() {
    local exp_name=$1
    local script_path="${SCRIPT_DIR}/${exp_name}.sh"
    local log_file="${LOG_DIR}/${exp_name}_${TIMESTAMP}.log"
    
    log_with_timestamp "Starting experiment: ${exp_name}"
    log_with_timestamp "Script: ${script_path}"
    log_with_timestamp "Log file: ${log_file}"
    
    if [ ! -f "${script_path}" ]; then
        log_with_timestamp "ERROR: Script not found: ${script_path}"
        return 1
    fi
    
    # Make script executable
    chmod +x "${script_path}"
    
    # Run experiment with logging
    if bash "${script_path}" 2>&1 | tee "${log_file}"; then
        log_with_timestamp "✅ Experiment ${exp_name} completed successfully"
        return 0
    else
        local exit_code=$?
        log_with_timestamp "❌ Experiment ${exp_name} failed with exit code: ${exit_code}"
        return ${exit_code}
    fi
}

# ========================
# Main Execution
# ========================
log_with_timestamp "===== STARTING KL ABLATION EXPERIMENTS ====="
log_with_timestamp "Total experiments: ${#EXPERIMENTS[@]}"
log_with_timestamp "Log directory: ${LOG_DIR}"
log_with_timestamp "Timestamp: ${TIMESTAMP}"

# Track results
declare -a SUCCESSFUL_EXPERIMENTS=()
declare -a FAILED_EXPERIMENTS=()

# Run each experiment
for exp in "${EXPERIMENTS[@]}"; do
    log_with_timestamp "----------------------------------------"
    log_with_timestamp "Running: ${exp}"
    log_with_timestamp "----------------------------------------"
    
    if run_experiment "${exp}"; then
        SUCCESSFUL_EXPERIMENTS+=("${exp}")
        log_with_timestamp "✅ ${exp} - SUCCESS"
    else
        FAILED_EXPERIMENTS+=("${exp}")
        log_with_timestamp "❌ ${exp} - FAILED"
    fi
    
    log_with_timestamp "----------------------------------------"
    log_with_timestamp "Completed: ${exp}"
    log_with_timestamp "----------------------------------------"
done

# ========================
# Summary Report
# ========================
log_with_timestamp "===== ABLATION EXPERIMENTS SUMMARY ====="
log_with_timestamp "Total experiments: ${#EXPERIMENTS[@]}"
log_with_timestamp "Successful: ${#SUCCESSFUL_EXPERIMENTS[@]}"
log_with_timestamp "Failed: ${#FAILED_EXPERIMENTS[@]}"

if [ ${#SUCCESSFUL_EXPERIMENTS[@]} -gt 0 ]; then
    log_with_timestamp "✅ Successful experiments:"
    for exp in "${SUCCESSFUL_EXPERIMENTS[@]}"; do
        log_with_timestamp "  - ${exp}"
    done
fi

if [ ${#FAILED_EXPERIMENTS[@]} -gt 0 ]; then
    log_with_timestamp "❌ Failed experiments:"
    for exp in "${FAILED_EXPERIMENTS[@]}"; do
        log_with_timestamp "  - ${exp}"
    done
fi

log_with_timestamp "===== EXPERIMENT LOGS ====="
log_with_timestamp "All logs saved to: ${LOG_DIR}"
log_with_timestamp "Log files:"
ls -la "${LOG_DIR}"/*.log 2>/dev/null || log_with_timestamp "No log files found"

# ========================
# Exit with appropriate code
# ========================
if [ ${#FAILED_EXPERIMENTS[@]} -eq 0 ]; then
    log_with_timestamp "🎉 All experiments completed successfully!"
    exit 0
else
    log_with_timestamp "⚠️  Some experiments failed. Check logs for details."
    exit 1
fi
