#!/usr/bin/env bash
set -euo pipefail

usage() {
  cat <<EOF
Usage: $(basename "$0") -c <CONFIG_DIR> [-s <SBATCH_SCRIPT>] [-p <PATTERN>] [--dry-run]

  -d  Directory that contains the config files (required)
  -s  Path to the sbatch script to call (default: cl_pipeline.sbatch)
  -p  Glob pattern to match configs (default: *.yaml)
      e.g. *.yml, *.json, config_*.yaml
  --dry-run  Only print the sbatch commands, don't submit

Example:
  $(basename "$0") -d configs/exp1
  $(basename "$0") -d configs/exp1 -p '*.yml' -s scripts/cl_pipeline.sbatch
EOF
}

CONFIG_DIR=""
SBATCH_SCRIPT="launch_pipeline.slurm"
PATTERN="*.yaml"
DRY_RUN=0

# Parse args
while [[ $# -gt 0 ]]; do
  case "$1" in
    -c) CONFIG_DIR="$2"; shift 2 ;;
    -s) SBATCH_SCRIPT="$2"; shift 2 ;;
    -p) PATTERN="$2"; shift 2 ;;
    --dry-run) DRY_RUN=1; shift ;;
    -h|--help) usage; exit 0 ;;
    *) echo "Unknown argument: $1"; usage; exit 1 ;;
  esac
done

if [[ -z "$CONFIG_DIR" ]]; then
  echo "Error: -d <CONFIG_DIR> is required."
  usage
  exit 1
fi

if [[ ! -d "$CONFIG_DIR" ]]; then
  echo "Error: '$CONFIG_DIR' is not a directory."
  exit 1
fi

if [[ ! -f "$SBATCH_SCRIPT" ]]; then
  echo "Error: sbatch script '$SBATCH_SCRIPT' not found."
  exit 1
fi

shopt -s nullglob
mapfile -t CONFIGS < <(find "$CONFIG_DIR" -maxdepth 1 -type f -name "$PATTERN" | sort)
shopt -u nullglob

if [[ ${#CONFIGS[@]} -eq 0 ]]; then
  echo "No config files matching '$PATTERN' in '$CONFIG_DIR'."
  exit 1
fi

echo "Found ${#CONFIGS[@]} config(s) in $CONFIG_DIR matching '$PATTERN'."
for cfg in "${CONFIGS[@]}"; do
  abs_cfg="$(realpath "$cfg")"
  cmd=(sbatch "$SBATCH_SCRIPT" -c "$abs_cfg")
  echo "+ ${cmd[*]}"
  if [[ $DRY_RUN -eq 0 ]]; then
    "${cmd[@]}"
  fi
done
