#!/usr/bin/env bash
# merge_and_push_last.sh
#
# Merge & push all runs that live under ROOT_DIR.
#
# Usage
#   ./merge_and_push_last.sh [-o HF_ORG] [-p] [-s MERGE_SCRIPT] ROOT_DIR
#
# Options
#   -o HF_ORG   Prefix repos with HF_ORG/   (default: none)
#   -p            Create the repos as private  (adds --private flag)
#   -s SCRIPT     Path to merge_adapter_and_push.py
#                   (default: script sitting next to this driver)
#
# Example
#   ./merge_and_push_last.sh -o alice -p ./experiments
#
set -euo pipefail

HF_ORG=""
PRIVATE_FLAG=""
MERGE_SCRIPT="$(dirname "$0")/merge_and_push.py"

# --- parse options ---------------------------------------------------------
while getopts ":o:ps:" opt; do
  case $opt in
    o) HF_ORG="$OPTARG" ;;
    p) PRIVATE_FLAG="--private" ;;
    s) MERGE_SCRIPT="$OPTARG" ;;
    \?) echo "Invalid option: -$OPTARG" >&2; exit 1 ;;
  esac
done
shift $((OPTIND -1))

# --- positional arg ---------------------------------------------------------
if [ $# -ne 1 ]; then
  echo "Usage: $0 [-u HF_ORG] [-p] [-s MERGE_SCRIPT] ROOT_DIR" >&2
  exit 1
fi
ROOT_DIR="$1"

# --- sanity checks ----------------------------------------------------------
[ -d "$ROOT_DIR"    ] || { echo "Root dir $ROOT_DIR not found";      exit 1; }
[ -f "$MERGE_SCRIPT" ] || { echo "Merge script $MERGE_SCRIPT missing"; exit 1; }

# --- main loop --------------------------------------------------------------
find "$ROOT_DIR" -mindepth 1 -maxdepth 1 -type d -print0 | \
while IFS= read -r -d '' RUN_DIR; do
  RUN_NAME="$(basename "$RUN_DIR")"
  echo -e "\n=== Processing $RUN_NAME ==="

  LATEST_FILE="$RUN_DIR/latest_opt_step_dir"
  if [ ! -f "$LATEST_FILE" ]; then
    echo "⚠️  Skipping (missing latest_opt_step_dir)" >&2
    continue
  fi

  LATEST_STEP="$(<"$LATEST_FILE")"
  if [ ! -d "$LATEST_STEP" ]; then
    echo "⚠️  Skipping (checkpoint $LATEST_STEP not found)" >&2
    continue
  fi

  # Compose HF repo id
  REPO_ID="${HF_ORG}/$RUN_NAME"
  echo "📦  Calling merge_adapter_and_push.py → $REPO_ID"

  set -x
  (
    python "$MERGE_SCRIPT" \
          --checkpoint_dir "$LATEST_STEP" \
          --repo_id "$REPO_ID" \
          ${PRIVATE_FLAG} 
  ) || echo "❌ Failed to push $REPO_ID"
  set +x
done
